package cool.scx.socket;

import cool.scx.util.URIBuilder;
import io.netty.util.Timeout;
import io.vertx.core.Future;
import io.vertx.core.http.WebSocket;
import io.vertx.core.http.WebSocketClient;
import io.vertx.core.http.WebSocketConnectOptions;

import java.util.function.Consumer;

import static cool.scx.socket.ScxSocketHelper.SCX_SOCKET_CLIENT_ID_KEY;
import static cool.scx.socket.ScxSocketHelper.setTimeout;
import static cool.scx.util.RandomUtils.randomUUID;
import static java.lang.System.Logger.Level.DEBUG;

public final class ScxSocketClient extends ScxSocket {

    private final WebSocketConnectOptions connectOptions;
    private final WebSocketClient webSocketClient;
    private final String clientID;
    private final ScxSocketClientOptions clientOptions;
    private Timeout reconnectTimeout;
    private Future<WebSocket> connectFuture;
    private Consumer<Void> onOpen;
    //为了解决 Future 无法移除回调 采取的折中方式
    private Consumer<WebSocket> _onConnectSuccess;
    private Consumer<Throwable> _onConnectFailure;

    public ScxSocketClient(String uri, WebSocketClient webSocketClient, String clientID, ScxSocketClientOptions clientOptions) {
        super(clientOptions);
        this.clientOptions = clientOptions;
        this.webSocketClient = webSocketClient;
        this.clientID = clientID;
        this.connectOptions = initConnectOptions(uri, this.clientID);
    }

    public ScxSocketClient(String uri, WebSocketClient webSocketClient, ScxSocketClientOptions options) {
        this(uri, webSocketClient, randomUUID(), options);
    }

    public ScxSocketClient(String uri, WebSocketClient webSocketClient, String clientID) {
        this(uri, webSocketClient, clientID, new ScxSocketClientOptions());
    }

    public ScxSocketClient(String uri, WebSocketClient webSocketClient) {
        this(uri, webSocketClient, randomUUID(), new ScxSocketClientOptions());
    }

    private static WebSocketConnectOptions initConnectOptions(String uri, String clientID) {
        var o = new WebSocketConnectOptions().setAbsoluteURI(uri);
        var oldUri = o.getURI();
        var newUri = URIBuilder.of(oldUri).addParam(SCX_SOCKET_CLIENT_ID_KEY, clientID).toString();
        o.setURI(newUri);
        return o;
    }

    //为了解决 Future 无法移除回调 采取的折中方式
    private void _onConnectSuccess(WebSocket webSocket) {
        if (_onConnectSuccess != null) {
            _onConnectSuccess.accept(webSocket);
        }
    }

    //为了解决 Future 无法移除回调 采取的折中方式
    private void _onConnectFailure(Throwable throwable) {
        if (_onConnectFailure != null) {
            _onConnectFailure.accept(throwable);
        }
    }

    private void _setConnectFuture() {
        //我们无法直接取消 Future 所以只能用这种折中的方式 将其回调设为 无作用
        this._onConnectSuccess = webSocket -> {
            this.start(webSocket);
            this.doOpen();
        };
        this._onConnectFailure = (v) -> this.reconnect();
    }

    private void _removeConnectFuture() {
        //我们无法直接取消 Future 所以只能用这种折中的方式 将其回调设为 无作用
        _onConnectSuccess = null;
        _onConnectFailure = null;
    }

    public ScxSocketClient onOpen(Consumer<Void> onOpen) {
        this.onOpen = onOpen;
        return this;
    }

    private void cancelReconnect() {
        if (this.reconnectTimeout != null) {
            this.reconnectTimeout.cancel();
            this.reconnectTimeout = null;
        }
    }

    public void connect() {
        //当前已经存在一个连接中的任务
        if (this.connectFuture != null && !this.connectFuture.isComplete()) {
            return;
        }
        //关闭上一次连接
        this.close();
        this._setConnectFuture();
        this.connectFuture = webSocketClient.connect(connectOptions);
        this.connectFuture.onSuccess(this::_onConnectSuccess).onFailure(this::_onConnectFailure);
    }

    private void doOpen() {
        callOnOpen(null);
    }

    @Override
    protected void doClose(Void unused) {
        super.doClose(unused);
        this.connect();
    }

    @Override
    protected void doError(Throwable e) {
        super.doError(e);
        this.connect();
    }

    private void reconnect() {
        //如果当前已经存在一个重连进程 则不进行重连
        if (this.reconnectTimeout != null) {
            return;
        }
        logger.log(DEBUG, "WebSocket 重连中... ");
        this.reconnectTimeout = setTimeout(() -> {  //没连接上会一直重连，设置延迟为5000毫秒避免请求过多
            this.reconnectTimeout = null;
            this.connect();
        }, clientOptions.getReconnectTimeout());
    }

    @Override
    public void close() {
        _removeConnectFuture();
        cancelReconnect();
        super.close();
    }

    public String clientID() {
        return clientID;
    }

    @Override
    protected void doPingTimeout() {
        //心跳失败直接重连
        this.connect();
    }

    private void callOnOpen(Void v) {
        if (this.onOpen != null) {
            this.onOpen.accept(v);
        }
    }

    private void callOnOpenAsync(Void v) {
        if (this.onOpen != null) {
            Thread.ofVirtual().start(() -> this.onOpen.accept(v));
        }
    }

}
