package cool.scx.socket;

import io.netty.util.Timeout;
import io.vertx.core.http.WebSocketBase;

import static cool.scx.socket.ScxSocketHelper.setTimeout;

/**
 * 客户端连接对象
 */
public final class ScxSocketClientConnect extends ScxSocket {

    private final String clientID;
    private final ScxSocketServer scxSocketServer;
    private final ScxSocketServerOptions serverOptions;
    private Timeout removeClosedClientTimeout;

    public ScxSocketClientConnect(String clientID, ScxSocketServerOptions serverOptions, ScxSocketServer scxSocketServer) {
        super(serverOptions);
        this.clientID = clientID;
        this.serverOptions = serverOptions;
        this.scxSocketServer = scxSocketServer;
    }

    public String clientID() {
        return clientID;
    }

    @Override
    void start(WebSocketBase webSocket) {
        super.start(webSocket);
        cancelRemoveClosedClientTask();
    }

    @Override
    public void close() {
        super.close();
        startRemoveClosedClientTask();
    }

    private void startRemoveClosedClientTask() {
        cancelRemoveClosedClientTask();
        this.removeClosedClientTimeout = setTimeout(this::removeClosedClient, serverOptions.getRemoveClosedClientTimeout());
    }

    private void cancelRemoveClosedClientTask() {
        if (this.removeClosedClientTimeout != null) {
            this.removeClosedClientTimeout.cancel();
            this.removeClosedClientTimeout = null;
        }
    }

    private void removeClosedClient() {
        this.scxSocketServer.clientConnectMap.remove(this.clientID);
    }

    /**
     * 服务端不需要 ping 客户端 只需要等待 pong 并在心跳超时后关闭即可
     */
    @Override
    protected void startPing() {
        //什么也不需要做
    }

}
