package cool.scx.socket;

import io.netty.util.Timeout;
import io.vertx.core.Future;

import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Consumer;

import static cool.scx.socket.ScxSocketHelper.setTimeout;
import static java.lang.Math.max;

public final class SendTask {

    private final ScxSocketFrame socketFrame;
    private final ScxSocket scxSocket;
    private final SendOptions options;
    private final AtomicInteger sendTimes = new AtomicInteger(0);
    private Timeout resendThread;
    private Future<Void> sendFuture;
    //为了解决 Future 无法移除回调 采取的折中方式
    private Consumer<Void> _onSendSuccess;
    private Consumer<Throwable> _onSendFailure;

    public SendTask(ScxSocketFrame socketFrame, ScxSocket scxSocket, SendOptions options) {
        this.socketFrame = socketFrame;
        this.scxSocket = scxSocket;
        this.options = options;
    }

    /**
     * 根据次数获取延时时间
     * 根据次数进行 2的 次方倍增 , 如 1, 2 ,4 ,8 ,16 等
     *
     * @param times 次数 (0 起始)
     * @return 延时时间 (毫秒)
     */
    private static long getDelayed(int times) {
        return 1000L * (1L << times);
    }

    public void start() {
        //当前 websocket 不可用
        if (this.scxSocket.isClosed()) {
            return;
        }
        //当前已经存在一个 发送中(并未完成发送) 的任务
        if (this.sendFuture != null && !this.sendFuture.isComplete()) {
            return;
        }
        //超过最大发送次数
        if (this.sendTimes.get() > options.getMaxResendTimes()) {
            if (options.getGiveUpIfReachMaxResendTimes()) {
                clear();
            }
            return;
        }
        //根据不同序列化配置发送不同消息
        this.sendFuture = scxSocket.webSocket.writeTextMessage(this.socketFrame.toJson());

        _setConnectFuture();

        this.sendFuture.onSuccess(this::_onSendSuccess).onFailure(this::_onSendFailure);

    }

    /**
     * 取消重发任务
     */
    public void cancelResend() {
        _removeConnectFuture();
        if (this.resendThread != null) {
            this.resendThread.cancel();
            this.resendThread = null;
        }
    }

    /**
     * 从任务列表中移除此任务
     */
    public void clear() {
        cancelResend();
        this.scxSocket.sendTaskMap.remove(socketFrame.seq_id);
    }

    public ScxSocketFrame socketFrame() {
        return socketFrame;
    }

    //为了解决 Future 无法移除回调 采取的折中方式
    private void _onSendSuccess(Void v) {
        if (_onSendSuccess != null) {
            _onSendSuccess.accept(v);
        }
    }

    //为了解决 Future 无法移除回调 采取的折中方式
    private void _onSendFailure(Throwable throwable) {
        if (_onSendFailure != null) {
            _onSendFailure.accept(throwable);
        }
    }

    private void _setConnectFuture() {
        //我们无法直接取消 Future 所以只能用这种折中的方式 将其回调设为 无作用
        this._onSendSuccess = webSocket -> {
            var currentSendTime = sendTimes.getAndIncrement();
            //当需要 ack 时 创建 重复发送 延时
            if (options.getNeedAck()) {
                this.resendThread = setTimeout(this::start, max(getDelayed(currentSendTime), options.getMaxResendDelayed()));
            } else {
                clear();
            }
        };
        this._onSendFailure = (v) -> {
            //todo
        };
    }

    private void _removeConnectFuture() {
        //我们无法直接取消 Future 所以只能用这种折中的方式 将其回调设为 无作用
        _onSendSuccess = null;
        _onSendFailure = null;
    }

}
