package cool.scx.socket;

import io.netty.handler.codec.http.QueryStringDecoder;
import io.vertx.core.http.ServerWebSocket;

import java.util.Collection;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.function.Consumer;

import static cool.scx.socket.ScxSocketHelper.SCX_SOCKET_CLIENT_ID_KEY;

public final class ScxSocketServer {

    final ConcurrentMap<String, ScxSocketClientConnect> clientConnectMap;
    final ScxSocketServerOptions options;
    private Consumer<ScxSocketClientConnect> onClientConnect;

    public ScxSocketServer() {
        this(new ScxSocketServerOptions());
    }

    public ScxSocketServer(ScxSocketServerOptions options) {
        this.options = options;
        this.clientConnectMap = new ConcurrentHashMap<>();
    }

    private static String getClientID(ServerWebSocket serverWebSocket) {
        var decoder = new QueryStringDecoder(serverWebSocket.uri());
        var parameters = decoder.parameters();
        var clientIDValues = parameters.get(SCX_SOCKET_CLIENT_ID_KEY);
        return clientIDValues.isEmpty() ? null : clientIDValues.get(0);
    }

    public ScxSocketServer onClientConnect(Consumer<ScxSocketClientConnect> onClientConnect) {
        this.onClientConnect = onClientConnect;
        return this;
    }

    public ScxSocketClientConnect getClient(String clientID) {
        return clientConnectMap.get(clientID);
    }

    public ScxSocketClientConnect getOrCreateClient(String clientID) {
        return clientConnectMap.computeIfAbsent(clientID, (k) -> new ScxSocketClientConnect(clientID, options, this));
    }

    public Collection<ScxSocketClientConnect> getClients() {
        return clientConnectMap.values();
    }

    public void call(ServerWebSocket serverWebSocket) {
        var clientID = getClientID(serverWebSocket);
        if (clientID == null) {
            serverWebSocket.reject(400);
        }
        var clientConnect = getOrCreateClient(clientID);
        clientConnect.start(serverWebSocket);
        callOnClientConnectAsync(clientConnect);
    }

    private void callOnClientConnect(ScxSocketClientConnect clientConnect) {
        if (this.onClientConnect != null) {
            this.onClientConnect.accept(clientConnect);
        }
    }

    private void callOnClientConnectAsync(ScxSocketClientConnect clientConnect) {
        if (this.onClientConnect != null) {
            Thread.ofVirtual().start(() -> this.onClientConnect.accept(clientConnect));
        }
    }

}
