/**
 * Copyright (c) 2023 murenchao
 * taomu is licensed under Mulan PubL v2.
 * You can use this software according to the terms and conditions of the Mulan PubL v2.
 * You may obtain a copy of Mulan PubL v2 at:
 *       http://license.coscl.org.cn/MulanPubL-2.0
 * THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
 * EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
 * MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
 * See the Mulan PubL v2 for more details.
 */
package cool.taomu.box.crypto.util;

import cool.taomu.box.crypto.Base64;
import java.io.BufferedWriter;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileWriter;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.charset.Charset;
import java.security.SecureRandom;
import java.security.Security;
import java.util.Iterator;
import java.util.UUID;
import org.apache.commons.io.FileUtils;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.bouncycastle.openpgp.PGPCompressedData;
import org.bouncycastle.openpgp.PGPCompressedDataGenerator;
import org.bouncycastle.openpgp.PGPEncryptedData;
import org.bouncycastle.openpgp.PGPEncryptedDataGenerator;
import org.bouncycastle.openpgp.PGPEncryptedDataList;
import org.bouncycastle.openpgp.PGPLiteralData;
import org.bouncycastle.openpgp.PGPOnePassSignatureList;
import org.bouncycastle.openpgp.PGPPrivateKey;
import org.bouncycastle.openpgp.PGPPublicKey;
import org.bouncycastle.openpgp.PGPPublicKeyEncryptedData;
import org.bouncycastle.openpgp.PGPPublicKeyRing;
import org.bouncycastle.openpgp.PGPPublicKeyRingCollection;
import org.bouncycastle.openpgp.PGPSecretKey;
import org.bouncycastle.openpgp.PGPSecretKeyRingCollection;
import org.bouncycastle.openpgp.PGPSignatureList;
import org.bouncycastle.openpgp.PGPUtil;
import org.bouncycastle.openpgp.jcajce.JcaPGPObjectFactory;
import org.bouncycastle.openpgp.operator.PBESecretKeyDecryptor;
import org.bouncycastle.openpgp.operator.jcajce.JcaKeyFingerprintCalculator;
import org.bouncycastle.openpgp.operator.jcajce.JcePBESecretKeyDecryptorBuilder;
import org.bouncycastle.openpgp.operator.jcajce.JcePGPDataEncryptorBuilder;
import org.bouncycastle.openpgp.operator.jcajce.JcePublicKeyDataDecryptorFactoryBuilder;
import org.bouncycastle.openpgp.operator.jcajce.JcePublicKeyKeyEncryptionMethodGenerator;
import org.bouncycastle.util.io.Streams;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.Functions.Function0;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.InputOutput;
import org.eclipse.xtext.xbase.lib.IteratorExtensions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@SuppressWarnings("all")
public class PgpUtils {
  public static class CreateTempFile {
    public static File create(final byte[] content, final String charset) {
      Charset _forName = Charset.forName(charset);
      String _string = new String(content, _forName);
      return PgpUtils.CreateTempFile.create(_string);
    }
    
    public static File create(final String content) {
      try {
        String tempName = UUID.randomUUID().toString();
        File temp = File.createTempFile(tempName, ".temp");
        temp.deleteOnExit();
        try (BufferedWriter bw = new Function0<BufferedWriter>() {
          @Override
          public BufferedWriter apply() {
            try {
              FileWriter _fileWriter = new FileWriter(temp);
              return new BufferedWriter(_fileWriter);
            } catch (Throwable _e) {
              throw Exceptions.sneakyThrow(_e);
            }
          }
        }.apply()) {
          bw.write(content);
        }
        return temp;
      } catch (Throwable _e) {
        throw Exceptions.sneakyThrow(_e);
      }
    }
    
    public static File create(final byte[] content) {
      try {
        String tempName = UUID.randomUUID().toString();
        File temp = File.createTempFile(tempName, ".temp");
        temp.deleteOnExit();
        FileUtils.writeByteArrayToFile(temp, content);
        return temp;
      } catch (Throwable _e) {
        throw Exceptions.sneakyThrow(_e);
      }
    }
  }
  
  private static final Logger LOG = LoggerFactory.getLogger(PgpUtils.class);
  
  public static void main(final String[] args) {
    try {
      BouncyCastleProvider _bouncyCastleProvider = new BouncyCastleProvider();
      Security.addProvider(_bouncyCastleProvider);
      File _file = new File("src/test/resources/publicKey1.txt");
      String pk = FileUtils.readFileToString(_file, 
        Charset.forName("UTF-8"));
      byte[] data = PgpUtils.encrypt("Hello World!!!! ".getBytes(), pk, true);
      String _string = new String(data, "UTF-8");
      InputOutput.<String>println(_string);
      File _file_1 = new File("src/test/resources/privateKey1.txt");
      String prk = FileUtils.readFileToString(_file_1, 
        Charset.forName("UTF-8"));
      byte[] d = PgpUtils.decrypt(data, prk, "123456789");
      String _string_1 = new String(d, "UTF-8");
      String _plus = ("result : " + _string_1);
      InputOutput.<String>println(_plus);
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  public static byte[] decrypt(final byte[] data, final String privateKey, final String password) {
    try {
      try (ByteArrayInputStream bais = new ByteArrayInputStream(data)) {
        InputStream input = PGPUtil.getDecoderStream(bais);
        JcaPGPObjectFactory jpof = new JcaPGPObjectFactory(input);
        Object o = jpof.nextObject();
        PGPEncryptedDataList enc = null;
        if ((o instanceof PGPEncryptedDataList)) {
          enc = ((PGPEncryptedDataList)o);
        } else {
          Object _nextObject = jpof.nextObject();
          enc = ((PGPEncryptedDataList) _nextObject);
        }
        byte[] _bytes = privateKey.getBytes();
        byte[] pkeyBase64 = new Base64(_bytes).decode();
        JcaKeyFingerprintCalculator _jcaKeyFingerprintCalculator = new JcaKeyFingerprintCalculator();
        final PGPSecretKeyRingCollection pgpSec = new PGPSecretKeyRingCollection(pkeyBase64, _jcaKeyFingerprintCalculator);
        Iterator<PGPEncryptedData> edo = enc.getEncryptedDataObjects();
        final Function1<PGPEncryptedData, Boolean> _function = (PGPEncryptedData k) -> {
          try {
            PGPSecretKey _secretKey = pgpSec.getSecretKey(((PGPPublicKeyEncryptedData) k).getKeyID());
            return Boolean.valueOf((_secretKey != null));
          } catch (Throwable _e) {
            throw Exceptions.sneakyThrow(_e);
          }
        };
        PGPEncryptedData _findFirst = IteratorExtensions.<PGPEncryptedData>findFirst(edo, _function);
        PGPPublicKeyEncryptedData pbe = ((PGPPublicKeyEncryptedData) _findFirst);
        PGPSecretKey secKey = pgpSec.getSecretKey(pbe.getKeyID());
        PBESecretKeyDecryptor jpekdb = new JcePBESecretKeyDecryptorBuilder().setProvider("BC").build(password.toCharArray());
        PGPPrivateKey sKey = secKey.extractPrivateKey(jpekdb);
        InputStream clear = pbe.getDataStream(new JcePublicKeyDataDecryptorFactoryBuilder().setProvider("BC").build(sKey));
        JcaPGPObjectFactory plain = new JcaPGPObjectFactory(clear);
        Object message = plain.nextObject();
        if ((input != null)) {
          input.close();
        }
        try (ByteArrayOutputStream out = new ByteArrayOutputStream()) {
          if ((message instanceof PGPCompressedData)) {
            InputStream _dataStream = ((PGPCompressedData)message).getDataStream();
            JcaPGPObjectFactory fact = new JcaPGPObjectFactory(_dataStream);
            message = fact.nextObject();
          }
          if ((message instanceof PGPLiteralData)) {
            Streams.pipeAll(((PGPLiteralData)message).getInputStream(), out);
          } else {
            if ((message instanceof PGPOnePassSignatureList)) {
              PgpUtils.LOG.info("encrypted message contains a signed message - not literal data.");
            } else {
              if ((message instanceof PGPSignatureList)) {
                PgpUtils.LOG.info("encrypted message contains a signed message - not literal data.");
              }
            }
          }
          boolean _isIntegrityProtected = pbe.isIntegrityProtected();
          if (_isIntegrityProtected) {
            boolean _verify = pbe.verify();
            boolean _not = (!_verify);
            if (_not) {
              PgpUtils.LOG.info("message failed integrity check");
            } else {
              PgpUtils.LOG.info("message integrity check passed");
            }
          } else {
            PgpUtils.LOG.info("no message integrity check");
          }
          return out.toByteArray();
        }
      }
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  public static byte[] encrypt(final byte[] data, final String publicKey, final boolean withIntegrityCheck) {
    try {
      try (ByteArrayOutputStream out = new ByteArrayOutputStream()) {
        PgpUtils.encrypt(data, publicKey, withIntegrityCheck, out);
        return out.toByteArray();
      }
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  public static void encrypt(final byte[] data, final String publicKey, final boolean withIntegrityCheck, final OutputStream out) {
    try {
      try (ByteArrayInputStream bais = new ByteArrayInputStream(new Base64(publicKey.getBytes()).decode())) {
        InputStream input = PGPUtil.getDecoderStream(bais);
        JcaKeyFingerprintCalculator _jcaKeyFingerprintCalculator = new JcaKeyFingerprintCalculator();
        PGPPublicKeyRingCollection pkrc = new PGPPublicKeyRingCollection(input, _jcaKeyFingerprintCalculator);
        final Function1<PGPPublicKeyRing, Boolean> _function = (PGPPublicKeyRing it) -> {
          final Function1<PGPPublicKey, Boolean> _function_1 = (PGPPublicKey it_1) -> {
            return Boolean.valueOf(it_1.isEncryptionKey());
          };
          return Boolean.valueOf(IteratorExtensions.<PGPPublicKey>findFirst(it.getPublicKeys(), _function_1).isEncryptionKey());
        };
        PGPPublicKey encKey = IteratorExtensions.<PGPPublicKeyRing>findFirst(pkrc.getKeyRings(), _function).getPublicKey();
        File file = PgpUtils.CreateTempFile.create(data, "UTF-8");
        try (ByteArrayOutputStream baos = new ByteArrayOutputStream()) {
          PGPCompressedDataGenerator cdg = new PGPCompressedDataGenerator(PGPCompressedData.ZIP);
          PGPUtil.writeFileToLiteralData(cdg.open(baos), 'b', file);
          cdg.close();
          byte[] bytes = baos.toByteArray();
          JcePGPDataEncryptorBuilder jpdeb = new JcePGPDataEncryptorBuilder(PGPEncryptedData.CAST5);
          jpdeb = jpdeb.setWithIntegrityPacket(withIntegrityCheck);
          SecureRandom _secureRandom = new SecureRandom();
          JcePGPDataEncryptorBuilder bc = jpdeb.setSecureRandom(_secureRandom).setProvider("BC");
          PGPEncryptedDataGenerator encGen = new PGPEncryptedDataGenerator(bc);
          JcePublicKeyKeyEncryptionMethodGenerator _jcePublicKeyKeyEncryptionMethodGenerator = new JcePublicKeyKeyEncryptionMethodGenerator(encKey);
          encGen.addMethod(_jcePublicKeyKeyEncryptionMethodGenerator.setProvider("BC"));
          try (OutputStream cout = PgpUtils.encGenOpen(encGen, out, bytes)) {
            cout.write(bytes);
          }
        } finally {
          file.delete();
        }
      }
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  private static OutputStream encGenOpen(final PGPEncryptedDataGenerator encGen, final OutputStream out, final byte[] bytes) {
    try {
      return encGen.open(out, bytes.length);
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
}
