/**
 * Copyright (c) 2023 murenchao
 * taomu is licensed under Mulan PubL v2.
 * You can use this software according to the terms and conditions of the Mulan PubL v2.
 * You may obtain a copy of Mulan PubL v2 at:
 *       http://license.coscl.org.cn/MulanPubL-2.0
 * THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
 * EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
 * MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
 * See the Mulan PubL v2 for more details.
 */
package cool.taomu.box.dbutils;

import java.sql.Connection;
import java.sql.SQLException;
import java.util.List;
import java.util.Map;
import javax.inject.Inject;
import javax.sql.DataSource;
import org.apache.commons.dbutils.QueryRunner;
import org.apache.commons.dbutils.handlers.BeanListHandler;
import org.apache.commons.dbutils.handlers.MapListHandler;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@SuppressWarnings("all")
public class DbUtils extends QueryRunner implements IDbUtils {
  private static final Logger LOG = LoggerFactory.getLogger(DbUtils.class);
  
  private Connection conn = null;
  
  public DbUtils(final Connection conn) {
    this.conn = conn;
  }
  
  @Inject
  public DbUtils(final DataSource ds) {
    try {
      this.conn = ds.getConnection();
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  @Override
  public void transaction(final Callback cl) {
    try {
      try {
        DbUtils.LOG.info("执行数据库事务");
        this.conn.setAutoCommit(false);
        cl.execute(this);
        this.conn.commit();
      } catch (final Throwable _t) {
        if (_t instanceof SQLException) {
          final SQLException ex = (SQLException)_t;
          DbUtils.LOG.info("Sql  exception:", ex);
          this.conn.rollback();
        } else {
          throw Exceptions.sneakyThrow(_t);
        }
      }
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  @Override
  public int update(final String sql, final Object... args) {
    try {
      DbUtils.LOG.info("sql:{}", sql);
      return this.update(this.conn, sql, args);
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  @Override
  public int[] batch(final String sql, final List<List<Object>> args) {
    try {
      DbUtils.LOG.info("sql:{}", sql);
      Object[][] array = new Object[args.size()][0];
      for (int i = 0; (i < ((List<Object[]>)Conversions.doWrapArray(array)).size()); i++) {
        array[i] = args.get(i).toArray();
      }
      return this.batch(this.conn, sql, array);
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  @Override
  public <T extends Object> List<T> queryEntity(final String sql, final Class<T> zlass, final Object... args) {
    try {
      DbUtils.LOG.info("sql:{}", sql);
      List<T> results = null;
      if (((zlass != null) && (!zlass.equals(Map.class)))) {
        BeanListHandler<T> _beanListHandler = new BeanListHandler<T>(zlass);
        results = this.<List<T>>query(this.conn, sql, _beanListHandler, args);
      }
      return results;
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  @Override
  public List<Map<String, Object>> queryMap(final String sql, final Object... args) {
    try {
      DbUtils.LOG.info("sql:{}", sql);
      MapListHandler _mapListHandler = new MapListHandler();
      return this.<List<Map<String, Object>>>query(this.conn, sql, _mapListHandler, args);
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  @Override
  public void close() throws Exception {
    if ((this.conn != null)) {
      DbUtils.LOG.info("关闭数据库连接");
      this.conn.close();
    }
  }
}
