/**
 * Copyright (c) 2023 murenchao
 * taomu is licensed under Mulan PubL v2.
 * You can use this software according to the terms and conditions of the Mulan PubL v2.
 * You may obtain a copy of Mulan PubL v2 at:
 *       http://license.coscl.org.cn/MulanPubL-2.0
 * THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
 * EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
 * MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
 * See the Mulan PubL v2 for more details.
 */
package cool.taomu.box.git;

import com.google.inject.Inject;
import com.google.inject.name.Named;
import cool.taomu.box.git.entity.GitCommitInfoEntity;
import cool.taomu.box.git.entity.GitFileHistoryContentEntity;
import cool.taomu.box.utils.FileTools;
import cool.taomu.box.utils.inter.ICallback;
import java.io.File;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.function.Consumer;
import org.apache.commons.lang3.StringUtils;
import org.eclipse.jgit.api.AddCommand;
import org.eclipse.jgit.api.CheckoutCommand;
import org.eclipse.jgit.api.CloneCommand;
import org.eclipse.jgit.api.FetchCommand;
import org.eclipse.jgit.api.Git;
import org.eclipse.jgit.api.ListBranchCommand;
import org.eclipse.jgit.api.MergeCommand;
import org.eclipse.jgit.api.MergeResult;
import org.eclipse.jgit.api.PullCommand;
import org.eclipse.jgit.api.PullResult;
import org.eclipse.jgit.api.PushCommand;
import org.eclipse.jgit.api.ResetCommand;
import org.eclipse.jgit.api.RmCommand;
import org.eclipse.jgit.api.Status;
import org.eclipse.jgit.api.TransportCommand;
import org.eclipse.jgit.api.TransportConfigCallback;
import org.eclipse.jgit.lib.ObjectId;
import org.eclipse.jgit.lib.ObjectLoader;
import org.eclipse.jgit.lib.Ref;
import org.eclipse.jgit.lib.Repository;
import org.eclipse.jgit.revwalk.RevCommit;
import org.eclipse.jgit.revwalk.RevWalk;
import org.eclipse.jgit.storage.file.FileRepositoryBuilder;
import org.eclipse.jgit.transport.CredentialsProvider;
import org.eclipse.jgit.transport.FetchResult;
import org.eclipse.jgit.transport.PushResult;
import org.eclipse.jgit.transport.RefSpec;
import org.eclipse.jgit.treewalk.TreeWalk;
import org.eclipse.xtend.lib.annotations.Accessors;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.Functions.Function0;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.Pure;
import org.joda.time.DateTime;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Accessors
@SuppressWarnings("all")
public class JGitTools {
  private static final Logger LOG = LoggerFactory.getLogger(JGitTools.class);
  
  @Inject(optional = true)
  private TransportConfigCallback ssh;
  
  @Inject(optional = true)
  private CredentialsProvider cp;
  
  @Inject(optional = true)
  @Named("taomubox.git.repository")
  private String repository = "./repository";
  
  private boolean isAutoCommit = true;
  
  private void credentials(final TransportCommand<?, ?> command) {
    if ((this.ssh != null)) {
      command.setTransportConfigCallback(this.ssh);
    } else {
      if ((this.cp != null)) {
        command.setCredentialsProvider(this.cp);
      } else {
        JGitTools.LOG.info("GIT登陆credentials是空的");
      }
    }
  }
  
  private File getRepository(final String remoteRepository) {
    String projectName = remoteRepository;
    boolean _contains = remoteRepository.contains("/");
    if (_contains) {
      int _lastIndexOf = remoteRepository.lastIndexOf("/");
      int _plus = (_lastIndexOf + 1);
      projectName = remoteRepository.substring(_plus);
    }
    String _join = IterableExtensions.join(Collections.<String>unmodifiableList(CollectionLiterals.<String>newArrayList(this.repository, projectName)), File.separator);
    File file = new File(_join);
    boolean _exists = file.exists();
    boolean _not = (!_exists);
    if (_not) {
      JGitTools.LOG.info("创建目录:{}", file.getAbsolutePath());
      file.mkdirs();
    }
    return file;
  }
  
  public void init() {
    try {
      String _trim = this.repository.trim();
      String _join = IterableExtensions.join(Collections.<String>unmodifiableList(CollectionLiterals.<String>newArrayList(_trim, ".git")), File.separator);
      File file = new File(_join);
      Repository createRepo = FileRepositoryBuilder.create(file);
      if ((createRepo != null)) {
        createRepo.create();
      }
      createRepo.close();
      JGitTools.LOG.info("初始化本地仓库:{}", file.getAbsolutePath());
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  public void rmAndCommit(final String msg, final ICallback<RmCommand> call) {
    if ((call != null)) {
      try {
        String _trim = this.repository.trim();
        String _join = IterableExtensions.join(Collections.<String>unmodifiableList(CollectionLiterals.<String>newArrayList(_trim, ".git")), File.separator);
        File file = new File(_join);
        JGitTools.LOG.info("打开GIT仓库{}", file.getAbsolutePath());
        final Git git = Git.open(file);
        final RmCommand rm = git.rm();
        call.call(rm);
        rm.call();
        if (this.isAutoCommit) {
          RevCommit commit = git.commit().setMessage(StringUtils.<String>defaultIfBlank(msg, "无信息")).call();
          JGitTools.LOG.info("删除提交的id:{}", commit.getName());
        }
        git.close();
      } catch (final Throwable _t) {
        if (_t instanceof Exception) {
          final Exception ex = (Exception)_t;
          ex.printStackTrace();
        } else {
          throw Exceptions.sneakyThrow(_t);
        }
      }
    }
  }
  
  public void addAndCommit(final String msg, final ICallback<AddCommand> call) {
    if ((call != null)) {
      try {
        String _trim = this.repository.trim();
        String _join = IterableExtensions.join(Collections.<String>unmodifiableList(CollectionLiterals.<String>newArrayList(_trim, ".git")), File.separator);
        File file = new File(_join);
        JGitTools.LOG.info("打开GIT仓库{}", file.getAbsolutePath());
        final Git git = Git.open(file);
        final AddCommand add = git.add();
        call.call(add);
        add.call();
        if (this.isAutoCommit) {
          RevCommit commit = git.commit().setMessage(StringUtils.<String>defaultIfBlank(msg, "无信息")).call();
          JGitTools.LOG.info("添加提交的id:{}", commit.getName());
        }
        git.close();
      } catch (final Throwable _t) {
        if (_t instanceof Exception) {
          final Exception ex = (Exception)_t;
          ex.printStackTrace();
        } else {
          throw Exceptions.sneakyThrow(_t);
        }
      }
    }
  }
  
  public void getCommitIds(final Integer num, final ICallback<GitCommitInfoEntity> call) {
    try {
      JGitTools.LOG.info("获取commit历史");
      String _trim = this.repository.trim();
      String _join = IterableExtensions.join(Collections.<String>unmodifiableList(CollectionLiterals.<String>newArrayList(_trim, ".git")), File.separator);
      File _file = new File(_join);
      final Git git = Git.open(_file);
      Repository _repository = git.getRepository();
      final RevWalk walk = new RevWalk(_repository);
      RevCommit head = walk.parseCommit(git.getRepository().resolve("HEAD"));
      walk.reset();
      walk.markStart(head);
      List<RevCommit> walkList = IterableExtensions.<RevCommit>toList(walk);
      JGitTools.LOG.info("获取到{}条commit id", Integer.valueOf(walkList.size()));
      if (((num != null) && (call != null))) {
        int size = Math.min((num).intValue(), walkList.size());
        final Consumer<RevCommit> _function = (RevCommit it) -> {
          GitCommitInfoEntity entity = new GitCommitInfoEntity();
          entity.setId(it.getName());
          Date _when = it.getCommitterIdent().getWhen();
          DateTime _dateTime = new DateTime(_when);
          entity.setTime(_dateTime);
          JGitTools.LOG.info("commit id:{}", entity.toString());
          call.call(entity);
        };
        walkList.subList(0, size).forEach(_function);
      } else {
        if ((call != null)) {
          final Consumer<RevCommit> _function_1 = (RevCommit it) -> {
            GitCommitInfoEntity entity = new GitCommitInfoEntity();
            entity.setId(it.getName());
            Date _when = it.getCommitterIdent().getWhen();
            DateTime _dateTime = new DateTime(_when);
            entity.setTime(_dateTime);
            JGitTools.LOG.info("commit id:{}", it.getId());
            call.call(entity);
          };
          walkList.forEach(_function_1);
        }
      }
      walk.dispose();
      git.close();
    } catch (final Throwable _t) {
      if (_t instanceof IOException) {
        final IOException e = (IOException)_t;
        e.printStackTrace();
      } else {
        throw Exceptions.sneakyThrow(_t);
      }
    }
  }
  
  public void retrieveFileContent(final String filePath, final List<String> commitIds, final ICallback<GitFileHistoryContentEntity> call) {
    try {
      String _trim = this.repository.trim();
      String _join = IterableExtensions.join(Collections.<String>unmodifiableList(CollectionLiterals.<String>newArrayList(_trim, ".git")), File.separator);
      File _file = new File(_join);
      final Git git = Git.open(_file);
      Repository _repository = git.getRepository();
      final RevWalk walk = new RevWalk(_repository);
      boolean _isNotBlank = StringUtils.isNotBlank(filePath);
      if (_isNotBlank) {
        final Consumer<String> _function = (String commitId) -> {
          try {
            RevCommit commit = walk.parseCommit(git.getRepository().resolve(commitId));
            String _trim_1 = this.repository.trim();
            String _plus = (_trim_1 + File.separator);
            String fp = filePath.replace(_plus, "");
            this.handleCommit(git, commit, fp, walk, call);
          } catch (Throwable _e) {
            throw Exceptions.sneakyThrow(_e);
          }
        };
        commitIds.forEach(_function);
      } else {
        final ICallback<File> _function_1 = (File it) -> {
          if ((it.isFile() && (!it.getAbsolutePath().contains(".git")))) {
            final Consumer<String> _function_2 = (String commitId) -> {
              try {
                RevCommit commit = walk.parseCommit(git.getRepository().resolve(commitId));
                String _absolutePath = it.getAbsolutePath();
                String _trim_1 = it.getPath().trim();
                String _plus = (_trim_1 + File.separator);
                String fp = _absolutePath.replace(_plus, "");
                this.handleCommit(git, commit, fp, walk, call);
              } catch (Throwable _e) {
                throw Exceptions.sneakyThrow(_e);
              }
            };
            commitIds.forEach(_function_2);
          }
        };
        FileTools.scannerFile(this.repository, _function_1);
      }
      walk.dispose();
      git.close();
    } catch (final Throwable _t) {
      if (_t instanceof IOException) {
        final IOException e = (IOException)_t;
        e.printStackTrace();
      } else {
        throw Exceptions.sneakyThrow(_t);
      }
    }
  }
  
  private void handleCommit(final Git git, final RevCommit commit, final String filePath, final RevWalk walk, final ICallback<GitFileHistoryContentEntity> call) {
    try {
      try (TreeWalk treeWalk = new Function0<TreeWalk>() {
        @Override
        public TreeWalk apply() {
          try {
            return TreeWalk.forPath(git.getRepository(), filePath, commit.getTree());
          } catch (Throwable _e) {
            throw Exceptions.sneakyThrow(_e);
          }
        }
      }.apply()) {
        if ((treeWalk != null)) {
          ObjectId objectId = treeWalk.getObjectId(0);
          ObjectLoader loader = git.getRepository().open(objectId);
          byte[] data = loader.getBytes();
          String content = new String(data, StandardCharsets.UTF_8);
          JGitTools.LOG.info("从{}提交中获取{}文件内容", commit.getName(), filePath);
          GitFileHistoryContentEntity gfhc = new GitFileHistoryContentEntity();
          gfhc.setFilePath(filePath);
          gfhc.setContent(content);
          Date _when = commit.getCommitterIdent().getWhen();
          DateTime _dateTime = new DateTime(_when);
          gfhc.setTime(_dateTime);
          call.call(gfhc);
        } else {
          JGitTools.LOG.info("从{}提交没有发现文件{},提取结束", commit.getName(), filePath);
        }
      }
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  public void status(final ICallback<Status> call) {
    try {
      JGitTools.LOG.info("获取{}仓库的状态信息", this.repository);
      String _trim = this.repository.trim();
      String _join = IterableExtensions.join(Collections.<String>unmodifiableList(CollectionLiterals.<String>newArrayList(_trim, ".git")), File.separator);
      File file = new File(_join);
      try (final Git git = new Function0<Git>() {
        @Override
        public Git apply() {
          try {
            return Git.open(file);
          } catch (Throwable _e) {
            throw Exceptions.sneakyThrow(_e);
          }
        }
      }.apply()) {
        Status status = git.status().call();
        call.call(status);
      }
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  public String clone(final String remoteRepository, final boolean isAllBranches, final String branchOrTag) {
    try {
      String _xifexpression = null;
      if (isAllBranches) {
        _xifexpression = "是";
      } else {
        _xifexpression = "否";
      }
      JGitTools.LOG.info("克隆远程仓库{},是否克隆全部分支{},克隆分支或标签{}", remoteRepository, _xifexpression, branchOrTag);
      File file = this.getRepository(remoteRepository);
      file.mkdirs();
      CloneCommand clone = Git.cloneRepository().setURI(remoteRepository).setBare(false);
      if (isAllBranches) {
        clone.setCloneAllBranches(isAllBranches);
      } else {
        clone.setCloneAllBranches(isAllBranches);
        clone.setBranch(branchOrTag);
      }
      clone.setDirectory(file);
      this.credentials(clone);
      try (Git git = new Function0<Git>() {
        @Override
        public Git apply() {
          try {
            return clone.call();
          } catch (Throwable _e) {
            throw Exceptions.sneakyThrow(_e);
          }
        }
      }.apply()) {
        Collection<Ref> refs = git.lsRemote().setHeads(true).setRemote(remoteRepository).call();
        StringBuffer s = new StringBuffer();
        for (final Ref ref : refs) {
          {
            s.append(ref.getName());
            s.append("\n");
          }
        }
        s.append("clone success");
        JGitTools.LOG.info("克隆完成");
        return s.toString();
      }
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  public String push(final String remoteRepository, final String name, final String branch) {
    try {
      File file = this.getRepository(remoteRepository);
      try (Git git = new Function0<Git>() {
        @Override
        public Git apply() {
          try {
            return Git.open(file);
          } catch (Throwable _e) {
            throw Exceptions.sneakyThrow(_e);
          }
        }
      }.apply()) {
        PushCommand _push = git.push();
        RefSpec _refSpec = new RefSpec(branch);
        PushCommand push = _push.setRefSpecs(_refSpec).setRemote(name);
        this.credentials(push);
        Iterable<PushResult> infos = push.call();
        StringBuffer buffer = new StringBuffer();
        for (final PushResult info : infos) {
          {
            buffer.append(info.getMessages());
            buffer.append("\n");
          }
        }
        return buffer.toString();
      }
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  public String fetch(final String remoteRepository, final String branch) {
    try {
      File file = this.getRepository(remoteRepository);
      RefSpec _refSpec = new RefSpec(((branch + ":") + branch));
      FetchCommand fetchCommand = Git.open(file).fetch().setRemote(remoteRepository).setRefSpecs(
        Collections.<RefSpec>unmodifiableList(CollectionLiterals.<RefSpec>newArrayList(_refSpec)));
      this.credentials(fetchCommand);
      FetchResult result = fetchCommand.call();
      return result.getMessages();
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  public String fetchAll(final String remoteRepository) {
    try {
      JGitTools.LOG.info("fetch仓库{}全部分支", remoteRepository);
      File file = this.getRepository(remoteRepository);
      Collection<Ref> refs = Git.lsRemoteRepository().setHeads(true).setRemote(remoteRepository).call();
      StringBuffer log = new StringBuffer("fetch:\n");
      for (final Ref ref : refs) {
        {
          String refStr = ref.getName();
          RefSpec _refSpec = new RefSpec(((refStr + ":") + refStr));
          FetchCommand fetchCommand = Git.open(file).fetch().setRemote(remoteRepository).setRefSpecs(
            Collections.<RefSpec>unmodifiableList(CollectionLiterals.<RefSpec>newArrayList(_refSpec)));
          this.credentials(fetchCommand);
          FetchResult result = fetchCommand.call();
          log.append(result.getMessages());
          log.append("\n");
        }
      }
      log.append("Fetch success");
      JGitTools.LOG.info("fetch完成");
      return log.toString();
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  public RevCommit commit(final String remoteRepository, final String[] files, final String msg) {
    try {
      RevCommit _xblockexpression = null;
      {
        File file = this.getRepository(remoteRepository);
        RevCommit _xtrycatchfinallyexpression = null;
        try (final Git git = new Function0<Git>() {
          @Override
          public Git apply() {
            try {
              return Git.open(file);
            } catch (Throwable _e) {
              throw Exceptions.sneakyThrow(_e);
            }
          }
        }.apply()) {
          RevCommit _xblockexpression_1 = null;
          {
            final AddCommand addc = git.add();
            final Consumer<String> _function = (String it) -> {
              try {
                addc.addFilepattern(it).call();
              } catch (Throwable _e) {
                throw Exceptions.sneakyThrow(_e);
              }
            };
            ((List<String>)Conversions.doWrapArray(files)).forEach(_function);
            _xblockexpression_1 = git.commit().setMessage(msg).call();
          }
          _xtrycatchfinallyexpression = _xblockexpression_1;
        }
        _xblockexpression = _xtrycatchfinallyexpression;
      }
      return _xblockexpression;
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  public HashMap<String, String> status(final String remoteRepository) {
    try {
      HashMap<String, String> map = new HashMap<String, String>();
      File file = this.getRepository(remoteRepository);
      try (final Git git = new Function0<Git>() {
        @Override
        public Git apply() {
          try {
            return Git.open(file);
          } catch (Throwable _e) {
            throw Exceptions.sneakyThrow(_e);
          }
        }
      }.apply()) {
        Status status = git.status().call();
        map.put("Added", status.getAdded().toString());
        map.put("Changed", status.getChanged().toString());
        map.put("Conflicting", status.getConflicting().toString());
        map.put("ConflictingStageState", status.getConflictingStageState().toString());
        map.put("IgnoredNotInIndex", status.getIgnoredNotInIndex().toString());
        map.put("Missing", status.getMissing().toString());
        map.put("Modified", status.getModified().toString());
        map.put("Removed", status.getRemoved().toString());
        map.put("UntrackedFiles", status.getUntracked().toString());
        map.put("UntrackedFolders", status.getUntrackedFolders().toString());
        return map;
      }
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  public Ref branchCreate(final String remoteRepository, final String name) {
    try {
      Ref _xblockexpression = null;
      {
        File file = this.getRepository(remoteRepository);
        Ref _xtrycatchfinallyexpression = null;
        try (final Git git = new Function0<Git>() {
          @Override
          public Git apply() {
            try {
              return Git.open(file);
            } catch (Throwable _e) {
              throw Exceptions.sneakyThrow(_e);
            }
          }
        }.apply()) {
          _xtrycatchfinallyexpression = git.branchCreate().setName(name).call();
        }
        _xblockexpression = _xtrycatchfinallyexpression;
      }
      return _xblockexpression;
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  public List<String> branchDelete(final String remoteRepository, final String[] names) {
    try {
      List<String> _xblockexpression = null;
      {
        File file = this.getRepository(remoteRepository);
        List<String> _xtrycatchfinallyexpression = null;
        try (final Git git = new Function0<Git>() {
          @Override
          public Git apply() {
            try {
              return Git.open(file);
            } catch (Throwable _e) {
              throw Exceptions.sneakyThrow(_e);
            }
          }
        }.apply()) {
          _xtrycatchfinallyexpression = git.branchDelete().setBranchNames(names).call();
        }
        _xblockexpression = _xtrycatchfinallyexpression;
      }
      return _xblockexpression;
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  public Ref branchRename(final String remoteRepository, final String oldName, final String newName) {
    try {
      Ref _xblockexpression = null;
      {
        File file = this.getRepository(remoteRepository);
        Ref _xtrycatchfinallyexpression = null;
        try (final Git git = new Function0<Git>() {
          @Override
          public Git apply() {
            try {
              return Git.open(file);
            } catch (Throwable _e) {
              throw Exceptions.sneakyThrow(_e);
            }
          }
        }.apply()) {
          _xtrycatchfinallyexpression = git.branchRename().setNewName(newName).setOldName(oldName).call();
        }
        _xblockexpression = _xtrycatchfinallyexpression;
      }
      return _xblockexpression;
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  public ArrayList<String> branchList(final String remoteRepository) {
    return this.branchList(remoteRepository, null);
  }
  
  public ArrayList<String> branchList(final String remoteRepository, final ListBranchCommand.ListMode mode) {
    try {
      File file = this.getRepository(remoteRepository);
      try (final Git git = new Function0<Git>() {
        @Override
        public Git apply() {
          try {
            return Git.open(file);
          } catch (Throwable _e) {
            throw Exceptions.sneakyThrow(_e);
          }
        }
      }.apply()) {
        final ArrayList<String> result = CollectionLiterals.<String>newArrayList();
        final ListBranchCommand blist = git.branchList();
        if ((mode != null)) {
          blist.setListMode(mode);
        }
        final Consumer<Ref> _function = (Ref it) -> {
          result.add(it.getName());
        };
        blist.call().forEach(_function);
        return result;
      }
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  private boolean isBranchExist(final Git git, final String bname) {
    try {
      List<Ref> refs = git.branchList().call();
      for (final Ref ref : refs) {
        boolean _contains = ref.getName().contains(bname);
        if (_contains) {
          return true;
        }
      }
      return false;
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  public void checkoutBranch(final String remoteRepository, final String name) {
    try {
      JGitTools.LOG.info("从仓库 {} checkout 分支:{}", remoteRepository, name);
      FileRepositoryBuilder _fileRepositoryBuilder = new FileRepositoryBuilder();
      File _file = new File(remoteRepository);
      Repository repo = _fileRepositoryBuilder.setGitDir(_file).build();
      try (Git git = new Git(repo)) {
        boolean _isBranchExist = this.isBranchExist(git, name);
        if (_isBranchExist) {
          git.checkout().setCreateBranch(false).setName(name).call();
        } else {
          git.checkout().setCreateBranch(true).setName(name).setStartPoint(("origin" + name)).call();
        }
        git.pull().call();
      }
      JGitTools.LOG.info("checkout完成");
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  public void checkout(final String remoteRepository, final String name) {
    try {
      JGitTools.LOG.info("从仓库 {} checkout:{}", remoteRepository, name);
      File file = this.getRepository(remoteRepository);
      try (final Git git = new Function0<Git>() {
        @Override
        public Git apply() {
          try {
            return Git.open(file);
          } catch (Throwable _e) {
            throw Exceptions.sneakyThrow(_e);
          }
        }
      }.apply()) {
        CheckoutCommand _checkout = git.checkout();
        CheckoutCommand ck = _checkout.setName(name);
        ck.call();
      }
      JGitTools.LOG.info("checkout完成");
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  public MergeResult merge(final String remoteRepository, final String srcName, final String mergeBranch, final String mgs) {
    try {
      MergeResult _xblockexpression = null;
      {
        File file = this.getRepository(remoteRepository);
        MergeResult _xtrycatchfinallyexpression = null;
        try (final Git git = new Function0<Git>() {
          @Override
          public Git apply() {
            try {
              return Git.open(file);
            } catch (Throwable _e) {
              throw Exceptions.sneakyThrow(_e);
            }
          }
        }.apply()) {
          MergeResult _xblockexpression_1 = null;
          {
            Ref ref = git.checkout().setName(srcName).call();
            git.checkout().setName(mergeBranch).call();
            MergeCommand _setCommit = git.merge().include(ref).setCommit(true);
            MergeCommand merge = _setCommit.setFastForward(MergeCommand.FastForwardMode.NO_FF);
            merge.setMessage(mgs);
            _xblockexpression_1 = merge.call();
          }
          _xtrycatchfinallyexpression = _xblockexpression_1;
        }
        _xblockexpression = _xtrycatchfinallyexpression;
      }
      return _xblockexpression;
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  public PullResult pull(final String remoteRepository, final String remoteBranch) {
    try {
      PullResult _xblockexpression = null;
      {
        JGitTools.LOG.info("从仓库 {} pull => {}", remoteRepository, remoteBranch);
        File file = this.getRepository(remoteRepository);
        PullResult _xtrycatchfinallyexpression = null;
        try (final Git git = new Function0<Git>() {
          @Override
          public Git apply() {
            try {
              return Git.open(file);
            } catch (Throwable _e) {
              throw Exceptions.sneakyThrow(_e);
            }
          }
        }.apply()) {
          PullResult _xblockexpression_1 = null;
          {
            PullCommand _pull = git.pull();
            PullCommand pull = _pull.setRemoteBranchName(remoteBranch);
            this.credentials(pull);
            _xblockexpression_1 = pull.call();
          }
          _xtrycatchfinallyexpression = _xblockexpression_1;
        }
        _xblockexpression = _xtrycatchfinallyexpression;
      }
      return _xblockexpression;
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  public Ref reset(final String remoteRepository, final String objectId) {
    try {
      Ref _xblockexpression = null;
      {
        File file = this.getRepository(remoteRepository);
        Ref _xtrycatchfinallyexpression = null;
        try (final Git git = new Function0<Git>() {
          @Override
          public Git apply() {
            try {
              return Git.open(file);
            } catch (Throwable _e) {
              throw Exceptions.sneakyThrow(_e);
            }
          }
        }.apply()) {
          Ref _xblockexpression_1 = null;
          {
            Repository _repository = git.getRepository();
            RevWalk walk = new RevWalk(_repository);
            ObjectId objId = git.getRepository().resolve(objectId);
            RevCommit revCommit = walk.parseCommit(objId);
            String vision = revCommit.getParent(0).getName();
            _xblockexpression_1 = git.reset().setMode(ResetCommand.ResetType.HARD).setRef(vision).call();
          }
          _xtrycatchfinallyexpression = _xblockexpression_1;
        }
        _xblockexpression = _xtrycatchfinallyexpression;
      }
      return _xblockexpression;
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  public RevCommit revert(final String remoteRepository, final String commitId) {
    try {
      RevCommit _xblockexpression = null;
      {
        File file = this.getRepository(remoteRepository);
        RevCommit _xtrycatchfinallyexpression = null;
        try (final Git git = new Function0<Git>() {
          @Override
          public Git apply() {
            try {
              return Git.open(file);
            } catch (Throwable _e) {
              throw Exceptions.sneakyThrow(_e);
            }
          }
        }.apply()) {
          RevCommit _xblockexpression_1 = null;
          {
            Repository _repository = git.getRepository();
            RevWalk walk = new RevWalk(_repository);
            RevCommit revCommit = walk.parseCommit(ObjectId.fromString(commitId));
            _xblockexpression_1 = git.revert().include(revCommit).call();
          }
          _xtrycatchfinallyexpression = _xblockexpression_1;
        }
        _xblockexpression = _xtrycatchfinallyexpression;
      }
      return _xblockexpression;
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  public String logOne(final String remoteRepository) {
    try {
      JGitTools.LOG.info("从仓库 {} 获取第一条日志", remoteRepository);
      File file = this.getRepository(remoteRepository);
      try (final Git git = new Function0<Git>() {
        @Override
        public Git apply() {
          try {
            return Git.open(file);
          } catch (Throwable _e) {
            throw Exceptions.sneakyThrow(_e);
          }
        }
      }.apply()) {
        Iterable<RevCommit> log = git.log().call();
        final Iterable<RevCommit> _converted_log = (Iterable<RevCommit>)log;
        RevCommit revCommit = ((RevCommit[])Conversions.unwrapArray(_converted_log, RevCommit.class))[0];
        StringBuilder sb = new StringBuilder();
        sb.append(revCommit.getName()).append("\n");
        sb.append(revCommit.getCommitterIdent().getName()).append(":");
        sb.append(revCommit.getCommitterIdent().getEmailAddress()).append("\n");
        sb.append(revCommit.getCommitterIdent().getWhen()).append("\n");
        sb.append(revCommit.getFullMessage()).append("\n");
        return sb.toString();
      }
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  public List<String> logAll(final String remoteRepository) {
    try {
      JGitTools.LOG.info("从仓库 {} 获取所有日志", remoteRepository);
      File file = this.getRepository(remoteRepository);
      try (final Git git = new Function0<Git>() {
        @Override
        public Git apply() {
          try {
            return Git.open(file);
          } catch (Throwable _e) {
            throw Exceptions.sneakyThrow(_e);
          }
        }
      }.apply()) {
        Iterable<RevCommit> log = git.log().call();
        ArrayList<String> logs = new ArrayList<String>();
        for (final RevCommit revCommit : log) {
          {
            StringBuilder sb = new StringBuilder();
            sb.append(revCommit.getName()).append("\n");
            sb.append(revCommit.getCommitterIdent().getName()).append(":");
            sb.append(revCommit.getCommitterIdent().getEmailAddress()).append("\n");
            sb.append(revCommit.getCommitterIdent().getWhen()).append("\n");
            sb.append(revCommit.getFullMessage()).append("\n");
            logs.add(sb.toString());
          }
        }
        return logs;
      }
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  public boolean setAutoCommit(final boolean b) {
    return this.isAutoCommit = b;
  }
  
  @Pure
  public TransportConfigCallback getSsh() {
    return this.ssh;
  }
  
  public void setSsh(final TransportConfigCallback ssh) {
    this.ssh = ssh;
  }
  
  @Pure
  public CredentialsProvider getCp() {
    return this.cp;
  }
  
  public void setCp(final CredentialsProvider cp) {
    this.cp = cp;
  }
  
  @Pure
  public String getRepository() {
    return this.repository;
  }
  
  public void setRepository(final String repository) {
    this.repository = repository;
  }
  
  @Pure
  public boolean isAutoCommit() {
    return this.isAutoCommit;
  }
  
  public void setIsAutoCommit(final boolean isAutoCommit) {
    this.isAutoCommit = isAutoCommit;
  }
}
