/**
 * Copyright (c) 2023 murenchao
 * taomu is licensed under Mulan PubL v2.
 * You can use this software according to the terms and conditions of the Mulan PubL v2.
 * You may obtain a copy of Mulan PubL v2 at:
 *       http://license.coscl.org.cn/MulanPubL-2.0
 * THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
 * EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
 * MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
 * See the Mulan PubL v2 for more details.
 */
package cool.taomu.box.git;

import com.google.inject.Inject;
import com.jcraft.jsch.JSch;
import com.jcraft.jsch.Session;
import javax.inject.Named;
import javax.inject.Provider;
import org.eclipse.jgit.api.TransportConfigCallback;
import org.eclipse.jgit.transport.JschConfigSessionFactory;
import org.eclipse.jgit.transport.OpenSshConfig;
import org.eclipse.jgit.transport.SshSessionFactory;
import org.eclipse.jgit.transport.SshTransport;
import org.eclipse.jgit.transport.Transport;
import org.eclipse.jgit.util.FS;
import org.eclipse.xtend.lib.annotations.Accessors;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.Pure;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Accessors
@SuppressWarnings("all")
public class SshProvider implements Provider<TransportConfigCallback> {
  private static final Logger LOG = LoggerFactory.getLogger(SshProvider.class);
  
  @Inject(optional = true)
  @Named("taomubox.git.ssh.keypath")
  private String sshkeyPath;
  
  @Override
  public TransportConfigCallback get() {
    SshProvider.LOG.info("Guice从SshProvider中获取TransportConfigCallback实例");
    return new TransportConfigCallback() {
      @Override
      public void configure(final Transport transport) {
        SshTransport sshTransport = ((SshTransport) transport);
        JschConfigSessionFactory jcsf = new JschConfigSessionFactory() {
          @Override
          public void configure(final OpenSshConfig.Host host, final Session session) {
            session.setConfig("StrictHostKeyChecking", "no");
          }
          
          @Override
          public JSch createDefaultJSch(final FS fs) {
            try {
              JSch jSch = super.createDefaultJSch(fs);
              SshProvider.LOG.info("JschConfigSessionFactory设置sshkeypath:{}", SshProvider.this.sshkeyPath);
              jSch.addIdentity(SshProvider.this.sshkeyPath);
              return jSch;
            } catch (Throwable _e) {
              throw Exceptions.sneakyThrow(_e);
            }
          }
        };
        SshSessionFactory.setInstance(new JschConfigSessionFactory() {
          @Override
          public void configure(final OpenSshConfig.Host host, final Session session) {
            session.setConfig("StrictHostKeyChecking", "no");
          }
          
          @Override
          public JSch createDefaultJSch(final FS fs) {
            try {
              JSch jSch = super.createDefaultJSch(fs);
              SshProvider.LOG.info("JschConfigSessionFactory设置sshkeypath:{}", SshProvider.this.sshkeyPath);
              jSch.addIdentity(SshProvider.this.sshkeyPath);
              return jSch;
            } catch (Throwable _e) {
              throw Exceptions.sneakyThrow(_e);
            }
          }
        });
        sshTransport.setSshSessionFactory(jcsf);
      }
    };
  }
  
  @Pure
  public String getSshkeyPath() {
    return this.sshkeyPath;
  }
  
  public void setSshkeyPath(final String sshkeyPath) {
    this.sshkeyPath = sshkeyPath;
  }
}
