package cool.taomu.box.git;

import java.io.File;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.function.Consumer;
import javax.inject.Inject;
import org.eclipse.jgit.api.CheckoutCommand;
import org.eclipse.jgit.api.CloneCommand;
import org.eclipse.jgit.api.FetchCommand;
import org.eclipse.jgit.api.Git;
import org.eclipse.jgit.api.MergeCommand;
import org.eclipse.jgit.api.MergeResult;
import org.eclipse.jgit.api.PullCommand;
import org.eclipse.jgit.api.PullResult;
import org.eclipse.jgit.api.PushCommand;
import org.eclipse.jgit.api.ResetCommand;
import org.eclipse.jgit.api.Status;
import org.eclipse.jgit.api.TransportCommand;
import org.eclipse.jgit.api.TransportConfigCallback;
import org.eclipse.jgit.lib.ObjectId;
import org.eclipse.jgit.lib.Ref;
import org.eclipse.jgit.lib.Repository;
import org.eclipse.jgit.revwalk.RevCommit;
import org.eclipse.jgit.revwalk.RevWalk;
import org.eclipse.jgit.transport.CredentialsProvider;
import org.eclipse.jgit.transport.FetchResult;
import org.eclipse.jgit.transport.PushResult;
import org.eclipse.jgit.transport.RefSpec;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.Functions.Function0;
import org.eclipse.xtext.xbase.lib.IterableExtensions;

@SuppressWarnings("all")
public class GitUtils {
  @Inject
  private TransportConfigCallback ssh;
  
  @Inject
  private CredentialsProvider cp;
  
  private final String repository = "./repository";
  
  private void credentials(final TransportCommand<?, ?> command) {
    if ((this.ssh != null)) {
      command.setTransportConfigCallback(this.ssh);
    } else {
      if ((this.cp != null)) {
        command.setCredentialsProvider(this.cp);
      }
    }
  }
  
  private File getRepository(final String remoteRepository) {
    String projectName = remoteRepository;
    boolean _contains = remoteRepository.contains("/");
    if (_contains) {
      int _lastIndexOf = remoteRepository.lastIndexOf("/");
      int _plus = (_lastIndexOf + 1);
      projectName = remoteRepository.substring(_plus);
    }
    String _join = IterableExtensions.join(Collections.<String>unmodifiableList(CollectionLiterals.<String>newArrayList(this.repository, projectName)), File.separator);
    File file = new File(_join);
    boolean _exists = file.exists();
    boolean _not = (!_exists);
    if (_not) {
      file.mkdirs();
    }
    return file;
  }
  
  public String clone(final String remoteRepository, final boolean isAllBranches, final String branchOrTag) {
    try {
      File file = this.getRepository(remoteRepository);
      file.mkdirs();
      CloneCommand clone = Git.cloneRepository().setURI(remoteRepository).setBare(false);
      if (isAllBranches) {
        clone.setCloneAllBranches(isAllBranches);
      } else {
        clone.setCloneAllBranches(isAllBranches);
        clone.setBranch(branchOrTag);
      }
      clone.setDirectory(file);
      this.credentials(clone);
      try (Git git = new Function0<Git>() {
        @Override
        public Git apply() {
          try {
            return clone.call();
          } catch (Throwable _e) {
            throw Exceptions.sneakyThrow(_e);
          }
        }
      }.apply()) {
        Collection<Ref> refs = git.lsRemote().setHeads(true).setRemote(remoteRepository).call();
        StringBuffer s = new StringBuffer();
        for (final Ref ref : refs) {
          {
            s.append(ref.getName());
            s.append("\n");
          }
        }
        s.append("clone success");
        return s.toString();
      }
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  public String push(final String remoteRepository, final String name, final String branch) {
    try {
      File file = this.getRepository(remoteRepository);
      Git git = Git.open(file);
      PushCommand _push = git.push();
      RefSpec _refSpec = new RefSpec(branch);
      PushCommand push = _push.setRefSpecs(_refSpec).setRemote(name);
      this.credentials(push);
      Iterable<PushResult> infos = push.call();
      StringBuffer buffer = new StringBuffer();
      for (final PushResult info : infos) {
        {
          buffer.append(info.getMessages());
          buffer.append("\n");
        }
      }
      return buffer.toString();
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  public String fetch(final String remoteRepository) {
    try {
      File file = this.getRepository(remoteRepository);
      Collection<Ref> refs = Git.lsRemoteRepository().setHeads(true).setRemote(remoteRepository).call();
      StringBuffer log = new StringBuffer("fetch:\n");
      for (final Ref ref : refs) {
        {
          String refStr = ref.getName();
          RefSpec _refSpec = new RefSpec(((refStr + ":") + refStr));
          FetchCommand fetchCommand = Git.open(file).fetch().setRemote(remoteRepository).setRefSpecs(
            Collections.<RefSpec>unmodifiableList(CollectionLiterals.<RefSpec>newArrayList(_refSpec)));
          this.credentials(fetchCommand);
          FetchResult result = fetchCommand.call();
          log.append(result.getMessages());
          log.append("\n");
        }
      }
      log.append("Fetch success");
      return log.toString();
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  public RevCommit commit(final String remoteRepository, final String[] files, final String msg) {
    try {
      RevCommit _xblockexpression = null;
      {
        File file = this.getRepository(remoteRepository);
        final Git git = Git.open(file);
        final Consumer<String> _function = (String it) -> {
          try {
            git.add().addFilepattern(it).call();
          } catch (Throwable _e) {
            throw Exceptions.sneakyThrow(_e);
          }
        };
        ((List<String>)Conversions.doWrapArray(files)).forEach(_function);
        _xblockexpression = git.commit().setMessage(msg).call();
      }
      return _xblockexpression;
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  public HashMap<String, String> status(final String remoteRepository) {
    try {
      HashMap<String, String> map = new HashMap<String, String>();
      File file = this.getRepository(remoteRepository);
      Git git = Git.open(file);
      Status status = git.status().call();
      map.put("Added", status.getAdded().toString());
      map.put("Changed", status.getChanged().toString());
      map.put("Conflicting", status.getConflicting().toString());
      map.put("ConflictingStageState", status.getConflictingStageState().toString());
      map.put("IgnoredNotInIndex", status.getIgnoredNotInIndex().toString());
      map.put("Missing", status.getMissing().toString());
      map.put("Modified", status.getModified().toString());
      map.put("Removed", status.getRemoved().toString());
      map.put("UntrackedFiles", status.getUntracked().toString());
      map.put("UntrackedFolders", status.getUntrackedFolders().toString());
      return map;
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  public Ref branchCreate(final String remoteRepository, final String name) {
    try {
      Ref _xblockexpression = null;
      {
        File file = this.getRepository(remoteRepository);
        Git git = Git.open(file);
        _xblockexpression = git.branchCreate().setName(name).call();
      }
      return _xblockexpression;
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  public List<String> branchDelete(final String remoteRepository, final String[] names) {
    try {
      List<String> _xblockexpression = null;
      {
        File file = this.getRepository(remoteRepository);
        Git git = Git.open(file);
        _xblockexpression = git.branchDelete().setBranchNames(names).call();
      }
      return _xblockexpression;
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  public Ref branchRename(final String remoteRepository, final String oldName, final String newName) {
    try {
      Ref _xblockexpression = null;
      {
        File file = this.getRepository(remoteRepository);
        Git git = Git.open(file);
        _xblockexpression = git.branchRename().setNewName(newName).setOldName(oldName).call();
      }
      return _xblockexpression;
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  public ArrayList<String> branchList(final String remoteRepository) {
    try {
      File file = this.getRepository(remoteRepository);
      Git git = Git.open(file);
      final ArrayList<String> result = CollectionLiterals.<String>newArrayList();
      final Consumer<Ref> _function = (Ref it) -> {
        result.add(it.getName());
      };
      git.branchList().call().forEach(_function);
      return result;
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  public Ref checkout(final String remoteRepository, final String name) {
    try {
      Ref _xblockexpression = null;
      {
        File file = this.getRepository(remoteRepository);
        Git git = Git.open(file);
        CheckoutCommand _checkout = git.checkout();
        CheckoutCommand ck = _checkout.setName(name);
        _xblockexpression = ck.call();
      }
      return _xblockexpression;
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  public MergeResult merge(final String remoteRepository, final String srcName, final String mergeBranch, final String mgs) {
    try {
      MergeResult _xblockexpression = null;
      {
        File file = this.getRepository(remoteRepository);
        Git git = Git.open(file);
        Ref ref = git.checkout().setName(srcName).call();
        git.checkout().setName(mergeBranch).call();
        MergeCommand _setCommit = git.merge().include(ref).setCommit(true);
        MergeCommand merge = _setCommit.setFastForward(MergeCommand.FastForwardMode.NO_FF);
        merge.setMessage(mgs);
        _xblockexpression = merge.call();
      }
      return _xblockexpression;
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  public PullResult pull(final String remoteRepository, final String remoteBranch) {
    try {
      PullResult _xblockexpression = null;
      {
        File file = this.getRepository(remoteRepository);
        final Git git = Git.open(file);
        PullCommand _pull = git.pull();
        PullCommand pull = _pull.setRemoteBranchName(remoteBranch);
        this.credentials(pull);
        _xblockexpression = pull.call();
      }
      return _xblockexpression;
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  public Ref reset(final String remoteRepository, final String objectId) {
    try {
      Ref _xblockexpression = null;
      {
        File file = this.getRepository(remoteRepository);
        final Git git = Git.open(file);
        Repository _repository = git.getRepository();
        RevWalk walk = new RevWalk(_repository);
        ObjectId objId = git.getRepository().resolve(objectId);
        RevCommit revCommit = walk.parseCommit(objId);
        String vision = revCommit.getParent(0).getName();
        _xblockexpression = git.reset().setMode(ResetCommand.ResetType.HARD).setRef(vision).call();
      }
      return _xblockexpression;
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  public RevCommit revert(final String remoteRepository, final String commitId) {
    try {
      RevCommit _xblockexpression = null;
      {
        File file = this.getRepository(remoteRepository);
        final Git git = Git.open(file);
        Repository _repository = git.getRepository();
        RevWalk walk = new RevWalk(_repository);
        RevCommit revCommit = walk.parseCommit(ObjectId.fromString(commitId));
        _xblockexpression = git.revert().include(revCommit).call();
      }
      return _xblockexpression;
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
}
