/**
 * Copyright (c) 2023 murenchao
 * taomu is licensed under Mulan PubL v2.
 * You can use this software according to the terms and conditions of the Mulan PubL v2.
 * You may obtain a copy of Mulan PubL v2 at:
 *       http://license.coscl.org.cn/MulanPubL-2.0
 * THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
 * EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
 * MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
 * See the Mulan PubL v2 for more details.
 */
package cool.taomu.box.asm;

import com.google.common.base.Objects;
import cool.taomu.box.asm.entity.ClassEntity;
import cool.taomu.box.asm.entity.FieldEntity;
import cool.taomu.box.asm.entity.MethodEntity;
import java.lang.reflect.Method;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.ConcurrentHashMap;
import org.eclipse.xtend.lib.annotations.Accessors;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.Pure;
import org.eclipse.xtext.xbase.lib.StringExtensions;
import org.objectweb.asm.Opcodes;
import org.objectweb.asm.Type;
import org.objectweb.asm.commons.GeneratorAdapter;

@Accessors
@SuppressWarnings("all")
public class CreateMethod {
  @Accessors
  public static final class Variable {
    private Integer index;
    
    private Type type;
    
    public Variable(final Integer index, final Type type) {
      this.index = index;
      this.type = type;
    }
    
    @Pure
    public Integer getIndex() {
      return this.index;
    }
    
    public void setIndex(final Integer index) {
      this.index = index;
    }
    
    @Pure
    public Type getType() {
      return this.type;
    }
    
    public void setType(final Type type) {
      this.type = type;
    }
  }
  
  public interface Instruct {
    void coding(final CreateMethod cm);
  }
  
  private GeneratorAdapter ga;
  
  private ClassEntity classEntity;
  
  private String internalName;
  
  private final ConcurrentHashMap<String, CreateMethod.Variable> localVariable = new ConcurrentHashMap<String, CreateMethod.Variable>();
  
  public CreateMethod(final ClassEntity classEntity, final GeneratorAdapter ga) {
    this.ga = ga;
    this.classEntity = classEntity;
    String _packageName = classEntity.getPackageName();
    String _className = classEntity.getClassName();
    this.internalName = IterableExtensions.join(Collections.<String>unmodifiableList(CollectionLiterals.<String>newArrayList(_packageName, _className)), ".");
  }
  
  /**
   * TODO 以后尝试修改
   */
  public void innerClass(final String name, final int num) {
    this.classEntity.getCwriter().visitInnerClass(IterableExtensions.join(Collections.<Object>unmodifiableList(CollectionLiterals.<Object>newArrayList(name, Integer.valueOf(num))), "$"), null, null, Opcodes.ACC_PRIVATE);
  }
  
  public CreateMethod invokeVirtual(final MethodEntity method) {
    Method _method = method.getMethod();
    boolean _tripleNotEquals = (_method != null);
    if (_tripleNotEquals) {
      this.ga.invokeInterface(Type.getObjectType(method.getInternalName().replace(".", "/")), 
        org.objectweb.asm.commons.Method.getMethod(method.getMethod()));
    } else {
      String internalName = this.internalName.replace(".", "/");
      boolean _isNullOrEmpty = StringExtensions.isNullOrEmpty(method.getInternalName());
      boolean _not = (!_isNullOrEmpty);
      if (_not) {
        internalName = method.getInternalName();
      }
      this.ga.invokeVirtual(Type.getObjectType(internalName), 
        org.objectweb.asm.commons.Method.getMethod(method.getDefineMethod(), method.isDefaultPackage()));
    }
    return this;
  }
  
  public CreateMethod invokeInterface(final MethodEntity method) {
    boolean _isNullOrEmpty = StringExtensions.isNullOrEmpty(method.getDefineMethod());
    if (_isNullOrEmpty) {
      this.ga.invokeInterface(Type.getObjectType(method.getInternalName().replace(".", "/")), 
        org.objectweb.asm.commons.Method.getMethod(method.getMethod()));
    } else {
      this.ga.invokeInterface(Type.getObjectType(method.getInternalName().replace(".", "/")), 
        org.objectweb.asm.commons.Method.getMethod(method.getDefineMethod(), method.isDefaultPackage()));
    }
    return this;
  }
  
  public CreateMethod invokeStatic(final MethodEntity method) {
    boolean _isNullOrEmpty = StringExtensions.isNullOrEmpty(method.getDefineMethod());
    if (_isNullOrEmpty) {
      this.ga.invokeStatic(Type.getObjectType(method.getInternalName().replace(".", "/")), org.objectweb.asm.commons.Method.getMethod(method.getMethod()));
    } else {
      this.ga.invokeStatic(Type.getObjectType(method.getInternalName().replace(".", "/")), 
        org.objectweb.asm.commons.Method.getMethod(method.getDefineMethod(), method.isDefaultPackage()));
    }
    return this;
  }
  
  public CreateMethod invokeSpecial(final MethodEntity method) {
    boolean _isNullOrEmpty = StringExtensions.isNullOrEmpty(method.getDefineMethod());
    if (_isNullOrEmpty) {
      this.ga.invokeConstructor(Type.getObjectType(method.getInternalName().replace(".", "/")), 
        org.objectweb.asm.commons.Method.getMethod(method.getMethod()));
    } else {
      this.ga.invokeConstructor(Type.getObjectType(method.getInternalName().replace(".", "/")), 
        org.objectweb.asm.commons.Method.getMethod(method.getDefineMethod(), method.isDefaultPackage()));
    }
    return this;
  }
  
  public CreateMethod getStatic(final String internalName, final String name, final String type) {
    this.ga.getStatic(Type.getObjectType(internalName.replace(".", "/")), name, Type.getType(type));
    return this;
  }
  
  public CreateMethod getField(final FieldEntity field) {
    this.ga.loadThis();
    this.ga.getField(Type.getObjectType(this.internalName.replace(".", "/")), field.getName(), 
      Type.getType(field.getDescriptor()));
    return this;
  }
  
  public CreateMethod setField(final FieldEntity field, final CreateMethod.Instruct is) {
    this.ga.loadThis();
    if ((is != null)) {
      is.coding(this);
    }
    this.ga.putField(Type.getObjectType(this.internalName.replace(".", "/")), field.getName(), Type.getType(field.getDescriptor()));
    return this;
  }
  
  public CreateMethod store(final String name, final Class<?> zlass) {
    Type type = Type.getType(zlass);
    int index = this.ga.newLocal(type);
    this.ga.storeLocal(index, type);
    Type _type = Type.getType(zlass);
    CreateMethod.Variable _variable = new CreateMethod.Variable(Integer.valueOf(index), _type);
    this.localVariable.put(name, _variable);
    return this;
  }
  
  public CreateMethod load(final String name) {
    boolean _containsKey = this.localVariable.containsKey(name);
    if (_containsKey) {
      final CreateMethod.Variable local = this.localVariable.get(name);
      this.ga.loadLocal((local.index).intValue(), local.type);
    } else {
    }
    return this;
  }
  
  public CreateMethod This() {
    this.ga.loadThis();
    return this;
  }
  
  public CreateMethod pop() {
    this.ga.pop();
    return this;
  }
  
  protected CreateMethod _ldc(final Type type) {
    this.ga.push(type);
    return this;
  }
  
  protected CreateMethod _ldc(final String value) {
    this.ga.push(value);
    return this;
  }
  
  protected CreateMethod _ldc(final int value) {
    this.ga.push(value);
    return this;
  }
  
  protected CreateMethod _ldc(final long value) {
    this.ga.push(value);
    return this;
  }
  
  protected CreateMethod _ldc(final short value) {
    this.ga.push(value);
    return this;
  }
  
  protected CreateMethod _ldc(final double value) {
    this.ga.push(value);
    return this;
  }
  
  protected CreateMethod _ldc(final float value) {
    this.ga.push(value);
    return this;
  }
  
  protected CreateMethod _ldc(final boolean value) {
    this.ga.push(value);
    return this;
  }
  
  protected CreateMethod _ldc(final char value) {
    this.ga.push(value);
    return this;
  }
  
  public CreateMethod NEW(final Class<?> zlass) {
    this.ga.newInstance(Type.getType(zlass));
    return this;
  }
  
  public CreateMethod dup() {
    this.ga.dup();
    return this;
  }
  
  public CreateMethod returnValue() {
    this.ga.returnValue();
    return this;
  }
  
  public CreateMethod endMethod() {
    this.ga.endMethod();
    return this;
  }
  
  public CreateMethod checkCast(final Class<?> type) {
    this.ga.checkCast(Type.getType(type));
    return this;
  }
  
  public CreateMethod invokeVirtual(final Class<?> zlass, final String method) {
    return this.invokeVirtual(zlass.getName(), method);
  }
  
  public CreateMethod invokeVirtual(final String internalName, final String method) {
    return this.invokeVirtual(internalName, method, true);
  }
  
  public CreateMethod invokeVirtual(final String internalName, final String method, final boolean defaultPackage) {
    this.ga.invokeVirtual(Type.getObjectType(internalName.replace(".", "/")), org.objectweb.asm.commons.Method.getMethod(method, defaultPackage));
    return this;
  }
  
  public CreateMethod buildReturnValue(final Method m) {
    Class<?> _returnType = m.getReturnType();
    boolean _matched = false;
    if (Objects.equal(_returnType, int.class)) {
      _matched=true;
      this.checkCast(Integer.class);
      this.invokeVirtual(Integer.class, "int intValue()");
    }
    if (!_matched) {
      if (Objects.equal(_returnType, long.class)) {
        _matched=true;
        this.checkCast(Long.class);
        this.invokeVirtual(Long.class, "long longValue()");
      }
    }
    if (!_matched) {
      if (Objects.equal(_returnType, double.class)) {
        _matched=true;
        this.checkCast(Double.class);
        this.invokeVirtual(Double.class, "double doubleValue()");
      }
    }
    if (!_matched) {
      if (Objects.equal(_returnType, float.class)) {
        _matched=true;
        this.checkCast(Float.class);
        this.invokeVirtual(Float.class, "float floatValue()");
      }
    }
    if (!_matched) {
      if (Objects.equal(_returnType, boolean.class)) {
        _matched=true;
        this.checkCast(Boolean.class);
        this.invokeVirtual(Boolean.class, "boolean booleanValue()");
      }
    }
    if (!_matched) {
      if (Objects.equal(_returnType, short.class)) {
        _matched=true;
        this.checkCast(Short.class);
        this.invokeVirtual(Short.class, "short shortValue()");
      }
    }
    if (!_matched) {
      if (Objects.equal(_returnType, char.class)) {
        _matched=true;
        this.checkCast(Character.class);
        this.invokeVirtual(Character.class, "char charValue()");
      }
    }
    if (!_matched) {
      if (Objects.equal(_returnType, byte.class)) {
        _matched=true;
        this.checkCast(Byte.class);
        this.invokeVirtual(Byte.class, "byte byteValue()");
      }
    }
    if (!_matched) {
      if (Objects.equal(_returnType, void.class)) {
        _matched=true;
      }
    }
    if (!_matched) {
      this.checkCast(m.getReturnType());
    }
    return this;
  }
  
  public CreateMethod buildArguments(final org.objectweb.asm.commons.Method method) {
    int _size = ((List<Type>)Conversions.doWrapArray(method.getArgumentTypes())).size();
    boolean _greaterThan = (_size > 0);
    if (_greaterThan) {
      for (int index = 0; (index < ((List<Type>)Conversions.doWrapArray(method.getArgumentTypes())).size()); index++) {
        {
          this.ga.dup();
          this.ga.push(index);
          String _descriptor = (method.getArgumentTypes()[index]).getDescriptor();
          if (_descriptor != null) {
            switch (_descriptor) {
              case "Z":
                this.ga.loadArg(index);
                this.invokeStatic(Boolean.class, "Boolean valueOf(boolean)");
                break;
              case "C":
                this.ga.loadArg(index);
                this.invokeStatic(Character.class, "Character valueOf(char)");
                break;
              case "B":
                this.ga.loadArg(index);
                this.invokeStatic(Byte.class, "Byte valueOf(byte)");
                break;
              case "S":
                this.ga.loadArg(index);
                this.invokeStatic(Short.class, "Short valueOf(short)");
                break;
              case "I":
                this.ga.loadArg(index);
                this.invokeStatic(Integer.class, "Integer valueOf(int)");
                break;
              case "F":
                this.ga.loadArg(index);
                this.invokeStatic(Float.class, "Float valueOf(float)");
                break;
              case "J":
                this.ga.loadArg(index);
                this.invokeStatic(Float.class, "Long valueOf(long)");
                break;
              case "D":
                this.ga.loadArg(index);
                this.invokeStatic(Float.class, "Double valueOf(double)");
                break;
              default:
                this.ga.visitVarInsn(Opcodes.ALOAD, (index + 1));
                break;
            }
          } else {
            this.ga.visitVarInsn(Opcodes.ALOAD, (index + 1));
          }
          this.ga.visitInsn(Opcodes.AASTORE);
        }
      }
    } else {
      this.ga.visitInsn(Opcodes.ACONST_NULL);
    }
    return this;
  }
  
  public CreateMethod invokeStatic(final Class<?> type, final String method) {
    return this.invokeStatic(type.getName(), method, true);
  }
  
  public CreateMethod invokeStatic(final String internalName, final String method, final boolean defaultPackage) {
    this.ga.invokeStatic(Type.getObjectType(internalName.replace(".", "/")), org.objectweb.asm.commons.Method.getMethod(method, defaultPackage));
    return this;
  }
  
  public CreateMethod ldc(final Object value) {
    if (value instanceof Double) {
      return _ldc((Double)value);
    } else if (value instanceof Float) {
      return _ldc((Float)value);
    } else if (value instanceof Integer) {
      return _ldc((Integer)value);
    } else if (value instanceof Long) {
      return _ldc((Long)value);
    } else if (value instanceof Short) {
      return _ldc((Short)value);
    } else if (value instanceof Boolean) {
      return _ldc((Boolean)value);
    } else if (value instanceof Character) {
      return _ldc((Character)value);
    } else if (value instanceof String) {
      return _ldc((String)value);
    } else if (value instanceof Type) {
      return _ldc((Type)value);
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(value).toString());
    }
  }
  
  @Pure
  public GeneratorAdapter getGa() {
    return this.ga;
  }
  
  public void setGa(final GeneratorAdapter ga) {
    this.ga = ga;
  }
  
  @Pure
  public ClassEntity getClassEntity() {
    return this.classEntity;
  }
  
  public void setClassEntity(final ClassEntity classEntity) {
    this.classEntity = classEntity;
  }
  
  @Pure
  public String getInternalName() {
    return this.internalName;
  }
  
  public void setInternalName(final String internalName) {
    this.internalName = internalName;
  }
  
  @Pure
  public ConcurrentHashMap<String, CreateMethod.Variable> getLocalVariable() {
    return this.localVariable;
  }
}
