/**
 * Copyright (c) 2021-2022 murenchao
 * fig is licensed under Mulan PubL v2.
 * You can use this software according to the terms and conditions of the Mulan PubL v2.
 * You may obtain a copy of Mulan PubL v2 at:
 *       http://license.coscl.org.cn/MulanPubL-2.0
 * THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
 * EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
 * MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
 * See the Mulan PubL v2 for more details.
 */
package cool.taomu.box.asm.aspect2;

import java.util.HashMap;
import java.util.Map;
import org.eclipse.xtend.lib.annotations.Accessors;
import org.eclipse.xtext.xbase.lib.Pure;
import org.objectweb.asm.ClassVisitor;
import org.objectweb.asm.Label;
import org.objectweb.asm.MethodVisitor;
import org.objectweb.asm.Opcodes;
import org.objectweb.asm.Type;
import org.objectweb.asm.commons.GeneratorAdapter;
import org.objectweb.asm.commons.Method;

@SuppressWarnings("all")
public class FigGeneratorAdapter {
  public interface FigAsm {
    void code(final FigGeneratorAdapter arg0);
  }
  
  @Accessors
  public static class Variable {
    private int index;
    
    private Type type;
    
    public Variable(final int index, final Type type) {
      this.index = index;
      this.type = type;
    }
    
    @Pure
    public int getIndex() {
      return this.index;
    }
    
    public void setIndex(final int index) {
      this.index = index;
    }
    
    @Pure
    public Type getType() {
      return this.type;
    }
    
    public void setType(final Type type) {
      this.type = type;
    }
  }
  
  private Map<String, FigGeneratorAdapter.Variable> fields = new HashMap<String, FigGeneratorAdapter.Variable>();
  
  private Map<String, Label> labels = new HashMap<String, Label>();
  
  private GeneratorAdapter ga = null;
  
  private MethodVisitor mv = null;
  
  public FigGeneratorAdapter(final int access, final Method method, final String signature, final Type[] exceptions, final ClassVisitor classVisitor) {
    this.mv = classVisitor.visitMethod(access, method.getName(), method.getDescriptor(), signature, null);
    GeneratorAdapter _generatorAdapter = new GeneratorAdapter(access, method, this.mv);
    this.ga = _generatorAdapter;
  }
  
  public Label newLables(final String key) {
    Label _label = new Label();
    return this.labels.put(key, _label);
  }
  
  public void visitTryCatchBlock(final String start, final String end, final String catchBlock) {
    this.visitTryCatchBlock(start, end, catchBlock, null);
  }
  
  public void visitTryCatchBlock(final String start, final String end, final String catchBlock, final Class<?> zlass) {
    Label l0 = this.labels.get(start);
    Label l1 = this.labels.get(end);
    Label cb = this.labels.get(catchBlock);
    if ((zlass == null)) {
      this.mv.visitTryCatchBlock(l0, l1, cb, null);
    } else {
      this.mv.visitTryCatchBlock(l0, l1, cb, Type.getType(zlass).getInternalName());
    }
  }
  
  public void aconst_null() {
    this.ga.visitInsn(Opcodes.ACONST_NULL);
  }
  
  public void mark(final String key) {
    this.ga.mark(this.labels.get(key));
  }
  
  public void goTo(final String key) {
    this.ga.goTo(this.labels.get(key));
  }
  
  public FigGeneratorAdapter.Variable newInstance(final String name, final Class<?> type) {
    FigGeneratorAdapter.Variable _xblockexpression = null;
    {
      Type classType = Type.getType(type);
      this.ga.newInstance(classType);
      this.ga.dup();
      this.ga.invokeConstructor(classType, Method.getMethod("void <init>()"));
      int l1 = this.ga.newLocal(classType);
      this.ga.storeLocal(l1, classType);
      FigGeneratorAdapter.Variable _variable = new FigGeneratorAdapter.Variable(l1, classType);
      _xblockexpression = this.fields.put(name, _variable);
    }
    return _xblockexpression;
  }
  
  public FigGeneratorAdapter.Variable newArray(final int size, final Class<?> type, final FigGeneratorAdapter.FigAsm asm) {
    return this.newArray(null, size, type, asm);
  }
  
  public FigGeneratorAdapter.Variable newArray(final String name, final int size, final Class<?> type, final FigGeneratorAdapter.FigAsm asm) {
    FigGeneratorAdapter.Variable _xblockexpression = null;
    {
      Type classType = Type.getType(type);
      this.ga.push(size);
      this.ga.newArray(classType);
      asm.code(this);
      int l1 = this.ga.newLocal(classType);
      FigGeneratorAdapter.Variable _xifexpression = null;
      if ((name != null)) {
        FigGeneratorAdapter.Variable _xblockexpression_1 = null;
        {
          this.ga.storeLocal(l1, classType);
          FigGeneratorAdapter.Variable _variable = new FigGeneratorAdapter.Variable(l1, classType);
          _xblockexpression_1 = this.fields.put(name, _variable);
        }
        _xifexpression = _xblockexpression_1;
      }
      _xblockexpression = _xifexpression;
    }
    return _xblockexpression;
  }
  
  public FigGeneratorAdapter.Variable store(final String name) {
    return this.store(name, Object.class);
  }
  
  public FigGeneratorAdapter.Variable store(final String name, final Class<?> type) {
    FigGeneratorAdapter.Variable _xblockexpression = null;
    {
      Type classType = Type.getType(type);
      int _newLocal = this.ga.newLocal(classType);
      FigGeneratorAdapter.Variable _variable = new FigGeneratorAdapter.Variable(_newLocal, classType);
      FigGeneratorAdapter.Variable v = this.fields.getOrDefault(name, _variable);
      this.ga.storeLocal(v.index, v.type);
      _xblockexpression = this.fields.put(name, v);
    }
    return _xblockexpression;
  }
  
  public void ifnull(final String key) {
    this.ga.ifNull(this.labels.get(key));
  }
  
  public void ifnonnull(final String key) {
    this.ga.ifNonNull(this.labels.get(key));
  }
  
  public void ifacmpeq(final String key) {
    this.ga.ifCmp(Type.getType(Object.class), Opcodes.IF_ACMPEQ, this.labels.get(key));
  }
  
  public void loadLocal(final String name) {
    FigGeneratorAdapter.Variable v = this.fields.get(name);
    this.ga.loadLocal(v.index, v.type);
  }
  
  public void athrow() {
    this.ga.throwException();
  }
  
  public void invokeVirtual(final Class<?> type, final String method, final boolean defaultPackage) {
    this.ga.invokeVirtual(Type.getType(type), Method.getMethod(method, defaultPackage));
  }
  
  public void invokeVirtual(final Class<?> type, final String method) {
    this.invokeVirtual(type, method, false);
  }
  
  public void invokeVirtual(final Class<?> type, final java.lang.reflect.Method method) {
    this.ga.invokeVirtual(Type.getType(type), Method.getMethod(method));
  }
  
  public void invokeStatic(final Class<?> type, final String method, final boolean defaultPackage) {
    this.ga.invokeStatic(Type.getType(type), Method.getMethod(method, defaultPackage));
  }
  
  public void invokeStatic(final Class<?> type, final String method) {
    this.invokeStatic(type, method, false);
  }
  
  public void dup() {
    this.ga.dup();
  }
  
  public void push(final Class<?> zlass) {
    this.ga.push(Type.getType(zlass));
  }
  
  public void push(final Type type) {
    this.ga.push(type);
  }
  
  public void push(final String value) {
    this.ga.push(value);
  }
  
  public void push(final int value) {
    this.ga.push(value);
  }
  
  public void returnValue() {
    this.ga.returnValue();
  }
  
  public void endMethod() {
    this.ga.endMethod();
  }
  
  public void checkCast(final Class<?> type) {
    this.ga.checkCast(Type.getType(type));
  }
  
  public void visitInsn(final int i) {
    this.ga.visitInsn(i);
  }
  
  public void visitVarInsn(final int opcode, final int v) {
    this.ga.visitVarInsn(opcode, v);
  }
  
  public void loadArg(final int arg) {
    this.ga.loadArg(arg);
  }
}
