/**
 * Copyright (c) 2023 murenchao
 * taomu is licensed under Mulan PubL v2.
 * You can use this software according to the terms and conditions of the Mulan PubL v2.
 * You may obtain a copy of Mulan PubL v2 at:
 *       http://license.coscl.org.cn/MulanPubL-2.0
 * THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
 * EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
 * MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
 * See the Mulan PubL v2 for more details.
 */
package cool.taomu.box.asm;

import cool.taomu.box.asm.entity.ClassEntity;
import cool.taomu.box.asm.entity.FieldEntity;
import cool.taomu.box.asm.entity.MethodEntity;
import java.io.PrintWriter;
import java.lang.reflect.Constructor;
import java.lang.reflect.Parameter;
import java.util.List;
import java.util.function.Consumer;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.objectweb.asm.ClassVisitor;
import org.objectweb.asm.ClassWriter;
import org.objectweb.asm.Opcodes;
import org.objectweb.asm.Type;
import org.objectweb.asm.commons.GeneratorAdapter;
import org.objectweb.asm.commons.Method;
import org.objectweb.asm.util.TraceClassVisitor;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@SuppressWarnings("all")
public class CreateClass extends ClassLoader {
  private static final Logger LOG = LoggerFactory.getLogger(CreateClass.class);
  
  private ClassWriter cwriter;
  
  private ClassVisitor tcvisitor;
  
  private GeneratorAdapter ga;
  
  private ClassEntity classEntity;
  
  public CreateClass(final ClassEntity classEntity, final int version, final int access) {
    this(classEntity, version, access, false);
  }
  
  public CreateClass(final ClassEntity classEntity, final int version, final int access, final boolean isDebug) {
    ClassWriter _classWriter = new ClassWriter((ClassWriter.COMPUTE_MAXS + ClassWriter.COMPUTE_FRAMES));
    this.cwriter = _classWriter;
    if (isDebug) {
      PrintWriter _printWriter = new PrintWriter(System.err);
      TraceClassVisitor _traceClassVisitor = new TraceClassVisitor(this.cwriter, _printWriter);
      this.tcvisitor = _traceClassVisitor;
    } else {
      this.tcvisitor = this.cwriter;
    }
    this.classEntity = classEntity;
    this.classEntity.setCwriter(this.cwriter);
    this.tcvisitor.visit(version, access, classEntity.getFullName(), null, classEntity.getSuperclass(), classEntity.getInterfaces());
  }
  
  public static CreateClass Class(final ClassEntity classEntity) {
    int _version = classEntity.getVersion();
    return new CreateClass(classEntity, _version, Opcodes.ACC_PUBLIC, false);
  }
  
  public static CreateClass Class(final ClassEntity classEntity, final boolean isDebug) {
    int _version = classEntity.getVersion();
    return new CreateClass(classEntity, _version, Opcodes.ACC_PUBLIC, isDebug);
  }
  
  public static CreateClass Class(final ClassEntity classEntity, final int version, final boolean isDebug) {
    return new CreateClass(classEntity, version, Opcodes.ACC_PUBLIC, isDebug);
  }
  
  public static CreateClass Class(final ClassEntity classEntity, final int version) {
    return new CreateClass(classEntity, version, Opcodes.ACC_PUBLIC);
  }
  
  public static CreateClass Interface(final ClassEntity classEntity, final int version) {
    return new CreateClass(classEntity, version, ((Opcodes.ACC_PUBLIC + Opcodes.ACC_ABSTRACT) + Opcodes.ACC_INTERFACE));
  }
  
  public static CreateClass Abstract(final ClassEntity classEntity, final int version) {
    return new CreateClass(classEntity, version, (Opcodes.ACC_PUBLIC + Opcodes.ACC_ABSTRACT));
  }
  
  public void field(final FieldEntity field) {
    this.tcvisitor.visitField(field.getAccess(), field.getName(), field.getDescriptor(), null, field.getValue()).visitEnd();
  }
  
  public CreateClass constructor(final Constructor<?>[] constructor) {
    final Consumer<Constructor<?>> _function = (Constructor<?> it) -> {
      Method m = Method.getMethod(it);
      GeneratorAdapter ga = new GeneratorAdapter(Opcodes.ACC_PUBLIC, m, null, null, this.tcvisitor);
      ga.loadThis();
      int _size = ((List<Parameter>)Conversions.doWrapArray(it.getParameters())).size();
      boolean _greaterThan = (_size > 0);
      if (_greaterThan) {
        int _size_1 = ((List<Parameter>)Conversions.doWrapArray(it.getParameters())).size();
        int _minus = (_size_1 - 1);
        ga.loadArgs(0, _minus);
      }
      ga.invokeConstructor(Type.getObjectType(this.classEntity.getSuperclass()), m);
      ga.returnValue();
      ga.endMethod();
    };
    IterableExtensions.<Constructor<?>>filterNull(((Iterable<Constructor<?>>)Conversions.doWrapArray(constructor))).forEach(_function);
    return this;
  }
  
  public CreateMethod method(final MethodEntity method) {
    int _access = method.getAccess();
    Method _method = Method.getMethod(method.getDefineMethod(), method.isDefaultPackage());
    GeneratorAdapter _generatorAdapter = new GeneratorAdapter(_access, _method, null, 
      null, this.tcvisitor);
    this.ga = _generatorAdapter;
    return new CreateMethod(this.classEntity, this.ga);
  }
  
  public CreateClass end() {
    this.tcvisitor.visitEnd();
    return this;
  }
  
  public byte[] toByteArray() {
    return this.cwriter.toByteArray();
  }
  
  @Override
  public Class<?> findClass(final String name) {
    byte[] bytes = this.toByteArray();
    CreateClass.LOG.info("执行findClass:{}", name);
    return super.defineClass(name, bytes, 0, bytes.length);
  }
}
