/**
 * Copyright (c) 2023 murenchao
 * taomu is licensed under Mulan PubL v2.
 * You can use this software according to the terms and conditions of the Mulan PubL v2.
 * You may obtain a copy of Mulan PubL v2 at:
 *       http://license.coscl.org.cn/MulanPubL-2.0
 * THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
 * EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
 * MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
 * See the Mulan PubL v2 for more details.
 */
package cool.taomu.box.jersery.jetty;

import com.google.inject.Inject;
import com.google.inject.Injector;
import com.google.inject.name.Named;
import com.google.inject.servlet.GuiceFilter;
import com.google.inject.servlet.GuiceServletContextListener;
import cool.taomu.box.jersery.GuiceResourceConfig;
import cool.taomu.box.jersery.JerseryGuiceInjector;
import java.util.Collections;
import java.util.EnumSet;
import javax.inject.Provider;
import javax.servlet.DispatcherType;
import org.apache.commons.lang3.StringUtils;
import org.eclipse.jetty.server.ConnectionFactory;
import org.eclipse.jetty.server.Handler;
import org.eclipse.jetty.server.HttpConfiguration;
import org.eclipse.jetty.server.HttpConnectionFactory;
import org.eclipse.jetty.server.Server;
import org.eclipse.jetty.server.ServerConnector;
import org.eclipse.jetty.server.handler.HandlerCollection;
import org.eclipse.jetty.server.handler.HandlerList;
import org.eclipse.jetty.server.handler.HandlerWrapper;
import org.eclipse.jetty.servlet.DefaultServlet;
import org.eclipse.jetty.servlet.ServletHolder;
import org.eclipse.jetty.webapp.WebAppContext;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.glassfish.jersey.servlet.ServletContainer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@SuppressWarnings("all")
public class JettyProvider implements Provider<Server> {
  private static final Logger LOG = LoggerFactory.getLogger(JettyProvider.class);
  
  @Inject(optional = true)
  @Named("resourceConfig")
  private String resourceConfig = GuiceResourceConfig.class.getName();
  
  @Inject(optional = true)
  @Named("taomubox.http.name")
  private String name;
  
  @Inject(optional = true)
  @Named("taomubox.http.host")
  private String host = "0.0.0.0";
  
  @Inject(optional = true)
  @Named("taomubox.http.port")
  private int port = 8080;
  
  @Inject(optional = true)
  @Named("taomubox.http.path")
  private String contextPath = "/";
  
  @Inject(optional = true)
  @Named("taomubox.http.static.path")
  private String staticBasePath = null;
  
  private HttpConfiguration httpConfiguration = new HttpConfiguration();
  
  private Server server;
  
  @Inject
  private JerseryGuiceInjector supplier;
  
  @Override
  public Server get() {
    Server _xblockexpression = null;
    {
      boolean _startsWith = this.contextPath.startsWith("/");
      boolean _not = (!_startsWith);
      if (_not) {
        this.contextPath = ("/" + this.contextPath);
      }
      if (((this.name == null) || this.name.equals(""))) {
        this.name = String.format("%s-%s", this.host, Integer.valueOf(this.port));
      }
      Server _server = new Server();
      this.server = _server;
      ServerConnector connector = new ServerConnector(this.server);
      connector.setName(this.name);
      connector.setHost(this.host);
      connector.setPort(this.port);
      HttpConnectionFactory _httpConnectionFactory = new HttpConnectionFactory(this.httpConfiguration);
      connector.setConnectionFactories(Collections.<ConnectionFactory>singleton(_httpConnectionFactory));
      this.server.addConnector(connector);
      WebAppContext webAppContext = new WebAppContext();
      webAppContext.setServer(this.server);
      webAppContext.addFilter(GuiceFilter.class, "/*", EnumSet.<DispatcherType>allOf(DispatcherType.class));
      ServletHolder holder = new ServletHolder(ServletContainer.class);
      holder.setInitParameter("javax.ws.rs.Application", this.resourceConfig);
      webAppContext.addServlet(holder, "/*");
      webAppContext.setResourceBase("/");
      webAppContext.setContextPath(this.contextPath);
      webAppContext.addEventListener(new GuiceServletContextListener() {
        @Override
        protected Injector getInjector() {
          return JettyProvider.this.supplier.getInjectorSupplier().get();
        }
      });
      ServletHolder defaultServlet = new ServletHolder("default", DefaultServlet.class);
      String _property = System.getProperty("user.dir");
      String _defaultIfBlank = StringUtils.<String>defaultIfBlank(this.staticBasePath, "static");
      String staticDir = IterableExtensions.join(Collections.<String>unmodifiableList(CollectionLiterals.<String>newArrayList(_property, _defaultIfBlank)), "/");
      JettyProvider.LOG.info("web static file directory : {}", staticDir);
      defaultServlet.setInitParameter(
        "resourceBase", staticDir);
      webAppContext.addServlet(defaultServlet, "/static/*");
      JettyProvider.setHandler(this.server, webAppContext);
      _xblockexpression = this.server;
    }
    return _xblockexpression;
  }
  
  public static void setHandler(final HandlerWrapper handlerWrapper, final Handler handlerToAdd) {
    Handler currentInnerHandler = handlerWrapper.getHandler();
    if ((currentInnerHandler == null)) {
      handlerWrapper.setHandler(handlerToAdd);
    } else {
      if ((currentInnerHandler instanceof HandlerCollection)) {
        ((HandlerCollection) currentInnerHandler).addHandler(handlerToAdd);
      } else {
        if ((currentInnerHandler instanceof HandlerWrapper)) {
          JettyProvider.setHandler(((HandlerWrapper) currentInnerHandler), handlerToAdd);
        } else {
          HandlerList handlerList = new HandlerList();
          handlerList.addHandler(currentInnerHandler);
          handlerList.addHandler(handlerToAdd);
          handlerWrapper.setHandler(handlerWrapper);
        }
      }
    }
  }
}
