/**
 * Copyright (c) 2023 murenchao
 * taomu is licensed under Mulan PubL v2.
 * You can use this software according to the terms and conditions of the Mulan PubL v2.
 * You may obtain a copy of Mulan PubL v2 at:
 *       http://license.coscl.org.cn/MulanPubL-2.0
 * THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
 * EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
 * MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
 * See the Mulan PubL v2 for more details.
 */
package cool.taomu.box.job.server.controller.impl;

import com.github.mustachejava.DefaultMustacheFactory;
import com.github.mustachejava.Mustache;
import com.google.common.base.Objects;
import com.google.gson.Gson;
import cool.taomu.box.git.JGitTools;
import cool.taomu.box.job.server.controller.inter.ITaskController;
import cool.taomu.box.job.server.entity.HtmlTemplateEntity;
import cool.taomu.box.job.server.entity.TaskEntity;
import cool.taomu.box.job.server.entity.TaskExtendColumnEntity;
import cool.taomu.box.job.server.service.TaskService;
import cool.taomu.box.job.server.utils.ChineseToPinyin;
import cool.taomu.box.job.server.utils.GetTaskDirectory;
import cool.taomu.box.job.server.utils.MarkdownUtils;
import cool.taomu.box.job.server.utils.ShellExecute;
import cool.taomu.box.script.rhino.JsRhinoScript;
import java.io.File;
import java.io.StringWriter;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;
import java.util.function.Function;
import java.util.stream.Collectors;
import javax.inject.Inject;
import javax.ws.rs.Path;
import javax.ws.rs.core.Response;
import org.apache.commons.io.FileUtils;
import org.apache.commons.lang3.StringUtils;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.Pair;
import org.eclipse.xtext.xbase.lib.StringExtensions;
import org.joda.time.DateTime;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Path("job")
@SuppressWarnings("all")
public class TaskController implements ITaskController<TaskEntity> {
  private static final Logger LOG = LoggerFactory.getLogger(TaskController.class);
  
  @Inject
  private TaskService taskService;
  
  @Inject
  private JGitTools jgit;
  
  @Override
  public Response getTask(final String uuid) {
    try {
      TaskController.LOG.info("获取 {} ID的任务", uuid);
      TaskEntity entity = this.taskService.getOneTask(uuid);
      HtmlTemplateEntity htemplate = new Gson().<HtmlTemplateEntity>fromJson(entity.getHtmlTemplate(), HtmlTemplateEntity.class);
      String _mustacheText = htemplate.getMustacheText();
      File tempFile = new File(_mustacheText);
      htemplate.setMustacheText(FileUtils.readFileToString(tempFile, "UTF-8"));
      String _analysisShellText = htemplate.getAnalysisShellText();
      File _file = new File(_analysisShellText);
      htemplate.setAnalysisShellText(FileUtils.readFileToString(_file, "UTF-8"));
      entity.setHtmlTemplate(new Gson().toJson(htemplate));
      String _shellPath = entity.getShellPath();
      File _file_1 = new File(_shellPath);
      entity.setShellPath(FileUtils.readFileToString(_file_1, "UTF-8"));
      return Response.ok(entity).build();
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  @Override
  public Response putTask(final String uuid, final TaskEntity entity) {
    try {
      TaskController.LOG.info("修改 {} ID的任务", uuid);
      TaskEntity old = this.taskService.getOneTask(uuid);
      String _shellPath = old.getShellPath();
      File _file = new File(_shellPath);
      FileUtils.writeStringToFile(_file, entity.getShellPath(), "UTF-8", false);
      HtmlTemplateEntity htemplate = new Gson().<HtmlTemplateEntity>fromJson(old.getHtmlTemplate(), HtmlTemplateEntity.class);
      HtmlTemplateEntity inputHtemplate = new Gson().<HtmlTemplateEntity>fromJson(entity.getHtmlTemplate(), HtmlTemplateEntity.class);
      String _mustacheText = htemplate.getMustacheText();
      File _file_1 = new File(_mustacheText);
      FileUtils.writeStringToFile(_file_1, inputHtemplate.getMustacheText(), "UTF-8", false);
      String _analysisShellText = htemplate.getAnalysisShellText();
      File _file_2 = new File(_analysisShellText);
      FileUtils.writeStringToFile(_file_2, inputHtemplate.getAnalysisShellText(), "UTF-8", 
        false);
      Integer _taskVersion = old.getTaskVersion();
      int _plus = ((_taskVersion).intValue() + 1);
      old.setTaskVersion(Integer.valueOf(_plus));
      this.taskService.update(old);
      HashMap<JGitTools.StatusType, Set<String>> gitStatus = this.jgit.status(old.getTaskDirectory(), JGitTools.StatusType.Modified);
      int _size = gitStatus.get(JGitTools.StatusType.Modified).size();
      boolean _greaterThan = (_size > 0);
      if (_greaterThan) {
        Integer _taskVersion_1 = old.getTaskVersion();
        this.jgit.addAndCommit(entity.getTaskDirectory(), IterableExtensions.join(Collections.<Object>unmodifiableList(CollectionLiterals.<Object>newArrayList("v", _taskVersion_1))));
      }
      return Response.status(201).build();
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  @Override
  public Response deleteTask(final String uuid) {
    TaskController.LOG.info("删除任务:{}", uuid);
    TaskEntity oldTask = this.taskService.getOneTask(uuid);
    String _trim = oldTask.getTaskDirectory().trim();
    File dir = new File(_trim);
    TaskController.LOG.info("要删除的目录为: {}", dir.getAbsolutePath());
    boolean _isNullOrEmpty = StringExtensions.isNullOrEmpty(oldTask.getTaskDirectory());
    boolean _not = (!_isNullOrEmpty);
    if (_not) {
      try {
        boolean _deleteQuietly = FileUtils.deleteQuietly(dir);
        if (_deleteQuietly) {
          TaskController.LOG.info("删除任务 {} 目录成功", dir.getAbsolutePath());
          int _deleteOneTask = this.taskService.deleteOneTask(oldTask.getTaskId());
          boolean _greaterThan = (_deleteOneTask > 0);
          if (_greaterThan) {
            TaskController.LOG.info("删除任务 {} 成功", uuid);
            return Response.status(200).build();
          } else {
            TaskController.LOG.info("删除任务 {} 失败", uuid);
            return Response.status(402).build();
          }
        } else {
          TaskController.LOG.info("删除任务 {} 目录失败", dir.getAbsolutePath());
          return Response.status(402).build();
        }
      } catch (final Throwable _t) {
        if (_t instanceof Exception) {
          final Exception ex = (Exception)_t;
          ex.printStackTrace();
          return Response.status(500).build();
        } else {
          throw Exceptions.sneakyThrow(_t);
        }
      }
    }
    return null;
  }
  
  @Override
  public Response postTask(final TaskEntity entity) {
    try {
      TaskEntity sentity = this.taskService.getOneTaskByTaskName(entity.getTaskName());
      if ((sentity != null)) {
        return Response.status(406).build();
      }
      entity.setTaskName(entity.getTaskName().trim());
      String taskNamePinyin = ChineseToPinyin.toPinyin(entity.getTaskName());
      TaskController.LOG.info("任务名:{} 转为拼音:{}", entity.getTaskName(), taskNamePinyin);
      String _property = System.getProperty("user.dir");
      String _join = IterableExtensions.join(Collections.<String>unmodifiableList(CollectionLiterals.<String>newArrayList(_property, taskNamePinyin)), "/");
      File taskNameDir = new File(_join);
      boolean _exists = taskNameDir.exists();
      boolean _not = (!_exists);
      if (_not) {
        boolean _mkdirs = taskNameDir.mkdirs();
        if (_mkdirs) {
          TaskController.LOG.info("创建目录:{}", taskNameDir.getAbsolutePath());
          entity.setTaskDirectory(taskNameDir.getAbsolutePath().trim());
          this.jgit.init(entity.getTaskDirectory());
        }
      }
      TaskController.LOG.info("任务目录:{}", taskNameDir);
      entity.setTaskId(UUID.randomUUID().toString());
      entity.setClientId(UUID.randomUUID().toString());
      File dir = GetTaskDirectory.getScriptPath(entity, null);
      File resdir = GetTaskDirectory.getResultPath(entity, null);
      File tempdir = GetTaskDirectory.getTemplatePath(entity, null);
      boolean _exists_1 = dir.exists();
      boolean _not_1 = (!_exists_1);
      if (_not_1) {
        TaskController.LOG.info("任务目录不存在");
        boolean _mkdirs_1 = dir.mkdirs();
        if (_mkdirs_1) {
          TaskController.LOG.info("创建目录:{}", dir.getAbsolutePath());
        }
        boolean _mkdirs_2 = resdir.mkdirs();
        if (_mkdirs_2) {
          TaskController.LOG.info("创建目录:{}", resdir.getAbsolutePath());
        }
        boolean _mkdirs_3 = tempdir.mkdirs();
        if (_mkdirs_3) {
          TaskController.LOG.info("创建目录:{}", tempdir.getAbsolutePath());
          File analysisDir = GetTaskDirectory.getAnalysisPath(entity, null);
          boolean _mkdirs_4 = analysisDir.mkdirs();
          if (_mkdirs_4) {
            TaskController.LOG.info("创建目录:{}", analysisDir.getAbsolutePath());
          }
        }
      }
      boolean _exists_2 = tempdir.exists();
      if (_exists_2) {
        String tempName = UUID.randomUUID().toString().replace("-", "");
        TaskController.LOG.info("生成模板文件:{}", tempName);
        HtmlTemplateEntity htemplate = new Gson().<HtmlTemplateEntity>fromJson(entity.getHtmlTemplate(), HtmlTemplateEntity.class);
        File tempFile = GetTaskDirectory.getTemplatePath(entity, IterableExtensions.join(Collections.<String>unmodifiableList(CollectionLiterals.<String>newArrayList(tempName, "mustache")), "."));
        TaskController.LOG.info("生成模板文件:{}", tempName);
        FileUtils.writeStringToFile(tempFile, StringUtils.<String>defaultIfEmpty(htemplate.getMustacheText(), ""), "UTF-8", false);
        htemplate.setMustacheText(tempFile.getAbsolutePath().trim());
        TaskController.LOG.info("生成模板文件路径:{}", htemplate.getMustacheText());
        File analysis = GetTaskDirectory.getAnalysisPath(entity, IterableExtensions.join(Collections.<String>unmodifiableList(CollectionLiterals.<String>newArrayList(tempName, "js")), "."));
        FileUtils.writeStringToFile(analysis, StringUtils.<String>defaultIfEmpty(htemplate.getAnalysisShellText(), ""), "UTF-8", false);
        htemplate.setAnalysisShellText(analysis.getAbsolutePath());
        entity.setHtmlTemplate(new Gson().toJson(htemplate));
      }
      int _shellType = entity.getShellType();
      boolean _matched = false;
      int _ordinal = TaskEntity.ShellType.Shell.ordinal();
      if (Objects.equal(_shellType, _ordinal)) {
        _matched=true;
        String _replace = UUID.randomUUID().toString().replace("-", "");
        String _plus = (_replace + ".sh");
        File file = GetTaskDirectory.getScriptPath(entity, _plus);
        TaskController.LOG.info("创建shell脚本:{}", file.getAbsolutePath());
        FileUtils.writeStringToFile(file, StringUtils.<String>defaultIfEmpty(entity.getShellPath(), ""), "UTF-8", false);
        entity.setShellPath(file.getAbsolutePath().trim());
      }
      if (!_matched) {
        int _ordinal_1 = TaskEntity.ShellType.BatchFile.ordinal();
        if (Objects.equal(_shellType, _ordinal_1)) {
          _matched=true;
          String _replace_1 = UUID.randomUUID().toString().replace("-", "");
          String _plus_1 = (_replace_1 + ".bat");
          File file_1 = GetTaskDirectory.getScriptPath(entity, _plus_1);
          TaskController.LOG.info("创建BAT脚本:{}", file_1.getAbsolutePath());
          FileUtils.writeStringToFile(file_1, StringUtils.<String>defaultIfEmpty(entity.getShellPath(), ""), "UTF-8", false);
          entity.setShellPath(file_1.getAbsolutePath().trim());
        }
      }
      if (!_matched) {
        int _ordinal_2 = TaskEntity.ShellType.Groovy.ordinal();
        if (Objects.equal(_shellType, _ordinal_2)) {
          _matched=true;
          String _replace_2 = UUID.randomUUID().toString().replace("-", "");
          String _plus_2 = (_replace_2 + ".groovy");
          File file_2 = GetTaskDirectory.getScriptPath(entity, _plus_2);
          TaskController.LOG.info("创建Groovy脚本:{}", file_2.getAbsolutePath());
          FileUtils.writeStringToFile(file_2, StringUtils.<String>defaultIfEmpty(entity.getShellPath(), ""), "UTF-8", false);
          entity.setShellPath(file_2.getAbsolutePath().trim());
        }
      }
      if (!_matched) {
        int _ordinal_3 = TaskEntity.ShellType.JavaScript.ordinal();
        if (Objects.equal(_shellType, _ordinal_3)) {
          _matched=true;
          String _replace_3 = UUID.randomUUID().toString().replace("-", "");
          String _plus_3 = (_replace_3 + ".js");
          File file_3 = GetTaskDirectory.getScriptPath(entity, _plus_3);
          TaskController.LOG.info("创建Javascript脚本:{}", file_3.getAbsolutePath());
          FileUtils.writeStringToFile(file_3, StringUtils.<String>defaultIfEmpty(entity.getShellPath(), ""), "UTF-8", false);
          entity.setShellPath(file_3.getAbsolutePath().trim());
        }
      }
      if (!_matched) {
        {
          TaskController.LOG.info("脚本类型错误:");
          return Response.status(500).build();
        }
      }
      TaskController.LOG.info("保存任务:{}", entity.toString());
      entity.setTaskVersion(Integer.valueOf(0));
      int _saveTask = this.taskService.saveTask(entity);
      boolean _greaterThan = (_saveTask > 0);
      if (_greaterThan) {
        TaskController.LOG.info("保存任务成功");
        Integer _taskVersion = entity.getTaskVersion();
        this.jgit.addAndCommit(entity.getTaskDirectory(), IterableExtensions.join(Collections.<Object>unmodifiableList(CollectionLiterals.<Object>newArrayList("v", _taskVersion))));
        return Response.status(201).build();
      }
      return null;
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  @Override
  public Response runTask(final String uuid) {
    TaskEntity entity = this.taskService.getOneTask(uuid);
    TaskController.LOG.info("运行任务:{}", entity.getTaskName());
    TaskExtendColumnEntity _xifexpression = null;
    String _extendColumn = entity.getExtendColumn();
    boolean _tripleNotEquals = (_extendColumn != null);
    if (_tripleNotEquals) {
      _xifexpression = new Gson().<TaskExtendColumnEntity>fromJson(entity.getExtendColumn(), TaskExtendColumnEntity.class);
    } else {
      _xifexpression = null;
    }
    TaskExtendColumnEntity extendColumn = _xifexpression;
    String _taskName = entity.getTaskName();
    Pair<String, String> _mappedTo = Pair.<String, String>of("taskName", _taskName);
    String _xifexpression_1 = null;
    if ((extendColumn != null)) {
      _xifexpression_1 = extendColumn.getProjectName();
    } else {
      _xifexpression_1 = "";
    }
    Pair<String, String> _mappedTo_1 = Pair.<String, String>of("projectName", _xifexpression_1);
    String _property = System.getProperty("user.dir");
    Pair<String, String> _mappedTo_2 = Pair.<String, String>of("userDir", _property);
    String _taskDirectory = entity.getTaskDirectory();
    Pair<String, String> _mappedTo_3 = Pair.<String, String>of("workspace", _taskDirectory);
    Map<String, String> env = Collections.<String, String>unmodifiableMap(CollectionLiterals.<String, String>newHashMap(_mappedTo, _mappedTo_1, _mappedTo_2, _mappedTo_3));
    String _shellPath = entity.getShellPath();
    File shellFile = new File(_shellPath);
    boolean _exists = shellFile.exists();
    if (_exists) {
      int _shellType = entity.getShellType();
      boolean _matched = false;
      int _ordinal = TaskEntity.ShellType.Shell.ordinal();
      if (Objects.equal(_shellType, _ordinal)) {
        _matched=true;
        final ArrayList<String> results = new ArrayList<String>();
        TaskController.LOG.info("运行任务脚本:{}", shellFile.getAbsolutePath());
        final ShellExecute.ShellCallback _function = (String it) -> {
          results.add(it);
        };
        ShellExecute.start(entity.getTaskName(), _function, env, "bash", shellFile.getAbsolutePath());
        this.taskResultProcessing(results, extendColumn, entity);
      }
      if (!_matched) {
        int _ordinal_1 = TaskEntity.ShellType.BatchFile.ordinal();
        if (Objects.equal(_shellType, _ordinal_1)) {
          _matched=true;
          final ArrayList<String> results_1 = new ArrayList<String>();
          TaskController.LOG.info("运行任务脚本:{}", shellFile.getAbsolutePath());
          final ShellExecute.ShellCallback _function_1 = (String it) -> {
            results_1.add(it);
          };
          ShellExecute.start(entity.getTaskName(), _function_1, env, shellFile.getAbsolutePath());
          this.taskResultProcessing(results_1, extendColumn, entity);
        }
      }
      if (!_matched) {
        int _ordinal_2 = TaskEntity.ShellType.Groovy.ordinal();
        if (Objects.equal(_shellType, _ordinal_2)) {
          _matched=true;
        }
      }
      if (!_matched) {
        int _ordinal_3 = TaskEntity.ShellType.JavaScript.ordinal();
        if (Objects.equal(_shellType, _ordinal_3)) {
          _matched=true;
        }
      }
      if (!_matched) {
      }
    }
    return Response.ok().build();
  }
  
  private void taskResultProcessing(final ArrayList<String> results, final TaskExtendColumnEntity extendColumn, final TaskEntity entity) {
    try {
      TaskController.LOG.info("需要处理运行结果行数:{}", Integer.valueOf(((Object[])Conversions.unwrapArray(results, Object.class)).length));
      HashMap<String, String> jsEnv = new HashMap<String, String>();
      HtmlTemplateEntity htemplate = new Gson().<HtmlTemplateEntity>fromJson(entity.getHtmlTemplate(), HtmlTemplateEntity.class);
      String _analysisShellText = htemplate.getAnalysisShellText();
      File _file = new File(_analysisShellText);
      String analysisFile = FileUtils.readFileToString(_file, "UTF-8");
      boolean _isNotBlank = StringUtils.isNotBlank(analysisFile);
      if (_isNotBlank) {
        try (JsRhinoScript js = new JsRhinoScript()) {
          js.set("env", jsEnv);
          js.set("otherEnv", extendColumn);
          StringBuffer buffer = new StringBuffer("function taomubox_template_entry(results){");
          String _taskName = entity.getTaskName();
          String _plus = ("env.put(\'taskName\',\'" + _taskName);
          String _plus_1 = (_plus + "\');");
          buffer.append(_plus_1);
          buffer.append(analysisFile);
          buffer.append("return env;");
          buffer.append("}");
          js.loaderScript(buffer.toString());
          Object _invoke = js.invoke("taomubox_template_entry", results);
          Map<String, String> resMap = ((Map<String, String>) _invoke);
          TaskController.LOG.info("解析器返回结果:{}", resMap.toString());
          this.saveRunResult(entity, resMap);
        }
      } else {
        TaskController.LOG.info("解析器脚本为空直接保存结果");
        String _string = DateTime.now().toString("yyyMMddHHmmss");
        String resultFileName = (_string + ".txt");
        File resFile = GetTaskDirectory.getResultPath(entity, resultFileName);
        TaskController.LOG.info("将结果保存到:{}", resFile.getAbsolutePath());
        entity.setResultPath(resFile.getAbsolutePath().trim());
        FileUtils.writeLines(resFile, "UTF-8", results);
        Integer _taskVersion = entity.getTaskVersion();
        int _plus = ((_taskVersion).intValue() + 1);
        entity.setTaskVersion(Integer.valueOf(_plus));
        HashMap<JGitTools.StatusType, Set<String>> gitStatus = this.jgit.status(entity.getTaskDirectory(), JGitTools.StatusType.Modified);
        int _size = gitStatus.get(JGitTools.StatusType.Modified).size();
        boolean _greaterThan = (_size > 0);
        if (_greaterThan) {
          Integer _taskVersion_1 = entity.getTaskVersion();
          this.jgit.addAndCommit(entity.getTaskDirectory(), IterableExtensions.join(Collections.<Object>unmodifiableList(CollectionLiterals.<Object>newArrayList("v", _taskVersion_1))));
        }
        this.taskService.update(entity);
      }
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  protected void saveRunResult(final TaskEntity entity, final Map<String, String> results) {
    try {
      TaskController.LOG.info("保存运行结果");
      StringWriter writer = new StringWriter();
      boolean _isNullOrEmpty = StringExtensions.isNullOrEmpty(entity.getHtmlTemplate());
      boolean _not = (!_isNullOrEmpty);
      if (_not) {
        TaskController.LOG.info("运行模版引擎生成格式化数据");
        DefaultMustacheFactory mf = new DefaultMustacheFactory();
        HtmlTemplateEntity htemplate = new Gson().<HtmlTemplateEntity>fromJson(entity.getHtmlTemplate(), HtmlTemplateEntity.class);
        boolean _endsWith = htemplate.getMustacheText().endsWith(".mustache");
        if (_endsWith) {
          Mustache mustache = mf.compile(htemplate.getMustacheText());
          mustache.execute(writer, results).flush();
        }
      } else {
        writer.append(IterableExtensions.join(results.values(), ""));
        writer.flush();
      }
      String _string = DateTime.now().toString("yyyMMddHHmmss");
      String resultFileName = (_string + ".txt");
      File resFile = GetTaskDirectory.getResultPath(entity, resultFileName);
      TaskController.LOG.info("将结果保存到:{}", resFile.getAbsolutePath());
      entity.setResultPath(resFile.getAbsolutePath().trim());
      String _string_1 = writer.toString();
      FileUtils.writeLines(resFile, "UTF-8", Collections.<Object>unmodifiableList(CollectionLiterals.<Object>newArrayList(_string_1)));
      Integer _taskVersion = entity.getTaskVersion();
      int _plus = ((_taskVersion).intValue() + 1);
      entity.setTaskVersion(Integer.valueOf(_plus));
      HashMap<JGitTools.StatusType, Set<String>> gitStatus = this.jgit.status(entity.getTaskDirectory(), JGitTools.StatusType.Modified);
      int _size = gitStatus.get(JGitTools.StatusType.Modified).size();
      boolean _greaterThan = (_size > 0);
      if (_greaterThan) {
        Integer _taskVersion_1 = entity.getTaskVersion();
        this.jgit.addAndCommit(entity.getTaskDirectory(), IterableExtensions.join(Collections.<Object>unmodifiableList(CollectionLiterals.<Object>newArrayList("v", _taskVersion_1))));
      }
      this.taskService.update(entity);
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  @Override
  public Response exitTask(final String uuid) {
    TaskEntity e = this.taskService.getOneTask(uuid);
    TaskController.LOG.info("结束任务:{}", e.getTaskName());
    return Response.ok().build();
  }
  
  @Override
  public Response getTaskType() {
    TaskController.LOG.info("获取任务类型");
    TaskEntity.TaskType[] _values = TaskEntity.TaskType.values();
    Pair<String, TaskEntity.TaskType[]> _mappedTo = Pair.<String, TaskEntity.TaskType[]>of("task_type", _values);
    final Function<TaskEntity.TaskType, Integer> _function = (TaskEntity.TaskType it) -> {
      return Integer.valueOf(it.ordinal());
    };
    List<Integer> _collect = ((List<TaskEntity.TaskType>)Conversions.doWrapArray(TaskEntity.TaskType.values())).stream().<Integer>map(_function).collect(Collectors.<Integer>toList());
    Pair<String, List<Integer>> _mappedTo_1 = Pair.<String, List<Integer>>of("task_type_ordinal", _collect);
    TaskEntity.ShellType[] _values_1 = TaskEntity.ShellType.values();
    Pair<String, TaskEntity.ShellType[]> _mappedTo_2 = Pair.<String, TaskEntity.ShellType[]>of("shell_type", _values_1);
    final Function<TaskEntity.ShellType, Integer> _function_1 = (TaskEntity.ShellType it) -> {
      return Integer.valueOf(it.ordinal());
    };
    List<Integer> _collect_1 = ((List<TaskEntity.ShellType>)Conversions.doWrapArray(TaskEntity.ShellType.values())).stream().<Integer>map(_function_1).collect(Collectors.<Integer>toList());
    Pair<String, List<Integer>> _mappedTo_3 = Pair.<String, List<Integer>>of("shell_type_ordinal", _collect_1);
    return Response.ok(
      Collections.<Pair<String, ?>>unmodifiableList(CollectionLiterals.<Pair<String, ?>>newArrayList(_mappedTo, _mappedTo_1, _mappedTo_2, _mappedTo_3))).build();
  }
  
  @Override
  public Response getTaskResult(final String uuid) {
    TaskEntity entity = this.taskService.getOneTask(uuid);
    TaskController.LOG.info("获取任务{}结果", entity.getTaskName());
    TaskController.LOG.info("格式化markdown数据");
    String result = MarkdownUtils.getHtml(entity.getResultPath());
    return Response.ok(result).build();
  }
  
  @Override
  public Response getTaskList() {
    List<TaskEntity> entitys = this.taskService.getTaskAll();
    final List<TaskEntity> _converted_entitys = (List<TaskEntity>)entitys;
    TaskController.LOG.info("获取任务列表:{}", Integer.valueOf(((Object[])Conversions.unwrapArray(_converted_entitys, Object.class)).length));
    return Response.ok(entitys).build();
  }
}
