/**
 * Copyright (c) 2023 murenchao
 * taomu is licensed under Mulan PubL v2.
 * You can use this software according to the terms and conditions of the Mulan PubL v2.
 * You may obtain a copy of Mulan PubL v2 at:
 *       http://license.coscl.org.cn/MulanPubL-2.0
 * THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
 * EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
 * MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
 * See the Mulan PubL v2 for more details.
 */
package cool.taomu.box.job.server.controller.impl;

import com.github.mustachejava.DefaultMustacheFactory;
import com.github.mustachejava.Mustache;
import com.google.common.base.Objects;
import com.google.gson.Gson;
import cool.taomu.box.git.JGitTools;
import cool.taomu.box.job.server.controller.abs.AbsPreTaskAndPostTask;
import cool.taomu.box.job.server.controller.entity.FileWriteEntity;
import cool.taomu.box.job.server.controller.entity.HtmlTemplateEntity;
import cool.taomu.box.job.server.controller.entity.ResponseEntity;
import cool.taomu.box.job.server.controller.entity.TaskExtendColumnEntity;
import cool.taomu.box.job.server.controller.entity.TaskResultEntity;
import cool.taomu.box.job.server.controller.inter.ITaskController;
import cool.taomu.box.job.server.entity.TaskEntity;
import cool.taomu.box.job.server.service.TaskService;
import cool.taomu.box.job.server.utils.ChineseToPinyin;
import cool.taomu.box.job.server.utils.GetTaskDirectory;
import cool.taomu.box.job.server.utils.GitFileHistory;
import cool.taomu.box.job.server.utils.MarkdownUtils;
import cool.taomu.box.job.server.utils.ShellExecute;
import cool.taomu.box.script.rhino.JsRhinoScript;
import cool.taomu.box.utils.FileTools;
import cool.taomu.box.utils.inter.ICallback2;
import java.io.File;
import java.io.FileInputStream;
import java.io.InputStreamReader;
import java.io.StringWriter;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import javax.inject.Inject;
import javax.ws.rs.Path;
import javax.ws.rs.core.Response;
import org.apache.commons.io.FileUtils;
import org.apache.commons.lang3.StringUtils;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.InputOutput;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.Pair;
import org.eclipse.xtext.xbase.lib.StringExtensions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Path("job")
@SuppressWarnings("all")
public class TaskController implements ITaskController<TaskEntity> {
  private static final Logger LOG = LoggerFactory.getLogger(TaskController.class);
  
  @Inject
  private TaskService taskService;
  
  @Inject
  private JGitTools jgit;
  
  @Inject
  private AbsPreTaskAndPostTask iTask;
  
  @Override
  public Response getTask(final String uuid) {
    try {
      TaskController.LOG.info("获取 {} ID的任务", uuid);
      final TaskEntity entity = this.taskService.getOneTask(uuid);
      HtmlTemplateEntity htemplate = new Gson().<HtmlTemplateEntity>fromJson(entity.getHtmlTemplate(), HtmlTemplateEntity.class);
      final Consumer<HtmlTemplateEntity.DataEntity> _function = (HtmlTemplateEntity.DataEntity it) -> {
        try {
          TaskController.LOG.info("mustache path:{}", it.getMustachePath());
          TaskController.LOG.info("mustache path:{}", it.getAnalysisShellPath());
          String _mustachePath = it.getMustachePath();
          File tempFile = new File(_mustachePath);
          it.setMustacheText(FileUtils.readFileToString(tempFile, "UTF-8"));
          String _analysisShellPath = it.getAnalysisShellPath();
          File _file = new File(_analysisShellPath);
          it.setAnalysisShellText(FileUtils.readFileToString(_file, "UTF-8"));
        } catch (Throwable _e) {
          throw Exceptions.sneakyThrow(_e);
        }
      };
      htemplate.getData().forEach(_function);
      entity.setHtmlTemplate(new Gson().toJson(htemplate));
      String shellPath = entity.getShellPath();
      TaskController.LOG.info("shell path:{}", shellPath);
      File _file = new File(shellPath);
      entity.setShellText(FileUtils.readFileToString(_file, "UTF-8"));
      return Response.ok(entity).build();
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  @Override
  public Response putTask(final String uuid, final TaskEntity entity) {
    TaskController.LOG.info("修改 {} ID的任务", uuid);
    final TaskEntity old = this.taskService.getOneTask(uuid);
    Integer _taskVersion = old.getTaskVersion();
    int _plus = ((_taskVersion).intValue() + 1);
    old.setTaskVersion(Integer.valueOf(_plus));
    final ArrayList<FileWriteEntity> fileWriteList = new ArrayList<FileWriteEntity>();
    String _shellPath = old.getShellPath();
    String _shellText = entity.getShellText();
    FileWriteEntity _fileWriteEntity = new FileWriteEntity(_shellPath, _shellText);
    fileWriteList.add(_fileWriteEntity);
    HtmlTemplateEntity htemplate = new Gson().<HtmlTemplateEntity>fromJson(old.getHtmlTemplate(), HtmlTemplateEntity.class);
    this.jgit.setRepository(old.getTaskDirectory());
    boolean _isEmpty = htemplate.getData().isEmpty();
    boolean _not = (!_isEmpty);
    if (_not) {
      final Consumer<HtmlTemplateEntity.DataEntity> _function = (HtmlTemplateEntity.DataEntity it) -> {
        final String mustachePath = it.getMustachePath();
        final String analysisShellPath = it.getAnalysisShellPath();
        FileWriteEntity _fileWriteEntity_1 = new FileWriteEntity(mustachePath, "", FileWriteEntity.OperationType.RM);
        fileWriteList.add(_fileWriteEntity_1);
        FileWriteEntity _fileWriteEntity_2 = new FileWriteEntity(analysisShellPath, "", FileWriteEntity.OperationType.RM);
        fileWriteList.add(_fileWriteEntity_2);
      };
      htemplate.getData().forEach(_function);
    }
    final HtmlTemplateEntity inputHtemplate = new Gson().<HtmlTemplateEntity>fromJson(entity.getHtmlTemplate(), HtmlTemplateEntity.class);
    boolean _isEmpty_1 = inputHtemplate.getData().isEmpty();
    boolean _not_1 = (!_isEmpty_1);
    if (_not_1) {
      final String tempName = UUID.randomUUID().toString().replace("-", "");
      TaskController.LOG.info("生成模板文件:{}", tempName);
      TaskController.LOG.info("模板内容:{}", entity.getHtmlTemplate());
      final Consumer<HtmlTemplateEntity.DataEntity> _function_1 = (HtmlTemplateEntity.DataEntity nit) -> {
        nit.setPname(ChineseToPinyin.toPinyin(nit.getName()));
        String _pname = nit.getPname();
        final File mustachePath = GetTaskDirectory.getTemplatePath(old, IterableExtensions.join(Collections.<String>unmodifiableList(CollectionLiterals.<String>newArrayList(_pname, tempName, "mustache")), "."));
        nit.setMustachePath(mustachePath.getAbsolutePath().trim());
        String _pname_1 = nit.getPname();
        final File analysis = GetTaskDirectory.getAnalysisPath(old, IterableExtensions.join(Collections.<String>unmodifiableList(CollectionLiterals.<String>newArrayList(_pname_1, tempName, "js")), "."));
        nit.setAnalysisShellPath(analysis.getAbsolutePath().trim());
        String _absolutePath = mustachePath.getAbsolutePath();
        String _mustacheText = nit.getMustacheText();
        FileWriteEntity _fileWriteEntity_1 = new FileWriteEntity(_absolutePath, _mustacheText);
        String _absolutePath_1 = analysis.getAbsolutePath();
        String _analysisShellText = nit.getAnalysisShellText();
        FileWriteEntity _fileWriteEntity_2 = new FileWriteEntity(_absolutePath_1, _analysisShellText);
        fileWriteList.addAll(
          Collections.<FileWriteEntity>unmodifiableList(CollectionLiterals.<FileWriteEntity>newArrayList(_fileWriteEntity_1, _fileWriteEntity_2)));
      };
      inputHtemplate.getData().forEach(_function_1);
      final ICallback2<List<FileWriteEntity>> _function_2 = () -> {
        return fileWriteList;
      };
      this.taskService.FileWrite(old.getTaskDirectory(), (old.getTaskVersion()).intValue(), _function_2);
    }
    old.setHtmlTemplate(new Gson().toJson(inputHtemplate));
    old.setExtendColumn(StringUtils.<String>defaultIfEmpty(entity.getExtendColumn(), ""));
    this.taskService.update(old);
    return Response.status(201).build();
  }
  
  @Override
  public Response deleteTask(final String uuid) {
    TaskController.LOG.info("删除任务:{}", uuid);
    TaskEntity oldTask = this.taskService.getOneTask(uuid);
    String _trim = oldTask.getTaskDirectory().trim();
    File dir = new File(_trim);
    TaskController.LOG.info("要删除的目录为: {}", dir.getAbsolutePath());
    boolean _isNullOrEmpty = StringExtensions.isNullOrEmpty(oldTask.getTaskDirectory());
    boolean _not = (!_isNullOrEmpty);
    if (_not) {
      try {
        boolean _deleteQuietly = FileUtils.deleteQuietly(dir);
        if (_deleteQuietly) {
          TaskController.LOG.info("删除任务 {} 目录成功", dir.getAbsolutePath());
          int _deleteOneTask = this.taskService.deleteOneTask(oldTask.getTaskId());
          boolean _greaterThan = (_deleteOneTask > 0);
          if (_greaterThan) {
            TaskController.LOG.info("删除任务 {} 成功", uuid);
            return Response.status(200).build();
          } else {
            TaskController.LOG.info("删除任务 {} 失败", uuid);
            return Response.status(402).build();
          }
        } else {
          TaskController.LOG.info("删除任务 {} 目录失败", dir.getAbsolutePath());
          return Response.status(402).build();
        }
      } catch (final Throwable _t) {
        if (_t instanceof Exception) {
          final Exception ex = (Exception)_t;
          ex.printStackTrace();
          return Response.status(500).build();
        } else {
          throw Exceptions.sneakyThrow(_t);
        }
      }
    }
    return null;
  }
  
  @Override
  public Response postTask(final TaskEntity entity) {
    TaskEntity sentity = this.taskService.getOneTaskByTaskName(entity.getTaskName());
    final ArrayList<FileWriteEntity> fileWriteList = new ArrayList<FileWriteEntity>();
    if ((sentity != null)) {
      return Response.status(406).build();
    }
    entity.setTaskName(entity.getTaskName().trim());
    entity.setTaskVersion(Integer.valueOf(0));
    String taskNamePinyin = ChineseToPinyin.toPinyin(entity.getTaskName());
    TaskController.LOG.info("任务名:{} 转为拼音:{}", entity.getTaskName(), taskNamePinyin);
    String _property = System.getProperty("user.dir");
    String _join = IterableExtensions.join(Collections.<String>unmodifiableList(CollectionLiterals.<String>newArrayList(_property, taskNamePinyin)), File.separator);
    File taskNameDir = new File(_join);
    entity.setTaskDirectory(taskNameDir.getAbsolutePath().trim());
    this.jgit.setRepository(entity.getTaskDirectory().trim());
    boolean _exists = taskNameDir.exists();
    boolean _not = (!_exists);
    if (_not) {
      boolean _mkdirs = taskNameDir.mkdirs();
      if (_mkdirs) {
        TaskController.LOG.info("创建目录:{}", taskNameDir.getAbsolutePath());
        this.jgit.init();
      }
    }
    TaskController.LOG.info("任务目录:{}", taskNameDir);
    entity.setTaskId(UUID.randomUUID().toString());
    entity.setClientId(UUID.randomUUID().toString());
    GetTaskDirectory.getScriptPath(entity, null);
    GetTaskDirectory.getResultPath(entity, null);
    File tempdir = GetTaskDirectory.getTemplatePath(entity, null);
    boolean _exists_1 = tempdir.exists();
    if (_exists_1) {
      final String tempName = UUID.randomUUID().toString().replace("-", "");
      TaskController.LOG.info("生成模板文件:{}", tempName);
      TaskController.LOG.info("模板内容:{}", entity.getHtmlTemplate());
      HtmlTemplateEntity htemplate = new Gson().<HtmlTemplateEntity>fromJson(entity.getHtmlTemplate(), HtmlTemplateEntity.class);
      InputOutput.<HtmlTemplateEntity>println(htemplate);
      final Consumer<HtmlTemplateEntity.DataEntity> _function = (HtmlTemplateEntity.DataEntity nit) -> {
        nit.setPname(ChineseToPinyin.toPinyin(nit.getName()));
        String _pname = nit.getPname();
        nit.setMustachePath(GetTaskDirectory.getTemplatePath(entity, 
          IterableExtensions.join(Collections.<String>unmodifiableList(CollectionLiterals.<String>newArrayList(_pname, tempName, "mustache")), ".")).getAbsolutePath());
        String _pname_1 = nit.getPname();
        nit.setAnalysisShellPath(GetTaskDirectory.getAnalysisPath(entity, 
          IterableExtensions.join(Collections.<String>unmodifiableList(CollectionLiterals.<String>newArrayList(_pname_1, tempName, "js")), ".")).getAbsolutePath());
        InputOutput.<String>println(nit.getMustacheText());
        InputOutput.<String>println(nit.getAnalysisShellText());
        String _mustachePath = nit.getMustachePath();
        String _mustacheText = nit.getMustacheText();
        FileWriteEntity _fileWriteEntity = new FileWriteEntity(_mustachePath, _mustacheText);
        fileWriteList.add(_fileWriteEntity);
        String _analysisShellPath = nit.getAnalysisShellPath();
        String _analysisShellText = nit.getAnalysisShellText();
        FileWriteEntity _fileWriteEntity_1 = new FileWriteEntity(_analysisShellPath, _analysisShellText);
        fileWriteList.add(_fileWriteEntity_1);
        InputOutput.<String>println(fileWriteList.toString());
      };
      htemplate.getData().forEach(_function);
      entity.setHtmlTemplate(new Gson().toJson(htemplate));
    }
    int _shellType = entity.getShellType();
    boolean _matched = false;
    int _ordinal = TaskEntity.ShellType.Shell.ordinal();
    if (Objects.equal(_shellType, _ordinal)) {
      _matched=true;
      String _replace = UUID.randomUUID().toString().replace("-", "");
      String _plus = (_replace + ".sh");
      File file = GetTaskDirectory.getScriptPath(entity, _plus);
      TaskController.LOG.info("创建shell脚本:{}", file.getAbsolutePath());
      entity.setShellPath(file.getAbsolutePath().trim());
      String _shellPath = entity.getShellPath();
      String _shellText = entity.getShellText();
      FileWriteEntity _fileWriteEntity = new FileWriteEntity(_shellPath, _shellText);
      fileWriteList.add(_fileWriteEntity);
    }
    if (!_matched) {
      int _ordinal_1 = TaskEntity.ShellType.BatchFile.ordinal();
      if (Objects.equal(_shellType, _ordinal_1)) {
        _matched=true;
        String _replace_1 = UUID.randomUUID().toString().replace("-", "");
        String _plus_1 = (_replace_1 + ".bat");
        File file_1 = GetTaskDirectory.getScriptPath(entity, _plus_1);
        TaskController.LOG.info("创建BAT脚本:{}", file_1.getAbsolutePath());
        entity.setShellPath(file_1.getAbsolutePath().trim());
        String _shellPath_1 = entity.getShellPath();
        String _shellText_1 = entity.getShellText();
        FileWriteEntity _fileWriteEntity_1 = new FileWriteEntity(_shellPath_1, _shellText_1);
        fileWriteList.add(_fileWriteEntity_1);
      }
    }
    if (!_matched) {
      int _ordinal_2 = TaskEntity.ShellType.Groovy.ordinal();
      if (Objects.equal(_shellType, _ordinal_2)) {
        _matched=true;
        String _replace_2 = UUID.randomUUID().toString().replace("-", "");
        String _plus_2 = (_replace_2 + ".groovy");
        File file_2 = GetTaskDirectory.getScriptPath(entity, _plus_2);
        TaskController.LOG.info("创建Groovy脚本:{}", file_2.getAbsolutePath());
        entity.setShellPath(file_2.getAbsolutePath().trim());
        String _shellPath_2 = entity.getShellPath();
        String _shellText_2 = entity.getShellText();
        FileWriteEntity _fileWriteEntity_2 = new FileWriteEntity(_shellPath_2, _shellText_2);
        fileWriteList.add(_fileWriteEntity_2);
      }
    }
    if (!_matched) {
      int _ordinal_3 = TaskEntity.ShellType.JavaScript.ordinal();
      if (Objects.equal(_shellType, _ordinal_3)) {
        _matched=true;
        String _replace_3 = UUID.randomUUID().toString().replace("-", "");
        String _plus_3 = (_replace_3 + ".js");
        File file_3 = GetTaskDirectory.getScriptPath(entity, _plus_3);
        TaskController.LOG.info("创建Javascript脚本:{}", file_3.getAbsolutePath());
        entity.setShellPath(file_3.getAbsolutePath().trim());
        String _shellPath_3 = entity.getShellPath();
        String _shellText_3 = entity.getShellText();
        FileWriteEntity _fileWriteEntity_3 = new FileWriteEntity(_shellPath_3, _shellText_3);
        fileWriteList.add(_fileWriteEntity_3);
      }
    }
    if (!_matched) {
      {
        TaskController.LOG.info("脚本类型错误:");
        return Response.status(500).build();
      }
    }
    Integer _taskVersion = entity.getTaskVersion();
    int _plus_4 = ((_taskVersion).intValue() + 1);
    entity.setTaskVersion(Integer.valueOf(_plus_4));
    final ICallback2<List<FileWriteEntity>> _function_1 = () -> {
      return fileWriteList;
    };
    this.taskService.FileWrite(entity.getTaskDirectory(), (entity.getTaskVersion()).intValue(), _function_1);
    TaskController.LOG.info("保存任务:{}", entity.toString());
    int _saveTask = this.taskService.saveTask(entity);
    boolean _greaterThan = (_saveTask > 0);
    if (_greaterThan) {
      TaskController.LOG.info("保存任务成功");
      return Response.status(201).build();
    }
    return null;
  }
  
  @Override
  public Response runTask(final String uuid) {
    TaskEntity entity = this.taskService.getOneTask(uuid);
    final ArrayList<FileWriteEntity> fileWriteList = new ArrayList<FileWriteEntity>();
    if ((this.iTask != null)) {
      this.iTask.preTask(entity);
    }
    TaskController.LOG.info("运行任务:{}", entity.getTaskName());
    TaskExtendColumnEntity _xifexpression = null;
    String _extendColumn = entity.getExtendColumn();
    boolean _tripleNotEquals = (_extendColumn != null);
    if (_tripleNotEquals) {
      _xifexpression = new Gson().<TaskExtendColumnEntity>fromJson(entity.getExtendColumn(), TaskExtendColumnEntity.class);
    } else {
      _xifexpression = null;
    }
    TaskExtendColumnEntity extendColumn = _xifexpression;
    String _taskName = entity.getTaskName();
    Pair<String, String> _mappedTo = Pair.<String, String>of("taskName", _taskName);
    String _xifexpression_1 = null;
    if ((extendColumn != null)) {
      _xifexpression_1 = extendColumn.getProjectName();
    } else {
      _xifexpression_1 = "";
    }
    Pair<String, String> _mappedTo_1 = Pair.<String, String>of("projectName", _xifexpression_1);
    String _property = System.getProperty("user.dir");
    Pair<String, String> _mappedTo_2 = Pair.<String, String>of("userDir", _property);
    String _taskDirectory = entity.getTaskDirectory();
    Pair<String, String> _mappedTo_3 = Pair.<String, String>of("workspace", _taskDirectory);
    Map<String, String> env = Collections.<String, String>unmodifiableMap(CollectionLiterals.<String, String>newHashMap(_mappedTo, _mappedTo_1, _mappedTo_2, _mappedTo_3));
    String _shellPath = entity.getShellPath();
    File shellFile = new File(_shellPath);
    boolean _exists = shellFile.exists();
    if (_exists) {
      int _shellType = entity.getShellType();
      boolean _matched = false;
      int _ordinal = TaskEntity.ShellType.Shell.ordinal();
      if (Objects.equal(_shellType, _ordinal)) {
        _matched=true;
        final ArrayList<String> results = new ArrayList<String>();
        TaskController.LOG.info("运行任务脚本:{}", shellFile.getAbsolutePath());
        entity.setStatus(TaskEntity.TaskStatus.IN_PROGRESS.ordinal());
        this.taskService.update(entity);
        final ShellExecute.ShellCallback _function = (String it) -> {
          results.add(it);
        };
        ShellExecute.start(entity.getTaskName(), _function, env, "bash", shellFile.getAbsolutePath());
        File file = GetTaskDirectory.getScriptResultPath(entity, IterableExtensions.join(Collections.<String>unmodifiableList(CollectionLiterals.<String>newArrayList("execute_result", "txt")), "."));
        String _absolutePath = file.getAbsolutePath();
        String _join = IterableExtensions.join(results);
        FileWriteEntity fwentity = new FileWriteEntity(_absolutePath, _join);
        fileWriteList.add(fwentity);
        this.taskResultProcessing(results, extendColumn, entity, fileWriteList);
      }
      if (!_matched) {
        int _ordinal_1 = TaskEntity.ShellType.BatchFile.ordinal();
        if (Objects.equal(_shellType, _ordinal_1)) {
          _matched=true;
          final ArrayList<String> results_1 = new ArrayList<String>();
          TaskController.LOG.info("运行任务脚本:{}", shellFile.getAbsolutePath());
          entity.setStatus(TaskEntity.TaskStatus.IN_PROGRESS.ordinal());
          this.taskService.update(entity);
          final ShellExecute.ShellCallback _function_1 = (String it) -> {
            results_1.add(it);
          };
          ShellExecute.start(entity.getTaskName(), _function_1, env, shellFile.getAbsolutePath());
          File file_1 = GetTaskDirectory.getScriptResultPath(entity, IterableExtensions.join(Collections.<String>unmodifiableList(CollectionLiterals.<String>newArrayList("execute_result", "txt")), "."));
          String _absolutePath_1 = file_1.getAbsolutePath();
          String _join_1 = IterableExtensions.join(results_1);
          FileWriteEntity fwentity_1 = new FileWriteEntity(_absolutePath_1, _join_1);
          fileWriteList.add(fwentity_1);
          this.taskResultProcessing(results_1, extendColumn, entity, fileWriteList);
        }
      }
      if (!_matched) {
        int _ordinal_2 = TaskEntity.ShellType.Groovy.ordinal();
        if (Objects.equal(_shellType, _ordinal_2)) {
          _matched=true;
        }
      }
      if (!_matched) {
        int _ordinal_3 = TaskEntity.ShellType.JavaScript.ordinal();
        if (Objects.equal(_shellType, _ordinal_3)) {
          _matched=true;
        }
      }
      if (!_matched) {
      }
    }
    if ((this.iTask != null)) {
      this.iTask.postTask(entity);
    }
    final ICallback2<List<FileWriteEntity>> _function_2 = () -> {
      return fileWriteList;
    };
    this.taskService.FileWrite(entity.getTaskDirectory(), (entity.getTaskVersion()).intValue(), _function_2);
    return Response.ok().build();
  }
  
  private void taskResultProcessing(final ArrayList<String> results, final TaskExtendColumnEntity extendColumn, final TaskEntity entity, final ArrayList<FileWriteEntity> fileWriteList) {
    TaskController.LOG.info("需要处理运行结果行数:{}", Integer.valueOf(((Object[])Conversions.unwrapArray(results, Object.class)).length));
    final HashMap<String, Object> jsEnv = new HashMap<String, Object>();
    jsEnv.put("taskName", entity.getTaskName());
    jsEnv.put("taskId", entity.getTaskId());
    jsEnv.put("executeResultFilePath", 
      GetTaskDirectory.getScriptResultPath(entity, IterableExtensions.join(Collections.<String>unmodifiableList(CollectionLiterals.<String>newArrayList("execute_result", "txt")), ".")).getAbsolutePath());
    HtmlTemplateEntity htemplate = new Gson().<HtmlTemplateEntity>fromJson(entity.getHtmlTemplate(), HtmlTemplateEntity.class);
    try {
      final Consumer<HtmlTemplateEntity.DataEntity> _function = (HtmlTemplateEntity.DataEntity it) -> {
        try {
          if (((!FileTools.isEmpty(it.getAnalysisShellPath())) && (!FileTools.isEmpty(it.getMustachePath())))) {
            jsEnv.put("analysisFilePath", StringUtils.<String>defaultIfEmpty(it.getAnalysisShellPath(), ""));
            jsEnv.put("templateFilePath", StringUtils.<String>defaultIfEmpty(it.getMustachePath(), ""));
            jsEnv.put("resultFilePath", StringUtils.<String>defaultIfEmpty(it.getResultPath(), ""));
            jsEnv.put("name", StringUtils.<String>defaultIfEmpty(it.getName(), ""));
            jsEnv.put("pname", StringUtils.<String>defaultIfEmpty(it.getPname(), ""));
            String _analysisShellPath = it.getAnalysisShellPath();
            File _file = new File(_analysisShellPath);
            String analysisFile = FileUtils.readFileToString(_file, "UTF-8");
            boolean _isNotBlank = StringUtils.isNotBlank(analysisFile);
            if (_isNotBlank) {
              try (JsRhinoScript js = new JsRhinoScript()) {
                js.set("env", jsEnv);
                js.set("log", TaskController.LOG);
                js.set("otherEnv", extendColumn);
                String _taskDirectory = entity.getTaskDirectory();
                GitFileHistory _gitFileHistory = new GitFileHistory(this.jgit, _taskDirectory);
                js.set("fileHistory", _gitFileHistory);
                StringBuffer buffer = new StringBuffer("function taomubox_template_entry(results){");
                buffer.append(analysisFile);
                buffer.append("return env;");
                buffer.append("}");
                js.loaderScript(buffer.toString());
                Object _invoke = js.invoke("taomubox_template_entry", results);
                Map<String, String> resMap = ((Map<String, String>) _invoke);
                TaskController.LOG.info("解析器返回结果:{}", resMap.toString());
                this.saveRunResult(entity, it, resMap, fileWriteList);
              }
            }
          } else {
            String resultFileName = UUID.randomUUID().toString().replace("-", "");
            String _pname = it.getPname();
            File resFile = GetTaskDirectory.getResultPath(entity, IterableExtensions.join(Collections.<String>unmodifiableList(CollectionLiterals.<String>newArrayList(_pname, resultFileName, "txt")), "."));
            TaskController.LOG.info("将结果保存到:{}", resFile.getAbsolutePath());
            it.setResultPath(resFile.getAbsolutePath().trim());
            String _resultPath = it.getResultPath();
            String _join = IterableExtensions.join(results);
            FileWriteEntity _fileWriteEntity = new FileWriteEntity(_resultPath, _join);
            fileWriteList.add(_fileWriteEntity);
          }
        } catch (Throwable _e) {
          throw Exceptions.sneakyThrow(_e);
        }
      };
      htemplate.getData().forEach(_function);
      Integer _taskVersion = entity.getTaskVersion();
      int _plus = ((_taskVersion).intValue() + 1);
      entity.setTaskVersion(Integer.valueOf(_plus));
      entity.setHtmlTemplate(new Gson().toJson(htemplate));
      entity.setStatus(TaskEntity.TaskStatus.SUCCESS.ordinal());
      TaskController.LOG.info("任务 {} 更新数据库", entity.getTaskName());
      this.taskService.update(entity);
    } catch (final Throwable _t) {
      if (_t instanceof Exception) {
        final Exception ex = (Exception)_t;
        entity.setStatus(TaskEntity.TaskStatus.FAIL.ordinal());
        entity.setExceptionInfo(ex.getMessage());
        this.taskService.update(entity);
      } else {
        throw Exceptions.sneakyThrow(_t);
      }
    }
  }
  
  protected void saveRunResult(final TaskEntity entity, final HtmlTemplateEntity.DataEntity dataEntity, final Map<String, String> results, final List<FileWriteEntity> fileWriteList) {
    try {
      TaskController.LOG.info("保存运行结果");
      if ((dataEntity != null)) {
        TaskController.LOG.info("运行模版引擎生成格式化数据");
        boolean _endsWith = dataEntity.getMustachePath().endsWith(".mustache");
        if (_endsWith) {
          StringWriter writer = new StringWriter();
          DefaultMustacheFactory mf = new DefaultMustacheFactory();
          String _mustachePath = dataEntity.getMustachePath();
          FileInputStream _fileInputStream = new FileInputStream(_mustachePath);
          InputStreamReader reader = new InputStreamReader(_fileInputStream);
          Mustache mustache = mf.compile(reader, dataEntity.getMustachePath());
          mustache.execute(writer, results).flush();
          String resultFileName = UUID.randomUUID().toString().replace("-", "");
          String _pname = dataEntity.getPname();
          File resFile = GetTaskDirectory.getResultPath(entity, 
            IterableExtensions.join(Collections.<String>unmodifiableList(CollectionLiterals.<String>newArrayList(_pname, resultFileName, "txt")), "."));
          TaskController.LOG.info("将结果保存到:{}", resFile.getAbsolutePath());
          dataEntity.setResultPath(resFile.getAbsolutePath().trim());
          String _string = writer.toString();
          String content = IterableExtensions.join(Collections.<String>unmodifiableList(CollectionLiterals.<String>newArrayList(_string)));
          String _absolutePath = resFile.getAbsolutePath();
          FileWriteEntity _fileWriteEntity = new FileWriteEntity(_absolutePath, content);
          fileWriteList.add(_fileWriteEntity);
        }
      }
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  @Override
  public Response exitTask(final String uuid) {
    TaskEntity entity = this.taskService.getOneTask(uuid);
    TaskController.LOG.info("结束任务:{}", entity.getTaskName());
    entity.setStatus(TaskEntity.TaskStatus.DONE.ordinal());
    this.taskService.update(entity);
    return Response.ok().build();
  }
  
  @Override
  public Response getTaskType() {
    TaskController.LOG.info("获取任务类型");
    TaskEntity.TaskType[] _values = TaskEntity.TaskType.values();
    Pair<String, TaskEntity.TaskType[]> _mappedTo = Pair.<String, TaskEntity.TaskType[]>of("task_type", _values);
    final Function<TaskEntity.TaskType, Integer> _function = (TaskEntity.TaskType it) -> {
      return Integer.valueOf(it.ordinal());
    };
    List<Integer> _collect = ((List<TaskEntity.TaskType>)Conversions.doWrapArray(TaskEntity.TaskType.values())).stream().<Integer>map(_function).collect(Collectors.<Integer>toList());
    Pair<String, List<Integer>> _mappedTo_1 = Pair.<String, List<Integer>>of("task_type_ordinal", _collect);
    TaskEntity.ShellType[] _values_1 = TaskEntity.ShellType.values();
    Pair<String, TaskEntity.ShellType[]> _mappedTo_2 = Pair.<String, TaskEntity.ShellType[]>of("shell_type", _values_1);
    final Function<TaskEntity.ShellType, Integer> _function_1 = (TaskEntity.ShellType it) -> {
      return Integer.valueOf(it.ordinal());
    };
    List<Integer> _collect_1 = ((List<TaskEntity.ShellType>)Conversions.doWrapArray(TaskEntity.ShellType.values())).stream().<Integer>map(_function_1).collect(Collectors.<Integer>toList());
    Pair<String, List<Integer>> _mappedTo_3 = Pair.<String, List<Integer>>of("shell_type_ordinal", _collect_1);
    return Response.ok(
      Collections.<Pair<String, ?>>unmodifiableList(CollectionLiterals.<Pair<String, ?>>newArrayList(_mappedTo, _mappedTo_1, _mappedTo_2, _mappedTo_3))).build();
  }
  
  @Override
  public Response getTaskResult(final String uuid) {
    TaskEntity entity = this.taskService.getOneTask(uuid);
    TaskController.LOG.info("获取任务{}结果", entity.getTaskName());
    TaskController.LOG.info("格式化markdown数据");
    HtmlTemplateEntity htmlTemplate = new Gson().<HtmlTemplateEntity>fromJson(entity.getHtmlTemplate(), HtmlTemplateEntity.class);
    final Consumer<HtmlTemplateEntity.DataEntity> _function = (HtmlTemplateEntity.DataEntity it) -> {
      try {
        boolean _isNotBlank = StringUtils.isNotBlank(it.getResultPath());
        if (_isNotBlank) {
          String result = MarkdownUtils.getHtml(it.getResultPath());
          it.setMustacheText(result);
          String _resultPath = it.getResultPath();
          File _file = new File(_resultPath);
          it.setResultPath(FileUtils.readFileToString(_file, "UTF-8"));
        }
      } catch (Throwable _e) {
        throw Exceptions.sneakyThrow(_e);
      }
    };
    htmlTemplate.getData().forEach(_function);
    int _status = entity.getStatus();
    int _ordinal = TaskEntity.TaskStatus.SUCCESS.ordinal();
    boolean _equals = (_status == _ordinal);
    if (_equals) {
      int _status_1 = entity.getStatus();
      List<HtmlTemplateEntity.DataEntity> _data = htmlTemplate.getData();
      TaskResultEntity<List<HtmlTemplateEntity.DataEntity>> _taskId = new TaskResultEntity<List<HtmlTemplateEntity.DataEntity>>(_status_1, _data).taskId(entity.getTaskId());
      ResponseEntity<TaskResultEntity<List<HtmlTemplateEntity.DataEntity>>> res = new ResponseEntity<TaskResultEntity<List<HtmlTemplateEntity.DataEntity>>>(_taskId);
      return Response.ok(res.msg("获取任务结果")).build();
    }
    ResponseEntity<Object> res_1 = new ResponseEntity<Object>(Response.Status.NOT_FOUND);
    return Response.status(Response.Status.NOT_FOUND).entity(res_1).build();
  }
  
  @Override
  public Response getTaskList() {
    List<TaskEntity> entitys = this.taskService.getTaskAll();
    final List<TaskEntity> _converted_entitys = (List<TaskEntity>)entitys;
    TaskController.LOG.info("获取任务列表:{}", Integer.valueOf(((Object[])Conversions.unwrapArray(_converted_entitys, Object.class)).length));
    return Response.ok(entitys).build();
  }
  
  @Override
  public Response getTaskStatus(final String uuid) {
    TaskEntity entity = this.taskService.getOneTask(uuid);
    int _status = entity.getStatus();
    int _ordinal = TaskEntity.TaskStatus.SUCCESS.ordinal();
    boolean _equals = (_status == _ordinal);
    if (_equals) {
      ResponseEntity<Object> res = new ResponseEntity<Object>(201);
      return Response.status(201).entity(res).build();
    } else {
      int _status_1 = entity.getStatus();
      int _ordinal_1 = TaskEntity.TaskStatus.IN_PROGRESS.ordinal();
      boolean _equals_1 = (_status_1 == _ordinal_1);
      if (_equals_1) {
        ResponseEntity<Object> res_1 = new ResponseEntity<Object>(100);
        return Response.status(100).entity(res_1).build();
      } else {
        int _status_2 = entity.getStatus();
        int _ordinal_2 = TaskEntity.TaskStatus.FAIL.ordinal();
        boolean _equals_2 = (_status_2 == _ordinal_2);
        if (_equals_2) {
          ResponseEntity<Object> res_2 = new ResponseEntity<Object>(500);
          return Response.status(500).entity(res_2).build();
        } else {
          int _status_3 = entity.getStatus();
          int _ordinal_3 = TaskEntity.TaskStatus.WAIT.ordinal();
          boolean _equals_3 = (_status_3 == _ordinal_3);
          if (_equals_3) {
            ResponseEntity<Object> res_3 = new ResponseEntity<Object>(202);
            return Response.status(202).entity(res_3).build();
          }
        }
      }
    }
    ResponseEntity<Object> res_4 = new ResponseEntity<Object>(100);
    return Response.status(100).entity(res_4).build();
  }
}
