/**
 * Copyright (c) 2023 murenchao
 * taomu is licensed under Mulan PubL v2.
 * You can use this software according to the terms and conditions of the Mulan PubL v2.
 * You may obtain a copy of Mulan PubL v2 at:
 *       http://license.coscl.org.cn/MulanPubL-2.0
 * THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
 * EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
 * MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
 * See the Mulan PubL v2 for more details.
 */
package cool.taomu.box.netty.mqtt.impl;

import cool.taomu.box.netty.mqtt.entity.MessageEntity;
import cool.taomu.box.netty.mqtt.extend.MqttUtils;
import cool.taomu.box.netty.mqtt.inter.INettyMqtt;
import cool.taomu.box.netty.mqtt.service.MqttPublishService;
import io.netty.channel.ChannelHandlerContext;
import io.netty.handler.codec.mqtt.MqttFixedHeader;
import io.netty.handler.codec.mqtt.MqttMessage;
import io.netty.handler.codec.mqtt.MqttMessageIdVariableHeader;
import io.netty.handler.codec.mqtt.MqttMessageType;
import io.netty.handler.codec.mqtt.MqttPubAckMessage;
import io.netty.handler.codec.mqtt.MqttPublishMessage;
import io.netty.handler.codec.mqtt.MqttQoS;
import io.netty.util.ReferenceCountUtil;
import javax.inject.Inject;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@SuppressWarnings("all")
public class MqttPublish implements INettyMqtt {
  private static final Logger LOG = LoggerFactory.getLogger(MqttPublish.class);
  
  @Inject
  private MqttPublishService publish;
  
  @Override
  public void request(final ChannelHandlerContext ctx, final MqttMessage mqttMessage) {
    try {
      if ((!(mqttMessage instanceof MqttPublishMessage))) {
        return;
      }
      MqttPublishMessage publishMessage = ((MqttPublishMessage) mqttMessage);
      MessageEntity message = new MessageEntity();
      message.setSenderId(MqttUtils.getClientId(ctx.channel()));
      String _senderId = message.getSenderId();
      String _plus = ("执行了MQTT Publish 命令 : " + _senderId);
      MqttPublish.LOG.info(_plus);
      MqttQoS qos = publishMessage.fixedHeader().qosLevel();
      message.setQos(qos.ordinal());
      message.setTopic(publishMessage.variableHeader().topicName());
      message.setPayload(((MqttPublishMessage) mqttMessage).payload());
      message.setType(Integer.valueOf(mqttMessage.fixedHeader().messageType().value()));
      message.setDup(publishMessage.fixedHeader().isDup());
      message.setRetain(publishMessage.fixedHeader().isRetain());
      message.setMsgId(publishMessage.variableHeader().packetId());
      message.setSenderChannel(ctx.channel());
      if (qos != null) {
        switch (qos) {
          case EXACTLY_ONCE:
          case AT_MOST_ONCE:
            MqttPublish.LOG.info(String.format("Qos0 and Qos2 message,clientId=%s", message.getSenderId()));
            this.publish.message(message);
            break;
          case AT_LEAST_ONCE:
            MqttPublish.LOG.info(String.format("Qos1 message,clientId=%s", message.getSenderId()));
            this.publish.message(message);
            MqttFixedHeader header = new MqttFixedHeader(MqttMessageType.PUBACK, false, MqttQoS.AT_MOST_ONCE, false, 0);
            MqttMessageIdVariableHeader varHeader = MqttMessageIdVariableHeader.from(message.getMsgId());
            MqttPubAckMessage _mqttPubAckMessage = new MqttPubAckMessage(header, varHeader);
            ctx.writeAndFlush(_mqttPubAckMessage);
            break;
          default:
            MqttPublish.LOG.info(String.format("Wrong mqtt message,clientId=%s", message.getSenderId()));
            break;
        }
      } else {
        MqttPublish.LOG.info(String.format("Wrong mqtt message,clientId=%s", message.getSenderId()));
      }
    } catch (final Throwable _t) {
      if (_t instanceof Exception) {
        final Exception ex = (Exception)_t;
        MqttPublish.LOG.debug("执行了MQTT Publish 命令出错了 : ", ex);
      } else {
        throw Exceptions.sneakyThrow(_t);
      }
    } finally {
      ReferenceCountUtil.release(mqttMessage.payload());
    }
  }
}
