/**
 * Copyright (c) 2023 murenchao
 * taomu is licensed under Mulan PubL v2.
 * You can use this software according to the terms and conditions of the Mulan PubL v2.
 * You may obtain a copy of Mulan PubL v2 at:
 *       http://license.coscl.org.cn/MulanPubL-2.0
 * THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
 * EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
 * MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
 * See the Mulan PubL v2 for more details.
 */
package cool.taomu.box.netty.mqtt.impl;

import com.google.inject.Inject;
import cool.taomu.box.netty.mqtt.extend.MqttUtils;
import cool.taomu.box.netty.mqtt.inter.INettyMqtt;
import cool.taomu.box.netty.mqtt.service.MqttConnectService;
import cool.taomu.box.netty.mqtt.service.MqttSessionService;
import cool.taomu.box.netty.mqtt.service.MqttWillService;
import cool.taomu.box.netty.mqtt.utils.CheckConnect;
import io.netty.channel.ChannelFuture;
import io.netty.channel.ChannelHandlerContext;
import io.netty.channel.ChannelPipeline;
import io.netty.handler.codec.mqtt.MqttConnAckMessage;
import io.netty.handler.codec.mqtt.MqttConnAckVariableHeader;
import io.netty.handler.codec.mqtt.MqttConnectMessage;
import io.netty.handler.codec.mqtt.MqttConnectReturnCode;
import io.netty.handler.codec.mqtt.MqttFixedHeader;
import io.netty.handler.codec.mqtt.MqttMessage;
import io.netty.handler.codec.mqtt.MqttMessageType;
import io.netty.handler.codec.mqtt.MqttQoS;
import io.netty.handler.timeout.IdleStateHandler;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@SuppressWarnings("all")
public class MqttConnect implements INettyMqtt {
  private static final Logger LOG = LoggerFactory.getLogger(MqttConnect.class);
  
  private MqttConnectReturnCode code;
  
  private boolean sessionPresent;
  
  @Inject
  private MqttSessionService session;
  
  @Inject
  private MqttWillService will;
  
  @Inject
  private MqttConnectService service;
  
  @Override
  public void request(final ChannelHandlerContext ctx, final MqttMessage mqttMessage) {
    if ((!(mqttMessage instanceof MqttConnectMessage))) {
      return;
    }
    MqttConnect.LOG.info("执行 Connect 操作");
    boolean sessionPresent = false;
    try {
      MqttConnectMessage connect = ((MqttConnectMessage) mqttMessage);
      int version = connect.variableHeader().version();
      MqttConnect.LOG.info("mqtt version:{}", Integer.valueOf(version));
      boolean isCleanSession = connect.variableHeader().isCleanSession();
      String clientId = connect.payload().clientIdentifier();
      String uname = connect.payload().userName();
      byte[] passwd = connect.payload().passwordInBytes();
      int heartbeatSec = connect.variableHeader().keepAliveTimeSeconds();
      MqttConnect.LOG.info("clientId:{},cleanSession:{}", clientId, Boolean.valueOf(isCleanSession));
      boolean _version = CheckConnect.version(version);
      boolean _not = (!_version);
      if (_not) {
        this.code = MqttConnectReturnCode.CONNECTION_REFUSED_UNACCEPTABLE_PROTOCOL_VERSION;
        this.sessionPresent = sessionPresent;
      } else {
        boolean _clientId = CheckConnect.clientId(ctx.channel(), clientId, this.service);
        boolean _not_1 = (!_clientId);
        if (_not_1) {
          this.code = MqttConnectReturnCode.CONNECTION_REFUSED_IDENTIFIER_REJECTED;
          this.sessionPresent = sessionPresent;
        } else {
          boolean _authorized = CheckConnect.authorized(MqttUtils.getRemoteAddr(ctx.channel()), clientId);
          boolean _not_2 = (!_authorized);
          if (_not_2) {
            this.code = MqttConnectReturnCode.CONNECTION_REFUSED_NOT_AUTHORIZED;
            this.sessionPresent = sessionPresent;
          } else {
            boolean _userAuth = CheckConnect.userAuth(clientId, uname, passwd);
            boolean _not_3 = (!_userAuth);
            if (_not_3) {
              this.code = MqttConnectReturnCode.CONNECTION_REFUSED_BAD_USER_NAME_OR_PASSWORD;
              this.sessionPresent = sessionPresent;
            } else {
              boolean _keepAlive = this.keepAlive(clientId, ctx, heartbeatSec);
              boolean _not_4 = (!_keepAlive);
              if (_not_4) {
                String failure = String.format("set heartbeat failure clientId:%s,heartbeatSec:%d", clientId, Integer.valueOf(heartbeatSec));
                throw new Exception(failure);
              }
              this.code = MqttConnectReturnCode.CONNECTION_ACCEPTED;
              sessionPresent = this.session.create(clientId, ctx, isCleanSession);
              this.will.store(connect, clientId);
            }
          }
        }
      }
    } catch (final Throwable _t) {
      if (_t instanceof Exception) {
        final Exception ex = (Exception)_t;
        MqttConnect.LOG.info("服务不可用 :", ex);
        this.code = MqttConnectReturnCode.CONNECTION_REFUSED_SERVER_UNAVAILABLE;
        this.sessionPresent = sessionPresent;
      } else {
        throw Exceptions.sneakyThrow(_t);
      }
    } finally {
      this.response(ctx, mqttMessage);
    }
  }
  
  public ChannelFuture response(final ChannelHandlerContext ctx, final MqttMessage mqttMessage) {
    ChannelFuture _xblockexpression = null;
    {
      MqttFixedHeader header = new MqttFixedHeader(MqttMessageType.CONNACK, false, MqttQoS.AT_MOST_ONCE, false, 0);
      MqttConnAckVariableHeader varHeader = new MqttConnAckVariableHeader(this.code, this.sessionPresent);
      MqttConnAckMessage _mqttConnAckMessage = new MqttConnAckMessage(header, varHeader);
      _xblockexpression = ctx.writeAndFlush(_mqttConnAckMessage);
    }
    return _xblockexpression;
  }
  
  public boolean keepAlive(final String clientId, final ChannelHandlerContext ctx, final int heatbeatSec) {
    MqttConnect.LOG.info("设置keep alive");
    int keepAlive = ((int) (heatbeatSec * 1.5f));
    boolean _contains = ctx.pipeline().names().contains("idleStateHandler");
    if (_contains) {
      ctx.pipeline().remove("idleStateHandler");
    }
    ChannelPipeline _pipeline = ctx.pipeline();
    IdleStateHandler _idleStateHandler = new IdleStateHandler(keepAlive, 0, 0);
    _pipeline.addFirst("idleStateHandler", _idleStateHandler);
    return true;
  }
}
