/**
 * Copyright (c) 2023 murenchao
 * taomu is licensed under Mulan PubL v2.
 * You can use this software according to the terms and conditions of the Mulan PubL v2.
 * You may obtain a copy of Mulan PubL v2 at:
 *       http://license.coscl.org.cn/MulanPubL-2.0
 * THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
 * EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
 * MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
 * See the Mulan PubL v2 for more details.
 */
package cool.taomu.box.netty.mqtt.impl;

import com.google.inject.Inject;
import cool.taomu.box.netty.mqtt.entity.MessageEntity;
import cool.taomu.box.netty.mqtt.entity.TopicEntity;
import cool.taomu.box.netty.mqtt.extend.MqttUtils;
import cool.taomu.box.netty.mqtt.inter.INettyMqtt;
import cool.taomu.box.netty.mqtt.service.MqttPublishService;
import cool.taomu.box.netty.mqtt.service.MqttSubscribeService;
import cool.taomu.box.netty.mqtt.utils.ISerializationUtils;
import io.netty.channel.ChannelHandlerContext;
import io.netty.handler.codec.mqtt.MqttFixedHeader;
import io.netty.handler.codec.mqtt.MqttMessage;
import io.netty.handler.codec.mqtt.MqttMessageIdVariableHeader;
import io.netty.handler.codec.mqtt.MqttMessageType;
import io.netty.handler.codec.mqtt.MqttQoS;
import io.netty.handler.codec.mqtt.MqttSubAckMessage;
import io.netty.handler.codec.mqtt.MqttSubAckPayload;
import io.netty.handler.codec.mqtt.MqttSubscribeMessage;
import io.netty.handler.codec.mqtt.MqttTopicSubscription;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.function.Consumer;
import org.apache.oro.text.perl.Perl5Util;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.lmdbjava.Dbi;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@SuppressWarnings("all")
public class MqttSubscribe implements INettyMqtt {
  private static final Logger LOG = LoggerFactory.getLogger(MqttSubscribe.class);
  
  @Inject
  private ISerializationUtils isu;
  
  @Inject
  private MqttPublishService publish;
  
  @Inject
  private MqttSubscribeService subscribeService;
  
  public byte[] toBytes(final TopicEntity s) {
    return this.isu.serialize(s);
  }
  
  @Override
  public void request(final ChannelHandlerContext ctx, final MqttMessage mqttMessage) {
    try {
      if ((!(mqttMessage instanceof MqttSubscribeMessage))) {
        return;
      }
      String clientId = MqttUtils.getClientId(ctx.channel());
      if ((clientId == null)) {
        Thread.sleep(100);
        clientId = MqttUtils.getClientId(ctx.channel());
      }
      MqttSubscribe.LOG.info(("执行了MQTT Subscribe 命令 : " + clientId));
      try {
        MqttSubscribeMessage subscribeMessage = ((MqttSubscribeMessage) mqttMessage);
        int messageId = subscribeMessage.variableHeader().messageId();
        ArrayList<TopicEntity> validTopicList = this.registerTopics(ctx, subscribeMessage.payload().topicSubscriptions());
        synchronized (validTopicList) {
          if (((validTopicList == null) || (validTopicList.size() == 0))) {
            MqttSubscribe.LOG.info(String.format("Valid all subscribe topic failure,messageId:%s", Integer.valueOf(messageId)));
            return;
          }
          MqttFixedHeader header = new MqttFixedHeader(MqttMessageType.SUBACK, false, MqttQoS.AT_MOST_ONCE, false, 0);
          MqttMessageIdVariableHeader varHeader = MqttMessageIdVariableHeader.from(messageId);
          List<Integer> _topicQos = this.getTopicQos(validTopicList);
          MqttSubAckPayload _mqttSubAckPayload = new MqttSubAckPayload(_topicQos);
          MqttSubAckMessage _mqttSubAckMessage = new MqttSubAckMessage(header, varHeader, _mqttSubAckPayload);
          ctx.writeAndFlush(_mqttSubAckMessage);
        }
      } catch (final Throwable _t) {
        if (_t instanceof Exception) {
          final Exception ex = (Exception)_t;
          MqttSubscribe.LOG.debug("subscribe requst exception:", ex);
        } else {
          throw Exceptions.sneakyThrow(_t);
        }
      }
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  public List<Integer> getTopicQos(final List<TopicEntity> topics) {
    int _size = topics.size();
    ArrayList<Integer> qoss = new ArrayList<Integer>(_size);
    for (final TopicEntity topic : topics) {
      qoss.add(Integer.valueOf(topic.getQos()));
    }
    return qoss;
  }
  
  public synchronized ArrayList<TopicEntity> registerTopics(final ChannelHandlerContext ctx, final List<MqttTopicSubscription> topics) {
    synchronized (ctx) {
      String clientId = MqttUtils.getClientId(ctx.channel());
      ArrayList<TopicEntity> topicList = new ArrayList<TopicEntity>();
      for (final MqttTopicSubscription subscription : topics) {
        {
          String _picFilter = subscription.topicFilter();
          int _value = subscription.qualityOfService().value();
          final TopicEntity topic = new TopicEntity(_picFilter, _value);
          topic.setClientId(clientId);
          MqttSubscribe.LOG.info("订阅Topic {} 的用户{}", topic.getName(), clientId);
          Perl5Util p5 = new Perl5Util();
          boolean _match = p5.match("/^[A-Za-z0-9]+([\\/A-Za-z0-9_]*|\\/\\+||\\/\\#)$/", topic.getName());
          if (_match) {
            this.subscribeService.add(topic);
            HashSet<MessageEntity> retain = new HashSet<MessageEntity>();
            try {
            } catch (final Throwable _t) {
              if (_t instanceof Dbi.KeyNotFoundException) {
              } else {
                throw Exceptions.sneakyThrow(_t);
              }
            }
            if (((!IterableExtensions.isNullOrEmpty(retain)) && (retain.size() > 0))) {
              final Consumer<MessageEntity> _function = (MessageEntity it) -> {
                this.publish.publishMessage(topic, it);
              };
              retain.forEach(_function);
            }
          }
          topicList.add(topic);
        }
      }
      return topicList;
    }
  }
}
