/**
 * Copyright (c) 2023 murenchao
 * taomu is licensed under Mulan PubL v2.
 * You can use this software according to the terms and conditions of the Mulan PubL v2.
 * You may obtain a copy of Mulan PubL v2 at:
 *       http://license.coscl.org.cn/MulanPubL-2.0
 * THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
 * EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
 * MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
 * See the Mulan PubL v2 for more details.
 */
package cool.taomu.box.netty.mqtt.service;

import com.google.common.base.Optional;
import com.google.inject.Inject;
import cool.taomu.box.crypto.Base64;
import cool.taomu.box.netty.mqtt.entity.ClientSessionEntity;
import cool.taomu.box.netty.mqtt.entity.MessageEntity;
import cool.taomu.box.netty.mqtt.entity.PublishEntity;
import cool.taomu.box.netty.mqtt.entity.TopicEntity;
import cool.taomu.box.netty.mqtt.extend.MqttUtils;
import cool.taomu.box.netty.mqtt.inter.ICache;
import cool.taomu.box.netty.mqtt.utils.ISerializationUtils;
import cool.taomu.box.task.dataentity.DataStructureEntity;
import io.netty.buffer.ByteBuf;
import io.netty.buffer.Unpooled;
import io.netty.handler.codec.mqtt.MqttFixedHeader;
import io.netty.handler.codec.mqtt.MqttMessageType;
import io.netty.handler.codec.mqtt.MqttPublishMessage;
import io.netty.handler.codec.mqtt.MqttPublishVariableHeader;
import io.netty.handler.codec.mqtt.MqttQoS;
import java.util.ArrayList;
import java.util.function.Consumer;
import javax.inject.Named;
import org.apache.commons.lang3.SerializationUtils;
import org.apache.oro.text.perl.Perl5Util;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@SuppressWarnings("all")
public class MqttPublishService {
  private static final Logger LOG = LoggerFactory.getLogger(MqttPublishService.class);
  
  @Inject
  @Named("ClientSessionEntity")
  private ICache<Optional<ClientSessionEntity>> cache;
  
  @Inject
  private ISerializationUtils isu;
  
  @Inject
  private MqttSubscribeService subscribeService;
  
  public byte[] toBytes(final MessageEntity s) {
    return this.isu.serialize(s);
  }
  
  public void message(final MessageEntity senderMessage) {
    MqttPublishService.LOG.debug("推送消息");
    ArrayList<TopicEntity> subscribes = this.subscribeService.get();
    try {
      byte[] _payload = senderMessage.getPayload();
      byte[] msg = new Base64(_payload).decode();
      Object _deserialize = this.isu.<Object>deserialize(msg);
      final DataStructureEntity messageEntity = ((DataStructureEntity) _deserialize);
      final Consumer<TopicEntity> _function = (TopicEntity it) -> {
        boolean _equals = messageEntity.getClientId().equals("ALL");
        if (_equals) {
          this.publishMessage(it, senderMessage);
        } else {
          boolean _equals_1 = it.getClientId().equals(messageEntity.getClientId());
          if (_equals_1) {
            this.publishMessage(it, senderMessage);
          }
        }
      };
      IterableExtensions.<TopicEntity>filterNull(subscribes).forEach(_function);
    } catch (final Throwable _t) {
      if (_t instanceof Exception) {
        final Exception ex = (Exception)_t;
        MqttPublishService.LOG.info("序列化错误:{}", ex.getMessage());
      } else {
        throw Exceptions.sneakyThrow(_t);
      }
    }
  }
  
  public void publishMessage(final TopicEntity topic, final MessageEntity senderMessage) {
    try {
      Perl5Util p5 = new Perl5Util();
      String subTopicName = topic.getName().replace("/+", "/[a-zA-Z]?[a-zA-Z0-9]+").replace("/#", 
        "/[a-zA-Z]?([a-zA-Z0-9/]*)").replace("/", "\\/");
      boolean _match = p5.match((("/" + subTopicName) + "/"), senderMessage.getTopic());
      if (_match) {
        MqttPublishService.LOG.info("发送者id : {},  Topic : {}", senderMessage.getSenderId(), senderMessage.getTopic());
        int minQos = MqttUtils.getQos(senderMessage.getQos(), topic.getQos());
        if ((minQos == 2)) {
          MessageEntity cloneMsg = SerializationUtils.<MessageEntity>clone(senderMessage);
          cloneMsg.setSenderChannel(senderMessage.getSenderChannel());
        }
        ClientSessionEntity clientSession = this.cache.get(topic.getClientId()).get();
        MqttPublishService.LOG.info("clientSession is not null {}", Boolean.valueOf((clientSession != null)));
        int _generateMessageId = clientSession.generateMessageId();
        Integer msgId = ((Integer) Integer.valueOf(_generateMessageId));
        MqttPublishService.LOG.info("订阅者id : {},  Topic : {}, 发送内容长度： {} messageId : {}", topic.getClientId(), topic.getName(), 
          Integer.valueOf(senderMessage.getPayload().length), msgId);
        MqttQoS _valueOf = MqttQoS.valueOf(minQos);
        String _name = topic.getName();
        byte[] _payload = senderMessage.getPayload();
        PublishEntity entity = new PublishEntity(_valueOf, _name, msgId, _payload, 
          Boolean.valueOf(false));
        clientSession.getCtx().writeAndFlush(this.response(entity));
      }
    } catch (final Throwable _t) {
      if (_t instanceof Exception) {
        final Exception ex = (Exception)_t;
        MqttPublishService.LOG.debug("publishMessage 方法出现错误 : ", ex);
      } else {
        throw Exceptions.sneakyThrow(_t);
      }
    }
  }
  
  private MqttPublishMessage response(final PublishEntity entity) {
    MqttPublishService.LOG.debug("------<>{}", entity);
    Boolean _dup = entity.getDup();
    MqttQoS _qos = entity.getQos();
    MqttFixedHeader header = new MqttFixedHeader(MqttMessageType.PUBLISH, (_dup).booleanValue(), _qos, false, 0);
    String _topicName = entity.getTopicName();
    Integer _messageId = entity.getMessageId();
    MqttPublishVariableHeader varHeader = new MqttPublishVariableHeader(_topicName, (_messageId).intValue());
    ByteBuf heapBuf = null;
    byte[] _payload = entity.getPayload();
    boolean _tripleEquals = (_payload == null);
    if (_tripleEquals) {
      heapBuf = Unpooled.EMPTY_BUFFER;
    } else {
      try {
        heapBuf = Unpooled.wrappedBuffer(entity.getPayload());
      } catch (final Throwable _t) {
        if (_t instanceof IllegalArgumentException) {
          final IllegalArgumentException e = (IllegalArgumentException)_t;
          e.printStackTrace();
        } else {
          throw Exceptions.sneakyThrow(_t);
        }
      }
    }
    return new MqttPublishMessage(header, varHeader, heapBuf);
  }
}
