/**
 * Copyright (c) 2023 murenchao
 * taomu is licensed under Mulan PubL v2.
 * You can use this software according to the terms and conditions of the Mulan PubL v2.
 * You may obtain a copy of Mulan PubL v2 at:
 *       http://license.coscl.org.cn/MulanPubL-2.0
 * THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
 * EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
 * MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
 * See the Mulan PubL v2 for more details.
 */
package cool.taomu.box.script.rhino;

import cool.taomu.box.script.AbsScriptClassLoader;
import java.io.File;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.function.Consumer;
import org.apache.commons.io.FileUtils;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure2;
import org.mozilla.javascript.Context;
import org.mozilla.javascript.Function;
import org.mozilla.javascript.ScriptableObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@SuppressWarnings("all")
public class JsRhinoScript extends AbsScriptClassLoader {
  private static final Logger LOG = LoggerFactory.getLogger(JsRhinoScript.class);
  
  private final Context cx = Context.enter();
  
  private final ScriptableObject scope = this.cx.initStandardObjects();
  
  public JsRhinoScript() {
  }
  
  public JsRhinoScript(final String path) {
    this.init(Collections.<String>unmodifiableList(CollectionLiterals.<String>newArrayList(path)));
  }
  
  @Override
  public void close() throws Exception {
    Context.exit();
  }
  
  @Override
  public void init(final List<String> paths, final ClassLoader loader) {
    JsRhinoScript.LOG.info(String.format("加载脚本:%s", paths));
    final Procedure2<String, Integer> _function = (String v, Integer i) -> {
      try {
        File _file = new File(v);
        String script = FileUtils.readFileToString(_file, "UTF-8");
        this.loaderScript(script, (i).intValue(), loader);
      } catch (Throwable _e) {
        throw Exceptions.sneakyThrow(_e);
      }
    };
    IterableExtensions.<String>forEach(paths, _function);
  }
  
  @Override
  public void loaderScript(final String src, final int lineon, final ClassLoader loader) {
    this.cx.evaluateString(this.scope, src, null, lineon, null);
  }
  
  @Override
  public Object invoke(final String zlassName, final String methodName, final Class<?>[] paramTypes, final Object... args) {
    Object func = this.scope.get(methodName);
    if ((func instanceof Function)) {
      Function jfunc = ((Function)func);
      final ArrayList<Object> jsArgs = CollectionLiterals.<Object>newArrayList();
      final Consumer<Object> _function = (Object it) -> {
        jsArgs.add(Context.javaToJS(it, this.scope));
      };
      ((List<Object>)Conversions.doWrapArray(args)).forEach(_function);
      Object result = jfunc.call(this.cx, this.scope, this.scope, ((Object[])Conversions.unwrapArray(jsArgs, Object.class)));
      Object _xifexpression = null;
      if ((result != null)) {
        _xifexpression = Context.jsToJava(result, Object.class);
      } else {
        _xifexpression = null;
      }
      return _xifexpression;
    } else {
      JsRhinoScript.LOG.info(("unknown method : " + methodName));
      return null;
    }
  }
  
  @Override
  public void loaderScript(final String coding) {
    this.cx.evaluateString(this.scope, coding, null, 2, null);
  }
  
  public void set(final String name, final Object value) {
    Object obj = Context.javaToJS(value, this.scope);
    ScriptableObject.putProperty(this.scope, name, obj);
  }
}
