/**
 * Copyright (c) 2023 murenchao
 * taomu is licensed under Mulan PubL v2.
 * You can use this software according to the terms and conditions of the Mulan PubL v2.
 * You may obtain a copy of Mulan PubL v2 at:
 *       http://license.coscl.org.cn/MulanPubL-2.0
 * THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
 * EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
 * MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
 * See the Mulan PubL v2 for more details.
 */
package cool.taomu.box.thread.cluster;

import com.google.inject.Guice;
import com.google.inject.Key;
import com.google.inject.name.Names;
import cool.taomu.box.guice.TaomuGuiceModule;
import cool.taomu.box.guice.ann.Binder;
import cool.taomu.box.guice.entity.Scope;
import cool.taomu.box.script.AbsScriptClassLoader;
import cool.taomu.box.script.groovy.GroovyScript;
import cool.taomu.box.thread.cluster.entity.TaskEntity;
import java.io.Serializable;
import org.apache.commons.lang3.Validate;
import org.eclipse.xtend.lib.annotations.Accessors;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.Pure;
import org.eclipse.xtext.xbase.lib.StringExtensions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Accessors
@Binder(bind = AbsScriptClassLoader.class, to = GroovyScript.class, named = "Groovy", in = Scope.SINGLETON)
@Binder(bind = AbsScriptClassLoader.class, to = GroovyScript.class, named = "JavaScript", in = Scope.SINGLETON)
@SuppressWarnings("all")
public class TaomuRunnable implements Runnable, Serializable {
  private static final Logger LOG = LoggerFactory.getLogger(TaomuRunnable.class);
  
  private TaskEntity task;
  
  private com.google.inject.Module module;
  
  public TaomuRunnable() {
  }
  
  public TaomuRunnable(final TaskEntity task) {
    TaomuRunnable.LOG.info("实例任务:{}", task.getMain_class());
    this.task = task;
    this.module = new TaomuGuiceModule().build(TaomuRunnable.class);
  }
  
  @Override
  public void run() {
    synchronized (TaskEntity.class) {
      String script = Validate.<String>notBlank(this.task.getScript(), "script 不能为空");
      try {
        TaomuRunnable.LOG.info("开始执行任务:{}->{}", this.task.getScriptType().name(), this.task.getMain_class());
        String[] splitMainClass = this.task.getMain_class().split("#");
        AbsScriptClassLoader ascl = Guice.createInjector(this.module).<AbsScriptClassLoader>getInstance(
          Key.<AbsScriptClassLoader>get(AbsScriptClassLoader.class, Names.named(this.task.getScriptType().name())));
        ascl.loaderScript(this.task.getScript());
        ascl.invoke(splitMainClass[0], splitMainClass[1], new Class[] { String.class }, this.task.getPayload());
      } catch (final Throwable _t) {
        if (_t instanceof Exception) {
          final Exception ex = (Exception)_t;
          TaomuRunnable.LOG.info("执行异常:", ex);
          try {
            boolean _isNullOrEmpty = StringExtensions.isNullOrEmpty(script);
            boolean _not = (!_isNullOrEmpty);
            if (_not) {
              TaomuRunnable.LOG.info("重新加载任务脚本");
            }
          } catch (final Throwable _t_1) {
            if (_t_1 instanceof Exception) {
              TaomuRunnable.LOG.info("执行异常2:", ex);
            } else {
              throw Exceptions.sneakyThrow(_t_1);
            }
          }
        } else {
          throw Exceptions.sneakyThrow(_t);
        }
      }
    }
  }
  
  @Pure
  public TaskEntity getTask() {
    return this.task;
  }
  
  public void setTask(final TaskEntity task) {
    this.task = task;
  }
  
  @Pure
  public com.google.inject.Module getModule() {
    return this.module;
  }
  
  public void setModule(final com.google.inject.Module module) {
    this.module = module;
  }
}
