/**
 * Copyright (c) 2022 murenchao
 * taomu framework is licensed under Mulan PubL v2.
 * You can use this software according to the terms and conditions of the Mulan PubL v2.
 * You may obtain a copy of Mulan PubL v2 at:
 *       http://license.coscl.org.cn/MulanPubL-2.0
 * THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
 * EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
 * MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
 * See the Mulan PubL v2 for more details.
 */
package cool.taomu.framework.service.mqtt.broker.impl.request;

import cool.taomu.framework.cache.KeyValueCache;
import cool.taomu.framework.configure.ConfigureManage;
import cool.taomu.framework.configure.entity.ConfigureEntity;
import cool.taomu.framework.service.mqtt.broker.entity.ClientSessionEntity;
import cool.taomu.framework.service.mqtt.broker.entity.ConnackEntity;
import cool.taomu.framework.service.mqtt.broker.entity.WillEntity;
import cool.taomu.framework.service.mqtt.broker.impl.response.ConnackResponse;
import cool.taomu.framework.service.mqtt.broker.inter.IRequest;
import cool.taomu.framework.service.utils.CommonUtils;
import cool.taomu.framework.utils.spi.Alias;
import io.netty.channel.ChannelHandlerContext;
import io.netty.channel.ChannelPipeline;
import io.netty.handler.codec.mqtt.MqttConnectMessage;
import io.netty.handler.codec.mqtt.MqttConnectReturnCode;
import io.netty.handler.codec.mqtt.MqttMessage;
import io.netty.handler.codec.mqtt.MqttQoS;
import io.netty.handler.timeout.IdleStateHandler;
import java.util.Collections;
import java.util.List;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * CONNECT  V    |   1   |   客户端到服务端     |   客户端请求连接服务端 T
 */
@Alias(value = "CONNECT")
@SuppressWarnings("all")
public class ConnectRequest implements IRequest {
  private final Logger LOG = LoggerFactory.getLogger(ConnectRequest.class);
  
  private KeyValueCache cache = KeyValueCache.getInstance();
  
  private ChannelHandlerContext ctx = null;
  
  public boolean checkVersion(final int mqttVersion) {
    switch (mqttVersion) {
      case 3:
      case 4:
        return true;
      default:
        return false;
    }
  }
  
  /**
   * TODO 未实现
   */
  public boolean checkClientId(final String clientId) {
    boolean _equals = CommonUtils.getClientId(this.ctx.channel()).equals(clientId);
    if (_equals) {
      return true;
    }
    return false;
  }
  
  /**
   * TODO 未实现
   */
  public boolean checkAuthorized(final String addr, final String clientId) {
    return true;
  }
  
  /**
   * TODO 部分实现
   */
  public boolean checkUserAuth(final String clientId, final String userName, final byte[] password) {
    ConfigureEntity config = ConfigureManage.loadConfig();
    boolean _isAnonymous = config.getMqtt().isAnonymous();
    if (_isAnonymous) {
      return true;
    }
    if ((userName.equals(config.getMqtt().getUsername()) && config.getMqtt().getPassword().equals(new String(password)))) {
      return true;
    }
    return false;
  }
  
  /**
   * TODO 部分未实现
   */
  public boolean keepAlive(final String clientId, final ChannelHandlerContext ctx, final int heatbeatSec) {
    this.LOG.info("设置keep alive");
    int keepAlive = ((int) (heatbeatSec * 1.5f));
    boolean _contains = ctx.pipeline().names().contains("idleStateHandler");
    if (_contains) {
      ctx.pipeline().remove("idleStateHandler");
    }
    ChannelPipeline _pipeline = ctx.pipeline();
    IdleStateHandler _idleStateHandler = new IdleStateHandler(keepAlive, 0, 0);
    _pipeline.addFirst("idleStateHandler", _idleStateHandler);
    return true;
  }
  
  @Override
  public List<MqttMessage> request(final ChannelHandlerContext ctx, final MqttMessage mqttMessage) {
    this.LOG.info("执行了MQTT Connect 命令");
    MqttConnectMessage connectMessage = ((MqttConnectMessage) mqttMessage);
    int mqttVersion = connectMessage.variableHeader().version();
    boolean cleanSession = connectMessage.variableHeader().isCleanSession();
    String clientId = connectMessage.payload().clientIdentifier();
    CommonUtils.setClientId(ctx.channel(), clientId);
    String userName = connectMessage.payload().userName();
    byte[] password = connectMessage.payload().passwordInBytes();
    boolean sessionPresent = false;
    this.LOG.info("clientId:{},cleanSession:{}", clientId, Boolean.valueOf(cleanSession));
    ConnackEntity entity = null;
    try {
      this.ctx = ctx;
      boolean _checkVersion = this.checkVersion(mqttVersion);
      boolean _not = (!_checkVersion);
      if (_not) {
        ConnackEntity _connackEntity = new ConnackEntity(MqttConnectReturnCode.CONNECTION_REFUSED_UNACCEPTABLE_PROTOCOL_VERSION, sessionPresent);
        entity = _connackEntity;
      } else {
        boolean _checkClientId = this.checkClientId(clientId);
        boolean _not_1 = (!_checkClientId);
        if (_not_1) {
          ConnackEntity _connackEntity_1 = new ConnackEntity(MqttConnectReturnCode.CONNECTION_REFUSED_IDENTIFIER_REJECTED, sessionPresent);
          entity = _connackEntity_1;
        } else {
          boolean _checkAuthorized = this.checkAuthorized(CommonUtils.getRemoteAddr(ctx.channel()), clientId);
          boolean _not_2 = (!_checkAuthorized);
          if (_not_2) {
            ConnackEntity _connackEntity_2 = new ConnackEntity(MqttConnectReturnCode.CONNECTION_REFUSED_NOT_AUTHORIZED, sessionPresent);
            entity = _connackEntity_2;
          } else {
            boolean _checkUserAuth = this.checkUserAuth(clientId, userName, password);
            boolean _not_3 = (!_checkUserAuth);
            if (_not_3) {
              ConnackEntity _connackEntity_3 = new ConnackEntity(MqttConnectReturnCode.CONNECTION_REFUSED_BAD_USER_NAME_OR_PASSWORD, sessionPresent);
              entity = _connackEntity_3;
            } else {
              int heartbeatSec = connectMessage.variableHeader().keepAliveTimeSeconds();
              boolean _keepAlive = this.keepAlive(clientId, ctx, heartbeatSec);
              boolean _not_4 = (!_keepAlive);
              if (_not_4) {
                String failure = String.format("set heartbeat failure clientId:%s,heartbeatSec:%d", clientId, Integer.valueOf(heartbeatSec));
                throw new Exception(failure);
              }
              sessionPresent = (this.createSession(clientId, ctx, cleanSession)).booleanValue();
              this.storeWill(connectMessage, clientId);
              ConnackEntity _connackEntity_4 = new ConnackEntity(MqttConnectReturnCode.CONNECTION_ACCEPTED, sessionPresent);
              entity = _connackEntity_4;
            }
          }
        }
      }
    } catch (final Throwable _t) {
      if (_t instanceof Exception) {
        final Exception ex = (Exception)_t;
        this.LOG.info("Service Unavailable:", ex);
        ConnackEntity _connackEntity_5 = new ConnackEntity(MqttConnectReturnCode.CONNECTION_REFUSED_SERVER_UNAVAILABLE, sessionPresent);
        entity = _connackEntity_5;
      } else {
        throw Exceptions.sneakyThrow(_t);
      }
    }
    MqttMessage _response = new ConnackResponse().response(entity);
    return Collections.<MqttMessage>unmodifiableList(CollectionLiterals.<MqttMessage>newArrayList(_response));
  }
  
  protected String storeWill(final MqttConnectMessage connectMessage, final String clientId) {
    String _xifexpression = null;
    boolean _isWillFlag = connectMessage.variableHeader().isWillFlag();
    if (_isWillFlag) {
      String _xblockexpression = null;
      {
        this.LOG.info("保存Will消息 ： clientId:{}", clientId);
        WillEntity will = new WillEntity();
        will.setClientId(clientId);
        will.setRetain(connectMessage.variableHeader().isWillRetain());
        will.setQos(MqttQoS.valueOf(connectMessage.variableHeader().willQos()));
        will.setTopic(connectMessage.payload().willTopic());
        will.setWill(true);
        will.setPayload(connectMessage.payload().willMessageInBytes());
        _xblockexpression = this.cache.store(CommonUtils.will(clientId), will);
      }
      _xifexpression = _xblockexpression;
    }
    return _xifexpression;
  }
  
  protected Boolean createSession(final String clientId, final ChannelHandlerContext ctx, final boolean cleanSession) {
    ClientSessionEntity mqttSession = new ClientSessionEntity();
    boolean sessionPresent = false;
    ClientSessionEntity _clientSessionEntity = new ClientSessionEntity();
    mqttSession = _clientSessionEntity;
    mqttSession.setClientId(clientId);
    mqttSession.setCtx(ctx);
    mqttSession.setCleanStatus(cleanSession);
    if ((!cleanSession)) {
      sessionPresent = true;
    }
    this.cache.store(CommonUtils.session(clientId), mqttSession);
    return Boolean.valueOf(sessionPresent);
  }
}
