/**
 * Copyright (c) 2022 murenchao
 * taomu framework is licensed under Mulan PubL v2.
 * You can use this software according to the terms and conditions of the Mulan PubL v2.
 * You may obtain a copy of Mulan PubL v2 at:
 *       http://license.coscl.org.cn/MulanPubL-2.0
 * THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
 * EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
 * MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
 * See the Mulan PubL v2 for more details.
 */
package cool.taomu.framework.service.mqtt.broker.impl.request;

import com.google.gson.Gson;
import cool.taomu.framework.cache.KeyValueCache;
import cool.taomu.framework.configure.ConfigureManage;
import cool.taomu.framework.configure.entity.ClusterEntity;
import cool.taomu.framework.configure.entity.ConfigureEntity;
import cool.taomu.framework.service.mqtt.broker.entity.MessageEntity;
import cool.taomu.framework.service.mqtt.broker.entity.PubAckEntity;
import cool.taomu.framework.service.mqtt.broker.impl.PublishObservable;
import cool.taomu.framework.service.mqtt.broker.impl.response.PubAckResponse;
import cool.taomu.framework.service.mqtt.broker.inter.IPublishObserver;
import cool.taomu.framework.service.mqtt.broker.inter.IRequest;
import cool.taomu.framework.service.rpc.Gateway;
import cool.taomu.framework.service.rpc.Response;
import cool.taomu.framework.service.rpc.TRpcClient;
import cool.taomu.framework.service.utils.CommonUtils;
import cool.taomu.framework.utils.spi.Alias;
import cool.taomu.framework.utils.spi.ServiceLoader;
import io.netty.channel.ChannelHandlerContext;
import io.netty.handler.codec.mqtt.MqttMessage;
import io.netty.handler.codec.mqtt.MqttPublishMessage;
import io.netty.handler.codec.mqtt.MqttQoS;
import io.netty.util.ReferenceCountUtil;
import java.util.Collections;
import java.util.List;
import java.util.function.Consumer;
import org.apache.thrift.protocol.TProtocol;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * PUBLISH 	|	3 	|	两个方向都允许 	|	发布消息
 */
@Alias(value = "PUBLISH")
@SuppressWarnings("all")
public class PublishRequest implements IRequest {
  private final Logger LOG = LoggerFactory.getLogger(PublishRequest.class);
  
  private KeyValueCache cache = KeyValueCache.getInstance();
  
  private final ConfigureEntity config = ConfigureManage.loadConfig();
  
  @Override
  public List<MqttMessage> request(final ChannelHandlerContext ctx, final MqttMessage mqttMessage) {
    try {
      MqttPublishMessage publishMessage = ((MqttPublishMessage) mqttMessage);
      MessageEntity message = new MessageEntity();
      message.setSenderId(CommonUtils.getClientId(ctx.channel()));
      String _senderId = message.getSenderId();
      String _plus = ("执行了MQTT Publish 命令 : " + _senderId);
      this.LOG.info(_plus);
      message.setQos(publishMessage.fixedHeader().qosLevel());
      message.setTopic(publishMessage.variableHeader().topicName());
      message.setPayload(((MqttPublishMessage) mqttMessage).payload());
      message.setType(Integer.valueOf(mqttMessage.fixedHeader().messageType().value()));
      message.setDup(publishMessage.fixedHeader().isDup());
      message.setRetain(publishMessage.fixedHeader().isRetain());
      message.setMsgId(publishMessage.variableHeader().packetId());
      message.setSenderChannel(ctx.channel());
      MqttQoS _qos = message.getQos();
      if (_qos != null) {
        switch (_qos) {
          case EXACTLY_ONCE:
          case AT_MOST_ONCE:
            this.LOG.info(String.format("Qos0 message,clientId=%s", message.getSenderId()));
            this.retainMessage(message);
            break;
          case AT_LEAST_ONCE:
            this.LOG.info(String.format("Qos1 message,clientId=%s", message.getSenderId()));
            this.retainMessage(message);
            int messageId = message.getMsgId();
            PubAckEntity entity = new PubAckEntity(Integer.valueOf(messageId));
            MqttMessage _response = new PubAckResponse().response(entity);
            return Collections.<MqttMessage>unmodifiableList(CollectionLiterals.<MqttMessage>newArrayList(_response));
          default:
            this.LOG.info(String.format("Wrong mqtt message,clientId=%s", message.getSenderId()));
            break;
        }
      } else {
        this.LOG.info(String.format("Wrong mqtt message,clientId=%s", message.getSenderId()));
      }
    } catch (final Throwable _t) {
      if (_t instanceof Exception) {
        final Exception ex = (Exception)_t;
        this.LOG.debug("执行了MQTT Publish 命令出错了 : ", ex);
      } else {
        throw Exceptions.sneakyThrow(_t);
      }
    } finally {
      ReferenceCountUtil.release(mqttMessage.payload());
    }
    return null;
  }
  
  protected synchronized void retainMessage(final MessageEntity message) {
    byte[] _payload = message.getPayload();
    String _string = new String(_payload);
    this.LOG.debug(
      "clientId 为 {} 是否存在 Retain 数据 {}, 接受到的数据为 {} ", 
      message.getSenderId(), 
      Boolean.valueOf(message.isRetain()), _string);
    this.cache.store(message.getSenderId(), message);
    boolean _isRetain = message.isRetain();
    if (_isRetain) {
      MqttQoS qos = message.getQos();
      byte[] payload = message.getPayload();
      if ((((qos == MqttQoS.AT_MOST_ONCE) || (payload == null)) || (payload.length == 0))) {
        this.LOG.info("清空 clientId 为 {} 的Retain数据", message.getSenderId());
        this.cache.remove(CommonUtils.retain(message.getSenderId()));
      } else {
        this.LOG.info("保存 clientId 为 {} 的Retain数据", message.getSenderId());
        this.cache.store(CommonUtils.retain(message.getSenderId()), message);
        final Runnable _function = () -> {
          final Consumer<ClusterEntity> _function_1 = (ClusterEntity addr) -> {
            final Runnable _function_2 = () -> {
              String _xifexpression = null;
              boolean _equals = addr.getHostname().equals("");
              boolean _not = (!_equals);
              if (_not) {
                _xifexpression = addr.getHostname();
              } else {
                _xifexpression = "localhost";
              }
              TRpcClient.<Object>client(_xifexpression, (addr.getPort()).intValue(), 600, 
                new TRpcClient() {
                  @Override
                  public <T extends Object> void call(final TProtocol protocol) {
                    try {
                      ServiceLoader<Gateway.Iface> client = ServiceLoader.<Gateway.Iface>load(Gateway.Iface.class, protocol);
                      Response resp = client.get("Client").refresh(addr.getHostname());
                      PublishRequest.this.LOG.info(resp.toString());
                    } catch (Throwable _e) {
                      throw Exceptions.sneakyThrow(_e);
                    }
                  }
                });
            };
            CommonUtils.exec(_function_2);
          };
          IterableExtensions.<ClusterEntity>filterNull(this.config.getMqtt().getCluster()).forEach(_function_1);
        };
        CommonUtils.exec(_function);
      }
    }
    PublishObservable.getInstance().start(message.getSenderId(), IPublishObserver.Type.MESSAGE);
    final Runnable _function_1 = () -> {
      final Consumer<ClusterEntity> _function_2 = (ClusterEntity addr) -> {
        final Runnable _function_3 = () -> {
          final ConfigureEntity config = ConfigureManage.loadConfig();
          final Consumer<ClusterEntity> _function_4 = (ClusterEntity cluster) -> {
            String _xifexpression = null;
            boolean _equals = cluster.getHostname().equals("");
            boolean _not = (!_equals);
            if (_not) {
              _xifexpression = cluster.getHostname();
            } else {
              _xifexpression = "localhost";
            }
            TRpcClient.<Object>client(_xifexpression, (cluster.getPort()).intValue(), 
              600, new TRpcClient() {
              @Override
              public <T extends Object> void call(final TProtocol protocol) {
                try {
                  ServiceLoader<Gateway.Iface> client = ServiceLoader.<Gateway.Iface>load(Gateway.Iface.class, protocol);
                  Response resp = client.get("Client").publish(config.getMqtt().getHostname(), 
                    new Gson().toJson(message));
                  PublishRequest.this.LOG.info(resp.toString());
                } catch (Throwable _e) {
                  throw Exceptions.sneakyThrow(_e);
                }
              }
            });
          };
          config.getMqtt().getCluster().forEach(_function_4);
        };
        CommonUtils.exec(_function_3);
      };
      IterableExtensions.<ClusterEntity>filterNull(this.config.getMqtt().getCluster()).forEach(_function_2);
    };
    CommonUtils.exec(_function_1);
  }
}
