/**
 * Copyright (c) 2022 murenchao
 * taomu framework is licensed under Mulan PubL v2.
 * You can use this software according to the terms and conditions of the Mulan PubL v2.
 * You may obtain a copy of Mulan PubL v2 at:
 *       http://license.coscl.org.cn/MulanPubL-2.0
 * THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
 * EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
 * MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
 * See the Mulan PubL v2 for more details.
 */
package cool.taomu.framework.cache.jedis;

import cool.taomu.framework.configure.entity.ConfigureEntity;
import cool.taomu.framework.configure.entity.HostEntity;
import cool.taomu.framework.configure.entity.JeidsConfigEntity;
import cool.taomu.framework.crypto.Base64;
import cool.taomu.framework.inter.IConfigureManage;
import cool.taomu.framework.inter.cache.ICache;
import cool.taomu.framework.inter.cache.IQueueCache;
import cool.taomu.framework.inter.cache.ISetCache;
import cool.taomu.framework.spi.annotation.Spi;
import java.io.Serializable;
import java.util.HashSet;
import java.util.Set;
import java.util.function.Consumer;
import org.apache.commons.lang3.SerializationUtils;
import org.apache.commons.pool2.impl.GenericObjectPoolConfig;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.StringExtensions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import redis.clients.jedis.Connection;
import redis.clients.jedis.HostAndPort;
import redis.clients.jedis.Jedis;
import redis.clients.jedis.JedisCluster;
import redis.clients.jedis.JedisPool;
import redis.clients.jedis.commands.JedisCommands;

/**
 * 名词 加ed 变 形容词
 * 动词 加ed 变 过去式
 * 动词 加ing 变 正在进行时
 */
@Spi(value = "net_cache", singleton = true)
@SuppressWarnings("all")
public class JedisUtils implements AutoCloseable, ICache<String, Serializable>, ISetCache<String, Serializable>, IQueueCache<String, Serializable> {
  private static final Logger LOG = LoggerFactory.getLogger(JedisUtils.class);
  
  private JedisCluster cluster = null;
  
  private JedisPool pool = null;
  
  @Spi(value = "taomu-configure", singleton = true)
  private IConfigureManage<ConfigureEntity> config;
  
  private JedisPool initJedisPool() {
    JedisPool _xblockexpression = null;
    {
      ConfigureEntity config = this.config.loadConfig();
      JedisPool _xifexpression = null;
      JeidsConfigEntity _jedis = config.getJedis();
      boolean _tripleNotEquals = (_jedis != null);
      if (_tripleNotEquals) {
        JedisPool _xblockexpression_1 = null;
        {
          GenericObjectPoolConfig<Jedis> gopc = new GenericObjectPoolConfig<Jedis>();
          gopc.setMaxIdle((config.getJedis().getMaxIdle()).intValue());
          gopc.setMinIdle((config.getJedis().getMinIdle()).intValue());
          gopc.setMaxTotal((config.getJedis().getMaxTotal()).intValue());
          gopc.setTestOnBorrow(config.getJedis().isTestOnBorrow());
          gopc.setTestOnReturn(config.getJedis().isTestOnReturn());
          String password = config.getJedis().getPassword();
          String user = config.getJedis().getUsername();
          final HashSet<HostAndPort> hostAndPort = new HashSet<HostAndPort>();
          Integer timeout = config.getJedis().getTimeout();
          final Consumer<HostEntity> _function = (HostEntity host) -> {
            JedisUtils.LOG.info("创建Redis连接: {}:{}", host.getIp(), host.getPort());
            String _ip = host.getIp();
            Integer _port = host.getPort();
            HostAndPort _hostAndPort = new HostAndPort(_ip, (_port).intValue());
            hostAndPort.add(_hostAndPort);
          };
          config.getJedis().getCluster().forEach(_function);
          JedisPool _xifexpression_1 = null;
          if ((user == null)) {
            JedisPool _xifexpression_2 = null;
            if ((password == null)) {
              String _host = (((HostAndPort[])Conversions.unwrapArray(hostAndPort, HostAndPort.class))[0]).getHost();
              int _port = (((HostAndPort[])Conversions.unwrapArray(hostAndPort, HostAndPort.class))[0]).getPort();
              JedisPool _jedisPool = new JedisPool(gopc, _host, _port);
              _xifexpression_2 = this.pool = _jedisPool;
            } else {
              String _host_1 = (((HostAndPort[])Conversions.unwrapArray(hostAndPort, HostAndPort.class))[0]).getHost();
              int _port_1 = (((HostAndPort[])Conversions.unwrapArray(hostAndPort, HostAndPort.class))[0]).getPort();
              JedisPool _jedisPool_1 = new JedisPool(gopc, _host_1, _port_1, (timeout).intValue(), password);
              _xifexpression_2 = this.pool = _jedisPool_1;
            }
            _xifexpression_1 = _xifexpression_2;
          } else {
            JedisPool _xifexpression_3 = null;
            if (((user != null) && (password != null))) {
              String _host_2 = (((HostAndPort[])Conversions.unwrapArray(hostAndPort, HostAndPort.class))[0]).getHost();
              int _port_2 = (((HostAndPort[])Conversions.unwrapArray(hostAndPort, HostAndPort.class))[0]).getPort();
              JedisPool _jedisPool_2 = new JedisPool(gopc, _host_2, _port_2, (timeout).intValue(), user, password);
              _xifexpression_3 = this.pool = _jedisPool_2;
            }
            _xifexpression_1 = _xifexpression_3;
          }
          _xblockexpression_1 = _xifexpression_1;
        }
        _xifexpression = _xblockexpression_1;
      }
      _xblockexpression = _xifexpression;
    }
    return _xblockexpression;
  }
  
  private JedisPool initJedisCluster() {
    JedisPool _xblockexpression = null;
    {
      ConfigureEntity config = this.config.loadConfig();
      JedisPool _xifexpression = null;
      JeidsConfigEntity _jedis = config.getJedis();
      boolean _tripleNotEquals = (_jedis != null);
      if (_tripleNotEquals) {
        JedisPool _xtrycatchfinallyexpression = null;
        try {
          GenericObjectPoolConfig<Connection> gopc = new GenericObjectPoolConfig<Connection>();
          gopc.setMaxIdle((config.getJedis().getMaxIdle()).intValue());
          gopc.setMinIdle((config.getJedis().getMinIdle()).intValue());
          gopc.setMaxTotal((config.getJedis().getMaxTotal()).intValue());
          gopc.setTestOnBorrow(config.getJedis().isTestOnBorrow());
          gopc.setTestOnReturn(config.getJedis().isTestOnReturn());
          final HashSet<HostAndPort> hostAndPort = new HashSet<HostAndPort>();
          final Consumer<HostEntity> _function = (HostEntity host) -> {
            JedisUtils.LOG.info("创建Redis连接: {}:{}", host.getIp(), host.getPort());
            String _ip = host.getIp();
            Integer _port = host.getPort();
            HostAndPort _hostAndPort = new HostAndPort(_ip, (_port).intValue());
            hostAndPort.add(_hostAndPort);
          };
          config.getJedis().getCluster().forEach(_function);
          Integer timeout = config.getJedis().getTimeout();
          Integer maxAttempts = config.getJedis().getMaxAttempts();
          String password = config.getJedis().getPassword();
          String user = config.getJedis().getUsername();
          if ((user == null)) {
            if ((password == null)) {
              JedisCluster _jedisCluster = new JedisCluster(hostAndPort, (timeout).intValue(), (timeout).intValue(), (maxAttempts).intValue(), gopc);
              this.cluster = _jedisCluster;
            } else {
              JedisCluster _jedisCluster_1 = new JedisCluster(hostAndPort, (timeout).intValue(), (timeout).intValue(), (maxAttempts).intValue(), password, gopc);
              this.cluster = _jedisCluster_1;
            }
          } else {
            if (((user != null) && (password != null))) {
              JedisCluster _jedisCluster_2 = new JedisCluster(hostAndPort, (timeout).intValue(), (timeout).intValue(), (maxAttempts).intValue(), user, password, null, gopc);
              this.cluster = _jedisCluster_2;
            }
          }
          Runtime _runtime = Runtime.getRuntime();
          _runtime.addShutdownHook(new Thread() {
            @Override
            public void run() {
              if ((JedisUtils.this.cluster != null)) {
                JedisUtils.LOG.info("addShutdownHook 关闭了 Redis");
                JedisUtils.this.cluster.close();
              }
            }
          });
        } catch (final Throwable _t) {
          if (_t instanceof Exception) {
            final Exception e = (Exception)_t;
            JedisPool _xblockexpression_1 = null;
            {
              if ((this.cluster != null)) {
                JedisUtils.LOG.info("关闭了 Redis");
                this.cluster.close();
              }
              if ((this.pool != null)) {
                JedisUtils.LOG.info("关闭了 Redis");
                this.pool.close();
              }
              JedisUtils.LOG.error("创建Redis失败", e);
              this.cluster = null;
              _xblockexpression_1 = this.pool = null;
            }
            _xtrycatchfinallyexpression = _xblockexpression_1;
          } else {
            throw Exceptions.sneakyThrow(_t);
          }
        }
        _xifexpression = _xtrycatchfinallyexpression;
      }
      _xblockexpression = _xifexpression;
    }
    return _xblockexpression;
  }
  
  public JedisCommands getJedis() {
    try {
      if ((((this.config.loadConfig().getJedis() != null) && (this.pool == null)) && (this.cluster == null))) {
        JedisUtils.LOG.info("Jedis 链接初始化");
        boolean _isClustered = this.config.loadConfig().getJedis().isClustered();
        if (_isClustered) {
          this.initJedisCluster();
        } else {
          this.initJedisPool();
        }
      }
      if ((this.cluster != null)) {
        return this.cluster;
      } else {
        if ((this.pool != null)) {
          return this.pool.borrowObject();
        }
      }
      throw new NullPointerException("jedis connection error");
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  @Override
  public void close() throws Exception {
    if ((this.pool != null)) {
      this.pool.close();
    }
  }
  
  @Override
  public void clear() {
    throw new UnsupportedOperationException("TODO: auto-generated method stub");
  }
  
  @Override
  public void put(final String key, final Serializable value) {
    try {
      JedisUtils.LOG.debug("向Redis中添加{}数据", key);
      byte[] ser = SerializationUtils.serialize(value);
      JedisCommands _jedis = this.getJedis();
      String _string = new String(ser, "UTF-8");
      _jedis.set(key, _string);
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  @Override
  public Serializable get(final String key) {
    JedisUtils.LOG.debug("从Redis中获取{}数据", key);
    String result = this.getJedis().get(key);
    boolean _isNullOrEmpty = StringExtensions.isNullOrEmpty(result);
    boolean _not = (!_isNullOrEmpty);
    if (_not) {
      byte[] _bytes = result.getBytes();
      byte[] base64 = new Base64(_bytes).decode();
      final byte[] _converted_base64 = (byte[])base64;
      boolean _isNullOrEmpty_1 = IterableExtensions.isNullOrEmpty(((Iterable<?>)Conversions.doWrapArray(_converted_base64)));
      boolean _not_1 = (!_isNullOrEmpty_1);
      if (_not_1) {
        return SerializationUtils.<Serializable>deserialize(base64);
      }
    }
    return null;
  }
  
  @Override
  public void remove(final String key) {
    JedisUtils.LOG.debug("从Redis中移出{}数据", key);
    this.getJedis().del(key);
  }
  
  @Override
  public Serializable qpop(final String key) {
    String result = this.getJedis().rpop(key);
    boolean _isNullOrEmpty = StringExtensions.isNullOrEmpty(result);
    boolean _not = (!_isNullOrEmpty);
    if (_not) {
      byte[] _bytes = result.getBytes();
      byte[] base64 = new Base64(_bytes).decode();
      final byte[] _converted_base64 = (byte[])base64;
      boolean _isNullOrEmpty_1 = IterableExtensions.isNullOrEmpty(((Iterable<?>)Conversions.doWrapArray(_converted_base64)));
      boolean _not_1 = (!_isNullOrEmpty_1);
      if (_not_1) {
        return SerializationUtils.<Serializable>deserialize(base64);
      }
    }
    return null;
  }
  
  @Override
  public void qpush(final String key, final Serializable value) {
    try {
      byte[] ser = SerializationUtils.serialize(value);
      JedisCommands _jedis = this.getJedis();
      byte[] _encode = new Base64(ser).encode();
      String _string = new String(_encode, "UTF-8");
      _jedis.lpush(key, _string);
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  @Override
  public void sadd(final String key, final Serializable value) {
    try {
      byte[] ser = SerializationUtils.serialize(value);
      JedisCommands _jedis = this.getJedis();
      byte[] _encode = new Base64(ser).encode();
      String _string = new String(_encode, "UTF-8");
      _jedis.sadd(key, _string);
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  @Override
  public Set<Serializable> sget(final String key) {
    Set<String> results = this.getJedis().smembers(key);
    boolean _isNullOrEmpty = IterableExtensions.isNullOrEmpty(results);
    boolean _not = (!_isNullOrEmpty);
    if (_not) {
      HashSet<Serializable> hashSet = new HashSet<Serializable>();
      for (final String result : results) {
        {
          byte[] _bytes = result.getBytes();
          byte[] base64 = new Base64(_bytes).decode();
          final byte[] _converted_base64 = (byte[])base64;
          boolean _isNullOrEmpty_1 = IterableExtensions.isNullOrEmpty(((Iterable<?>)Conversions.doWrapArray(_converted_base64)));
          boolean _not_1 = (!_isNullOrEmpty_1);
          if (_not_1) {
            hashSet.add(SerializationUtils.<Serializable>deserialize(base64));
          }
        }
      }
      return hashSet;
    }
    return null;
  }
  
  @Override
  public void sremove(final String key, final Serializable value) {
    try {
      byte[] ser = SerializationUtils.serialize(value);
      JedisCommands _jedis = this.getJedis();
      byte[] _encode = new Base64(ser).encode();
      String _string = new String(_encode, "UTF-8");
      _jedis.srem(key, _string);
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
}
