/**
 * Copyright (c) 2022 murenchao
 * taomu framework is licensed under Mulan PubL v2.
 * You can use this software according to the terms and conditions of the Mulan PubL v2.
 * You may obtain a copy of Mulan PubL v2 at:
 *       http://license.coscl.org.cn/MulanPubL-2.0
 * THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
 * EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
 * MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
 * See the Mulan PubL v2 for more details.
 */
package cool.taomu.framework.script;

import cool.taomu.framework.inter.IScript;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.InputStream;
import java.io.Serializable;
import java.util.ArrayList;
import org.apache.commons.io.IOUtils;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.Functions.Function0;
import org.jruby.Ruby;
import org.jruby.RubyRuntimeAdapter;
import org.jruby.javasupport.JavaEmbedUtils;
import org.jruby.runtime.builtin.IRubyObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@SuppressWarnings("all")
public class JRubyScript implements IScript {
  private static final Logger LOG = LoggerFactory.getLogger(JRubyScript.class);
  
  private Ruby runtime;
  
  private IRubyObject rubyObject;
  
  public JRubyScript() {
    ArrayList<String> loadPaths = new ArrayList<String>();
    loadPaths.add("/root/.jruby/lib/ruby/1.8");
    loadPaths.add("/root/.jruby/lib/ruby/site_ruby/1.8");
    this.runtime = JavaEmbedUtils.initialize(loadPaths);
  }
  
  public JRubyScript(final String path) {
    this();
    try {
      JRubyScript.LOG.info(String.format("加载脚本:%s", path));
      try (InputStream scriptInput = Thread.currentThread().getContextClassLoader().getResourceAsStream(path)) {
        if ((scriptInput == null)) {
          try (FileInputStream scriptInput2 = new Function0<FileInputStream>() {
            @Override
            public FileInputStream apply() {
              try {
                File _file = new File(path);
                return new FileInputStream(_file);
              } catch (Throwable _e) {
                throw Exceptions.sneakyThrow(_e);
              }
            }
          }.apply()) {
            this.loaderScript(IOUtils.toString(scriptInput2, "UTF-8"));
          } catch (final Throwable _t) {
            if (_t instanceof FileNotFoundException) {
              final FileNotFoundException ex = (FileNotFoundException)_t;
              JRubyScript.LOG.info("", ex);
            } else {
              throw Exceptions.sneakyThrow(_t);
            }
          }
        } else {
          this.loaderScript(IOUtils.toString(scriptInput, "UTF-8"));
        }
      }
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  @Override
  public <K extends Object, T extends Serializable> K invoke(final String name, final T value) {
    Object result = JavaEmbedUtils.invokeMethod(this.runtime, this.rubyObject, name, ((Object[])Conversions.unwrapArray(CollectionLiterals.<Object>newArrayList(value), Object.class)), Object.class);
    K _xifexpression = null;
    if ((result != null)) {
      _xifexpression = ((K) result);
    }
    return _xifexpression;
  }
  
  @Override
  public void close() throws Exception {
    JRubyScript.LOG.debug("input stream close");
  }
  
  @Override
  public void loaderScript(final String coding) {
    RubyRuntimeAdapter evaler = JavaEmbedUtils.newRuntimeAdapter();
    this.rubyObject = evaler.eval(this.runtime, coding);
  }
}
