/**
 * Copyright (c) 2022 murenchao
 * taomu framework is licensed under Mulan PubL v2.
 * You can use this software according to the terms and conditions of the Mulan PubL v2.
 * You may obtain a copy of Mulan PubL v2 at:
 *       http://license.coscl.org.cn/MulanPubL-2.0
 * THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
 * EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
 * MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
 * See the Mulan PubL v2 for more details.
 */
package cool.taomu.framework.service.mqtt.broker.impl;

import cool.taomu.framework.inter.IObservable;
import cool.taomu.framework.inter.IObserver;
import cool.taomu.framework.inter.cache.ICache;
import cool.taomu.framework.service.mqtt.broker.Retain;
import cool.taomu.framework.service.mqtt.broker.entity.MessageEntity;
import cool.taomu.framework.service.mqtt.broker.entity.MqttChannelEntity;
import cool.taomu.framework.service.mqtt.broker.entity.MqttDataEntity;
import cool.taomu.framework.service.utils.CommonUtils;
import cool.taomu.framework.spi.annotation.Spi;
import io.netty.channel.ChannelHandlerContext;
import io.netty.handler.codec.mqtt.MqttFixedHeader;
import io.netty.handler.codec.mqtt.MqttMessage;
import io.netty.handler.codec.mqtt.MqttMessageIdVariableHeader;
import io.netty.handler.codec.mqtt.MqttMessageType;
import io.netty.handler.codec.mqtt.MqttPubAckMessage;
import io.netty.handler.codec.mqtt.MqttPublishMessage;
import io.netty.handler.codec.mqtt.MqttQoS;
import io.netty.util.ReferenceCountUtil;
import java.io.Serializable;
import java.util.Collections;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * PUBLISH 	|	3 	|	两个方向都允许 	|	发布消息
 */
@SuppressWarnings("all")
public class PublishRequest implements IObserver {
  private final Logger LOG = LoggerFactory.getLogger(PublishRequest.class);
  
  @Spi(value = "kvCache", singleton = true)
  private ICache<String, Serializable> cache;
  
  @Spi(value = "mqtt_pub_observable", singleton = true)
  private IObservable<IObserver> observable;
  
  @Spi(value = "mqtt_retain_observable", singleton = true)
  private IObservable<Retain> retainObservable;
  
  public Object request(final ChannelHandlerContext ctx, final MqttMessage mqttMessage) {
    try {
      if ((!(mqttMessage instanceof MqttPublishMessage))) {
        return null;
      }
      MqttPublishMessage publishMessage = ((MqttPublishMessage) mqttMessage);
      MessageEntity message = new MessageEntity();
      message.setSenderId(CommonUtils.getClientId(ctx.channel()));
      String _senderId = message.getSenderId();
      String _plus = ("执行了MQTT Publish 命令 : " + _senderId);
      this.LOG.info(_plus);
      MqttQoS qos = publishMessage.fixedHeader().qosLevel();
      message.setQos(qos.ordinal());
      message.setTopic(publishMessage.variableHeader().topicName());
      message.setPayload(((MqttPublishMessage) mqttMessage).payload());
      message.setType(Integer.valueOf(mqttMessage.fixedHeader().messageType().value()));
      message.setDup(publishMessage.fixedHeader().isDup());
      message.setRetain(publishMessage.fixedHeader().isRetain());
      message.setMsgId(publishMessage.variableHeader().packetId());
      message.setSenderChannel(ctx.channel());
      if (qos != null) {
        switch (qos) {
          case EXACTLY_ONCE:
          case AT_MOST_ONCE:
            this.LOG.info(String.format("Qos0 message,clientId=%s", message.getSenderId()));
            this.retainMessage(message);
            break;
          case AT_LEAST_ONCE:
            this.LOG.info(String.format("Qos1 message,clientId=%s", message.getSenderId()));
            this.retainMessage(message);
            MqttFixedHeader header = new MqttFixedHeader(MqttMessageType.PUBACK, false, MqttQoS.AT_MOST_ONCE, false, 0);
            MqttMessageIdVariableHeader varHeader = MqttMessageIdVariableHeader.from(message.getMsgId());
            MqttPubAckMessage _mqttPubAckMessage = new MqttPubAckMessage(header, varHeader);
            ctx.writeAndFlush(_mqttPubAckMessage);
            break;
          default:
            this.LOG.info(String.format("Wrong mqtt message,clientId=%s", message.getSenderId()));
            break;
        }
      } else {
        this.LOG.info(String.format("Wrong mqtt message,clientId=%s", message.getSenderId()));
      }
    } catch (final Throwable _t) {
      if (_t instanceof Exception) {
        final Exception ex = (Exception)_t;
        this.LOG.debug("执行了MQTT Publish 命令出错了 : ", ex);
      } else {
        throw Exceptions.sneakyThrow(_t);
      }
    } finally {
      ReferenceCountUtil.release(mqttMessage.payload());
    }
    return null;
  }
  
  protected synchronized void retainMessage(final MessageEntity message) {
    byte[] _payload = message.getPayload();
    String _string = new String(_payload);
    this.LOG.debug(
      "clientId 为 {} 是否存在 Retain 数据 {}, 接受到的数据为 {} ", 
      message.getSenderId(), 
      Boolean.valueOf(message.isRetain()), _string);
    this.cache.put(message.getSenderId(), message);
    boolean _isRetain = message.isRetain();
    if (_isRetain) {
      int qos = message.getQos();
      byte[] payload = message.getPayload();
      if ((((qos == MqttQoS.AT_MOST_ONCE.ordinal()) || (payload == null)) || (payload.length == 0))) {
        this.LOG.info("清空 clientId 为 {} 的Retain数据", message.getSenderId());
        String _senderId = message.getSenderId();
        String _topic = message.getTopic();
        this.retainObservable.unregister(IterableExtensions.join(Collections.<String>unmodifiableList(CollectionLiterals.<String>newArrayList(_senderId, _topic)), "-"));
      } else {
        this.LOG.info("保存 clientId 为 {} 的Retain数据", message.getSenderId());
        String _senderId_1 = message.getSenderId();
        String _topic_1 = message.getTopic();
        String _join = IterableExtensions.join(Collections.<String>unmodifiableList(CollectionLiterals.<String>newArrayList(_senderId_1, _topic_1)), "-");
        Retain _retain = new Retain(message);
        this.retainObservable.register(_join, _retain);
      }
    }
    this.observable.publish(message);
  }
  
  @Override
  public void publish(final IObservable<?> o, final Object arg) {
    if ((arg instanceof MqttDataEntity)) {
      boolean _equals = ((MqttDataEntity)arg).getDataType().equals(MqttDataEntity.Type.PUBLISH);
      if (_equals) {
        Object _data = ((MqttDataEntity)arg).getData();
        MqttChannelEntity mce = ((MqttChannelEntity) _data);
        this.request(mce.getCtx(), mce.getMessage());
      }
    }
  }
}
