/**
 * Copyright (c) 2022 murenchao
 * taomu framework is licensed under Mulan PubL v2.
 * You can use this software according to the terms and conditions of the Mulan PubL v2.
 * You may obtain a copy of Mulan PubL v2 at:
 *       http://license.coscl.org.cn/MulanPubL-2.0
 * THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
 * EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
 * MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
 * See the Mulan PubL v2 for more details.
 */
package cool.taomu.framework.service.mqtt.broker.impl;

import cool.taomu.framework.inter.IBeanContainer;
import cool.taomu.framework.inter.IObservable;
import cool.taomu.framework.inter.IObserver;
import cool.taomu.framework.service.mqtt.broker.Publish;
import cool.taomu.framework.service.mqtt.broker.Retain;
import cool.taomu.framework.service.mqtt.broker.entity.MqttChannelEntity;
import cool.taomu.framework.service.mqtt.broker.entity.MqttDataEntity;
import cool.taomu.framework.service.mqtt.broker.entity.TopicEntity;
import cool.taomu.framework.service.utils.CommonUtils;
import cool.taomu.framework.spi.annotation.Spi;
import io.netty.channel.ChannelHandlerContext;
import io.netty.handler.codec.mqtt.MqttFixedHeader;
import io.netty.handler.codec.mqtt.MqttMessage;
import io.netty.handler.codec.mqtt.MqttMessageIdVariableHeader;
import io.netty.handler.codec.mqtt.MqttMessageType;
import io.netty.handler.codec.mqtt.MqttQoS;
import io.netty.handler.codec.mqtt.MqttSubAckMessage;
import io.netty.handler.codec.mqtt.MqttSubAckPayload;
import io.netty.handler.codec.mqtt.MqttSubscribeMessage;
import io.netty.handler.codec.mqtt.MqttTopicSubscription;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import org.apache.oro.text.perl.Perl5Util;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * SUBSCRIBE V   |   8   |   客户端到服务端     |   客户端订阅请求
 */
@SuppressWarnings("all")
public class SubscribeRequest implements IObserver {
  private final Logger LOG = LoggerFactory.getLogger(SubscribeRequest.class);
  
  @Spi(value = "mqtt_pub_observable", singleton = true)
  private IObservable<IObserver> observable;
  
  @Spi(value = "TmBeanContainer", singleton = true)
  private IBeanContainer beanContainer;
  
  @Spi(value = "mqtt_retain_observable", singleton = true)
  private IObservable<Retain> retainObservable;
  
  public Object request(final ChannelHandlerContext ctx, final MqttMessage mqttMessage) {
    try {
      if ((!(mqttMessage instanceof MqttSubscribeMessage))) {
        return null;
      }
      String clientId = CommonUtils.getClientId(ctx.channel());
      if ((clientId == null)) {
        Thread.sleep(100);
        clientId = CommonUtils.getClientId(ctx.channel());
      }
      this.LOG.info(("执行了MQTT Subscribe 命令 : " + clientId));
      try {
        MqttSubscribeMessage subscribeMessage = ((MqttSubscribeMessage) mqttMessage);
        int messageId = subscribeMessage.variableHeader().messageId();
        ArrayList<TopicEntity> validTopicList = this.registerTopics(ctx, subscribeMessage.payload().topicSubscriptions());
        synchronized (validTopicList) {
          if (((validTopicList == null) || (validTopicList.size() == 0))) {
            this.LOG.info(String.format("Valid all subscribe topic failure,messageId:%s", Integer.valueOf(messageId)));
            return null;
          }
          MqttFixedHeader header = new MqttFixedHeader(MqttMessageType.SUBACK, false, MqttQoS.AT_MOST_ONCE, false, 0);
          MqttMessageIdVariableHeader varHeader = MqttMessageIdVariableHeader.from(messageId);
          List<Integer> _topicQos = this.getTopicQos(validTopicList);
          MqttSubAckPayload _mqttSubAckPayload = new MqttSubAckPayload(_topicQos);
          MqttSubAckMessage _mqttSubAckMessage = new MqttSubAckMessage(header, varHeader, _mqttSubAckPayload);
          ctx.writeAndFlush(_mqttSubAckMessage);
        }
      } catch (final Throwable _t) {
        if (_t instanceof Exception) {
          final Exception ex = (Exception)_t;
          this.LOG.debug("subscribe requst exception:", ex);
        } else {
          throw Exceptions.sneakyThrow(_t);
        }
      }
      return null;
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  public List<Integer> getTopicQos(final List<TopicEntity> topics) {
    int _size = topics.size();
    ArrayList<Integer> qoss = new ArrayList<Integer>(_size);
    for (final TopicEntity topic : topics) {
      qoss.add(Integer.valueOf(topic.getQos()));
    }
    return qoss;
  }
  
  public synchronized ArrayList<TopicEntity> registerTopics(final ChannelHandlerContext ctx, final List<MqttTopicSubscription> topics) {
    synchronized (ctx) {
      String clientId = CommonUtils.getClientId(ctx.channel());
      ArrayList<TopicEntity> topicList = new ArrayList<TopicEntity>();
      for (final MqttTopicSubscription subscription : topics) {
        {
          String _picName = subscription.topicName();
          int _value = subscription.qualityOfService().value();
          final TopicEntity topic = new TopicEntity(_picName, _value);
          topic.setClientId(clientId);
          this.LOG.info("订阅Topic {} 的用户{}", topic.getName(), clientId);
          Perl5Util p5 = new Perl5Util();
          boolean _match = p5.match("/^[A-Za-z0-9]+([\\/A-Za-z0-9_]*|\\/\\+||\\/\\#)$/", topic.getName());
          if (_match) {
            Object _bean = this.beanContainer.getBean(Publish.class.getName(), topic);
            IObserver instance = ((IObserver) _bean);
            String _name = topic.getName();
            this.observable.register(IterableExtensions.join(Collections.<String>unmodifiableList(CollectionLiterals.<String>newArrayList(clientId, _name)), "#"), instance);
            this.retainObservable.publish(topic);
          }
          topicList.add(topic);
        }
      }
      return topicList;
    }
  }
  
  @Override
  public void publish(final IObservable<?> o, final Object arg) {
    if ((arg instanceof MqttDataEntity)) {
      boolean _equals = ((MqttDataEntity)arg).getDataType().equals(MqttDataEntity.Type.SUBSCRIBE);
      if (_equals) {
        Object _data = ((MqttDataEntity)arg).getData();
        MqttChannelEntity mce = ((MqttChannelEntity) _data);
        this.request(mce.getCtx(), mce.getMessage());
      }
    }
  }
}
