/**
 * Copyright (c) 2022 murenchao
 * taomu framework is licensed under Mulan PubL v2.
 * You can use this software according to the terms and conditions of the Mulan PubL v2.
 * You may obtain a copy of Mulan PubL v2 at:
 *       http://license.coscl.org.cn/MulanPubL-2.0
 * THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
 * EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
 * MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
 * See the Mulan PubL v2 for more details.
 */
package cool.taomu.framework.objectpool;

import cool.taomu.framework.configure.entity.ConfigureEntity;
import cool.taomu.framework.inter.IConfigureManage;
import cool.taomu.framework.utils.reflect.ReflectUtils;
import java.lang.annotation.Annotation;
import java.lang.invoke.MethodHandle;
import java.lang.invoke.MethodHandles;
import java.lang.invoke.MethodType;
import java.lang.reflect.Field;
import java.util.concurrent.atomic.AtomicInteger;
import org.apache.commons.pool2.BaseKeyedPooledObjectFactory;
import org.apache.commons.pool2.PooledObject;
import org.apache.commons.pool2.impl.DefaultPooledObject;
import org.apache.commons.pool2.impl.GenericKeyedObjectPool;
import org.apache.commons.pool2.impl.GenericKeyedObjectPoolConfig;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@SuppressWarnings("all")
public class ObjectPoolFactory<T extends Object> extends BaseKeyedPooledObjectFactory<Class<T>, ObjectPool<T>> {
  private AtomicInteger idCount = new AtomicInteger(1);
  
  private static final Logger LOG = LoggerFactory.getLogger(ObjectPoolFactory.class);
  
  private GenericKeyedObjectPool<Class<T>, ObjectPool<T>> pool = null;
  
  private final String SPRING_AUTOWIRED = "org.springframework.beans.factory.annotation.Autowired";
  
  public static <T extends Object> GenericKeyedObjectPool<Class<T>, ObjectPool<T>> getPoolInstance(final IConfigureManage<ConfigureEntity> iconfig) {
    GenericKeyedObjectPoolConfig<ObjectPool<T>> config = new GenericKeyedObjectPoolConfig<ObjectPool<T>>();
    ConfigureEntity yml = iconfig.loadConfig();
    config.setMaxTotalPerKey((yml.getPool().getMaxTotalPerKey()).intValue());
    config.setMaxIdlePerKey((yml.getPool().getMaxIdlePerKey()).intValue());
    config.setMinIdlePerKey((yml.getPool().getMinIdlePerKey()).intValue());
    config.setLifo(yml.getPool().isLifo());
    ObjectPoolFactory<T> opf = new ObjectPoolFactory<T>();
    GenericKeyedObjectPool<Class<T>, ObjectPool<T>> res = new GenericKeyedObjectPool<Class<T>, ObjectPool<T>>(opf, config);
    opf.pool = res;
    return res;
  }
  
  @Override
  public PooledObject<ObjectPool<T>> makeObject(final Class<T> key) throws Exception {
    ObjectPoolFactory.LOG.info(("创建对象:" + key));
    return super.makeObject(key);
  }
  
  @Override
  public void activateObject(final Class<T> key, final PooledObject<ObjectPool<T>> p) throws Exception {
    String _name = key.getName();
    String _plus = ("活动对象：" + _name);
    String _plus_1 = (_plus + " ==> ");
    int _id = p.getObject().getId();
    String _plus_2 = (_plus_1 + Integer.valueOf(_id));
    ObjectPoolFactory.LOG.info(_plus_2);
    super.activateObject(key, p);
  }
  
  @Override
  public void destroyObject(final Class<T> key, final PooledObject<ObjectPool<T>> p) throws Exception {
    int _id = p.getObject().getId();
    String _plus = ((("销毁对象:" + key) + " ==> ") + Integer.valueOf(_id));
    ObjectPoolFactory.LOG.info(_plus);
    super.destroyObject(key, p);
  }
  
  @Override
  public void passivateObject(final Class<T> key, final PooledObject<ObjectPool<T>> p) throws Exception {
    int _id = p.getObject().getId();
    String _plus = ((("归还对象:" + key) + " ==> ") + Integer.valueOf(_id));
    ObjectPoolFactory.LOG.info(_plus);
    super.passivateObject(key, p);
  }
  
  @Override
  public PooledObject<ObjectPool<T>> wrap(final ObjectPool<T> value) {
    return new DefaultPooledObject<ObjectPool<T>>(value);
  }
  
  @Override
  public ObjectPool<T> create(final Class<T> zlass) throws Exception {
    T instance = zlass.newInstance();
    Field[] fields = ReflectUtils.getReflectFields(zlass);
    for (final Field field : fields) {
      {
        ClassLoader loader = Thread.currentThread().getContextClassLoader();
        try {
          Class<?> _loadClass = loader.loadClass(this.SPRING_AUTOWIRED);
          Annotation ann = field.<Annotation>getAnnotation(((Class<Annotation>) _loadClass));
          if ((ann != null)) {
            ReflectUtils.setReflectFields(zlass, field.getName(), instance, this.getSpringBean(field.getType()));
          }
        } catch (final Throwable _t) {
          if (_t instanceof ClassNotFoundException) {
            final ClassNotFoundException ex = (ClassNotFoundException)_t;
            ObjectPoolFactory.LOG.debug("Class Not Found Exception", ex);
          } else {
            throw Exceptions.sneakyThrow(_t);
          }
        }
      }
    }
    int _andAdd = this.idCount.getAndAdd(1);
    return new ObjectPool<T>(this.pool, instance, _andAdd);
  }
  
  public Object getSpringBean(final Class<?> zlass) {
    try {
      ClassLoader loader = Thread.currentThread().getContextClassLoader();
      Class<?> registerBeanToSpring = loader.loadClass("cool.taomu.framework.springplugin.TaomuFramework");
      if ((registerBeanToSpring != null)) {
        MethodType desc = MethodType.methodType(Object.class, Class.class);
        MethodHandle methodHandle = MethodHandles.lookup().findStatic(registerBeanToSpring, "getBean", desc);
        Object bean = methodHandle.invokeExact(zlass);
        if ((bean != null)) {
          ObjectPoolFactory.LOG.info("从spring中成功获取一个bean");
        } else {
          ObjectPoolFactory.LOG.info("从spring中未获取到一个bean");
        }
        return bean;
      }
    } catch (final Throwable _t) {
      if (_t instanceof Throwable) {
        final Throwable e = (Throwable)_t;
        ObjectPoolFactory.LOG.debug("", e);
      } else {
        throw Exceptions.sneakyThrow(_t);
      }
    }
    return null;
  }
}
