/**
 * Copyright (c) 2022 murenchao
 * taomu framework is licensed under Mulan PubL v2.
 * You can use this software according to the terms and conditions of the Mulan PubL v2.
 * You may obtain a copy of Mulan PubL v2 at:
 *       http://license.coscl.org.cn/MulanPubL-2.0
 * THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
 * EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
 * MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
 * See the Mulan PubL v2 for more details.
 */
package cool.taomu.framework.utils.asm;

import com.google.common.base.Objects;
import cool.taomu.framework.configure.Constant;
import java.io.PrintWriter;
import java.lang.invoke.CallSite;
import java.lang.invoke.MethodHandles;
import java.lang.invoke.MethodType;
import java.lang.reflect.Constructor;
import java.lang.reflect.Parameter;
import java.util.Collections;
import java.util.List;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.Consumer;
import org.eclipse.xtend.lib.annotations.Accessors;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.Pure;
import org.eclipse.xtext.xbase.lib.StringExtensions;
import org.objectweb.asm.ClassVisitor;
import org.objectweb.asm.ClassWriter;
import org.objectweb.asm.Handle;
import org.objectweb.asm.Label;
import org.objectweb.asm.Opcodes;
import org.objectweb.asm.Type;
import org.objectweb.asm.commons.GeneratorAdapter;
import org.objectweb.asm.commons.Method;
import org.objectweb.asm.util.TraceClassVisitor;

@SuppressWarnings("all")
public final class BytecodeUtils extends ClassLoader {
  @Accessors
  public static final class Variable {
    private Integer index;
    
    private Type type;
    
    public Variable(final Integer index, final Type type) {
      this.index = index;
      this.type = type;
    }
    
    @Pure
    public Integer getIndex() {
      return this.index;
    }
    
    public void setIndex(final Integer index) {
      this.index = index;
    }
    
    @Pure
    public Type getType() {
      return this.type;
    }
    
    public void setType(final Type type) {
      this.type = type;
    }
  }
  
  /**
   * 定义一个接口用来提供除构造函数外的字节码操作
   */
  public interface Asm<T extends Object> {
    void coding(final T code);
  }
  
  private ClassWriter cwriter;
  
  private ClassVisitor tcvisitor;
  
  private GeneratorAdapter ga;
  
  private final String name;
  
  private String superClassName;
  
  private int version;
  
  private final ConcurrentHashMap<String, BytecodeUtils.Variable> localVariable = new ConcurrentHashMap<String, BytecodeUtils.Variable>();
  
  private final ConcurrentHashMap<String, Label> localLabel = new ConcurrentHashMap<String, Label>();
  
  public BytecodeUtils() {
    this(
      String.format("%s.%s%s", Constant.DEFAULT_PACKAGE, Constant.DEFAULT_ClASS_PREFIX, 
        UUID.randomUUID().toString().replace("-", "")).replace(".", "/"));
  }
  
  public BytecodeUtils(final String name) {
    this(Opcodes.V1_8, name);
  }
  
  public BytecodeUtils(final int version, final String name) {
    ClassWriter _classWriter = new ClassWriter((ClassWriter.COMPUTE_MAXS + ClassWriter.COMPUTE_FRAMES));
    this.cwriter = _classWriter;
    PrintWriter _printWriter = new PrintWriter(System.err);
    TraceClassVisitor _traceClassVisitor = new TraceClassVisitor(this.cwriter, _printWriter);
    this.tcvisitor = _traceClassVisitor;
    this.name = name;
    this.version = version;
  }
  
  public BytecodeUtils createClass(final int access, final Class<?> superClass, final String... inters) {
    this.superClassName = superClass.getTypeName();
    this.tcvisitor.visit(this.version, access, this.name, null, this.superClassName, inters);
    return this;
  }
  
  public BytecodeUtils createClass(final Class<?> superClass, final String... inters) {
    return this.createClass(Opcodes.ACC_PUBLIC, superClass, inters);
  }
  
  public BytecodeUtils createClass(final String... inters) {
    return this.createClass(Object.class, inters);
  }
  
  public BytecodeUtils createClass() {
    return this.createClass(null);
  }
  
  public BytecodeUtils createInterface(final String... inters) {
    return this.createClass(((Opcodes.ACC_PUBLIC + Opcodes.ACC_ABSTRACT) + Opcodes.ACC_INTERFACE), Object.class, inters);
  }
  
  public BytecodeUtils createAbstract(final Class<?> superClass, final String... inters) {
    return this.createClass((Opcodes.ACC_PUBLIC + Opcodes.ACC_ABSTRACT), superClass, inters);
  }
  
  public BytecodeUtils createAbstract(final String... inters) {
    return this.createAbstract(Object.class, inters);
  }
  
  public BytecodeUtils createAbstract() {
    return this.createAbstract(null);
  }
  
  public BytecodeUtils init(final Constructor<?>[] constructor) {
    final Consumer<Constructor<?>> _function = (Constructor<?> it) -> {
      Method m = Method.getMethod(it);
      GeneratorAdapter ga = new GeneratorAdapter(Opcodes.ACC_PUBLIC, m, null, null, this.tcvisitor);
      ga.loadThis();
      int _size = ((List<Parameter>)Conversions.doWrapArray(it.getParameters())).size();
      boolean _greaterThan = (_size > 0);
      if (_greaterThan) {
        int _size_1 = ((List<Parameter>)Conversions.doWrapArray(it.getParameters())).size();
        int _minus = (_size_1 - 1);
        ga.loadArgs(0, _minus);
      }
      ga.invokeConstructor(Type.getObjectType(this.superClassName), m);
      ga.returnValue();
      ga.endMethod();
    };
    IterableExtensions.<Constructor<?>>filterNull(((Iterable<Constructor<?>>)Conversions.doWrapArray(constructor))).forEach(_function);
    return this;
  }
  
  public BytecodeUtils createField(final String name, final int access, final Class<?> type, final Object value) {
    this.tcvisitor.visitField(access, name, Type.getType(type).getDescriptor(), null, value).visitEnd();
    if (((access == Opcodes.ACC_PRIVATE) || (access == Opcodes.ACC_PROTECTED))) {
      String fName = StringExtensions.toFirstUpper(name);
      String _join = IterableExtensions.join(Collections.<String>unmodifiableList(CollectionLiterals.<String>newArrayList("set", fName)));
      String _name = type.getName();
      BytecodeUtils m = this.createMethod(IterableExtensions.join(Collections.<String>unmodifiableList(CollectionLiterals.<String>newArrayList("void", _join, "(", _name, ")")), " "), false);
      m.<GeneratorAdapter>coding(GeneratorAdapter.class, new BytecodeUtils.Asm<GeneratorAdapter>() {
        @Override
        public void coding(final GeneratorAdapter it) {
          it.loadThis();
          it.loadArg(0);
          it.putField(Type.getObjectType(BytecodeUtils.this.name), it.getName(), Type.getType(type));
        }
      }).returnValue().endMethod();
      String _name_1 = type.getName();
      String _join_1 = IterableExtensions.join(Collections.<String>unmodifiableList(CollectionLiterals.<String>newArrayList("get", fName)));
      m = this.createMethod(IterableExtensions.join(Collections.<String>unmodifiableList(CollectionLiterals.<String>newArrayList(_name_1, _join_1, "(", ")")), " "), false);
      m.<GeneratorAdapter>coding(GeneratorAdapter.class, new BytecodeUtils.Asm<GeneratorAdapter>() {
        @Override
        public void coding(final GeneratorAdapter it) {
          it.loadThis();
          it.getField(Type.getObjectType(BytecodeUtils.this.name), it.getName(), Type.getType(type));
        }
      }).returnValue().endMethod();
    }
    return this;
  }
  
  public BytecodeUtils createMethod(final String method, final boolean defaultPackage) {
    return this.createMethod(Opcodes.ACC_PUBLIC, method, defaultPackage);
  }
  
  public BytecodeUtils createMethod(final String method) {
    return this.createMethod(method, true);
  }
  
  public BytecodeUtils createMethod(final int access, final String method, final boolean defaultPackage) {
    Method _method = Method.getMethod(method, defaultPackage);
    GeneratorAdapter _generatorAdapter = new GeneratorAdapter(access, _method, null, null, this.tcvisitor);
    this.ga = _generatorAdapter;
    return this;
  }
  
  public BytecodeUtils createMethod(final java.lang.reflect.Method method) {
    return this.createMethod(Opcodes.ACC_PUBLIC, method);
  }
  
  public BytecodeUtils createMethod(final int access, final java.lang.reflect.Method method) {
    Method _method = Method.getMethod(method);
    GeneratorAdapter _generatorAdapter = new GeneratorAdapter(access, _method, null, null, this.tcvisitor);
    this.ga = _generatorAdapter;
    return this;
  }
  
  public BytecodeUtils invokeVirtual(final String internalName, final String method, final boolean defaultPackage) {
    this.ga.invokeVirtual(Type.getObjectType(internalName.replace(".", "/")), Method.getMethod(method, defaultPackage));
    return this;
  }
  
  public BytecodeUtils invokeVirtual(final String internalName, final String method) {
    return this.invokeVirtual(internalName, method, true);
  }
  
  public BytecodeUtils invokeVirtual(final Class<?> zlass, final java.lang.reflect.Method method) {
    return this.invokeVirtual(zlass.getName(), method);
  }
  
  public BytecodeUtils invokeVirtual(final Class<?> zlass, final String method) {
    return this.invokeVirtual(zlass.getName(), method);
  }
  
  public BytecodeUtils invokeVirtual(final String internalName, final java.lang.reflect.Method method) {
    this.ga.invokeVirtual(Type.getObjectType(internalName.replace(".", "/")), Method.getMethod(method));
    return this;
  }
  
  public BytecodeUtils invokeStatic(final Class<?> type, final String method) {
    return this.invokeStatic(type.getName(), method, true);
  }
  
  public BytecodeUtils invokeStatic(final String internalName, final String method, final boolean defaultPackage) {
    this.ga.invokeStatic(Type.getObjectType(internalName.replace(".", "/")), Method.getMethod(method, defaultPackage));
    return this;
  }
  
  public BytecodeUtils invokeDynamic(final int tag, final String internalName, final Method method) {
    MethodType mt = MethodType.methodType(CallSite.class, MethodHandles.Lookup.class, String.class, MethodType.class);
    String owner = Type.getObjectType(internalName.replace(".", "/")).getInternalName();
    String _methodDescriptorString = mt.toMethodDescriptorString();
    Handle bootstrap = new Handle(tag, owner, "bootstrap", _methodDescriptorString, false);
    this.ga.invokeDynamic(method.getName(), method.getDescriptor(), bootstrap);
    return this;
  }
  
  public BytecodeUtils invokeDynamic(final Class<?> zlass, final Method method) {
    return this.invokeDynamic(Opcodes.H_INVOKESTATIC, zlass.getName(), method);
  }
  
  public BytecodeUtils returnValue() {
    this.ga.returnValue();
    return this;
  }
  
  public BytecodeUtils load(final String key) {
    boolean _containsKey = this.localVariable.containsKey(key);
    if (_containsKey) {
      final BytecodeUtils.Variable local = this.localVariable.get(key);
      this.ga.loadLocal((local.index).intValue(), local.type);
    }
    return this;
  }
  
  public BytecodeUtils store(final String key, final Class<?> zlass) {
    int index = this.ga.newLocal(Type.getType(zlass));
    this.ga.storeLocal(index, Type.getType(zlass));
    Type _type = Type.getType(zlass);
    BytecodeUtils.Variable _variable = new BytecodeUtils.Variable(Integer.valueOf(index), _type);
    this.localVariable.put(key, _variable);
    return this;
  }
  
  public BytecodeUtils mark(final String key) {
    boolean _containsKey = this.localLabel.containsKey(key);
    if (_containsKey) {
      this.ga.mark(this.localLabel.get(key));
    } else {
      Label label = this.ga.newLabel();
      this.localLabel.put(key, label);
      this.ga.mark(label);
    }
    return this;
  }
  
  public BytecodeUtils goTo(final String key) {
    boolean _containsKey = this.localLabel.containsKey(key);
    if (_containsKey) {
      this.ga.goTo(this.localLabel.get(key));
    } else {
      Label label = this.ga.newLabel();
      this.localLabel.put(key, label);
      this.ga.goTo(label);
    }
    return this;
  }
  
  public BytecodeUtils ifNull(final String key) {
    boolean _containsKey = this.localLabel.containsKey(key);
    if (_containsKey) {
      this.ga.ifNull(this.localLabel.get(key));
    } else {
      Label label = this.ga.newLabel();
      this.localLabel.put(key, label);
      this.ga.ifNull(this.localLabel.get(key));
    }
    return this;
  }
  
  public BytecodeUtils ifNonNull(final String key) {
    boolean _containsKey = this.localLabel.containsKey(key);
    if (_containsKey) {
      this.ga.ifNonNull(this.localLabel.get(key));
    } else {
      Label label = this.ga.newLabel();
      this.localLabel.put(key, label);
      this.ga.ifNonNull(this.localLabel.get(key));
    }
    return this;
  }
  
  public BytecodeUtils ifCmp(final String key, final int mode, final Class<?> zlass) {
    boolean _containsKey = this.localLabel.containsKey(key);
    if (_containsKey) {
      this.ga.ifCmp(Type.getType(zlass), mode, this.localLabel.get(key));
    } else {
      Label label = this.ga.newLabel();
      this.localLabel.put(key, label);
      this.ga.ifCmp(Type.getType(zlass), mode, label);
    }
    return this;
  }
  
  public BytecodeUtils checkCast(final Class<?> type) {
    this.ga.checkCast(Type.getType(type));
    return this;
  }
  
  public BytecodeUtils dup() {
    this.ga.dup();
    return this;
  }
  
  public BytecodeUtils push(final int index) {
    this.ga.push(index);
    return this;
  }
  
  public BytecodeUtils push(final String value) {
    this.ga.push(value);
    return this;
  }
  
  public BytecodeUtils loadArg(final int index) {
    this.ga.loadArg(index);
    return this;
  }
  
  public <T extends Object> BytecodeUtils coding(final Class<T> zlass, final BytecodeUtils.Asm<T> code) {
    boolean _equals = zlass.equals(BytecodeUtils.class);
    if (_equals) {
      code.coding(zlass.cast(this));
    } else {
      boolean _equals_1 = zlass.equals(GeneratorAdapter.class);
      if (_equals_1) {
        code.coding(((T) this.ga));
      }
    }
    return this;
  }
  
  public BytecodeUtils endMethod() {
    this.ga.endMethod();
    return this;
  }
  
  public BytecodeUtils buildReturnValue(final java.lang.reflect.Method m) {
    Class<?> _returnType = m.getReturnType();
    boolean _matched = false;
    if (Objects.equal(_returnType, int.class)) {
      _matched=true;
      this.checkCast(Integer.class);
      this.invokeVirtual(Integer.class, "int intValue()");
    }
    if (!_matched) {
      if (Objects.equal(_returnType, long.class)) {
        _matched=true;
        this.checkCast(Long.class);
        this.invokeVirtual(Long.class, "long longValue()");
      }
    }
    if (!_matched) {
      if (Objects.equal(_returnType, double.class)) {
        _matched=true;
        this.checkCast(Double.class);
        this.invokeVirtual(Double.class, "double doubleValue()");
      }
    }
    if (!_matched) {
      if (Objects.equal(_returnType, float.class)) {
        _matched=true;
        this.checkCast(Float.class);
        this.invokeVirtual(Float.class, "float floatValue()");
      }
    }
    if (!_matched) {
      if (Objects.equal(_returnType, boolean.class)) {
        _matched=true;
        this.checkCast(Boolean.class);
        this.invokeVirtual(Boolean.class, "boolean booleanValue()");
      }
    }
    if (!_matched) {
      if (Objects.equal(_returnType, short.class)) {
        _matched=true;
        this.checkCast(Short.class);
        this.invokeVirtual(Short.class, "short shortValue()");
      }
    }
    if (!_matched) {
      if (Objects.equal(_returnType, char.class)) {
        _matched=true;
        this.checkCast(Character.class);
        this.invokeVirtual(Character.class, "char charValue()");
      }
    }
    if (!_matched) {
      if (Objects.equal(_returnType, byte.class)) {
        _matched=true;
        this.checkCast(Byte.class);
        this.invokeVirtual(Byte.class, "byte byteValue()");
      }
    }
    if (!_matched) {
      if (Objects.equal(_returnType, void.class)) {
        _matched=true;
      }
    }
    if (!_matched) {
      this.checkCast(m.getReturnType());
    }
    return this;
  }
  
  public BytecodeUtils buildArguments(final Method method) {
    int _size = ((List<Type>)Conversions.doWrapArray(method.getArgumentTypes())).size();
    boolean _greaterThan = (_size > 0);
    if (_greaterThan) {
      for (int index = 0; (index < ((List<Type>)Conversions.doWrapArray(method.getArgumentTypes())).size()); index++) {
        {
          this.dup();
          this.push(index);
          String _descriptor = (method.getArgumentTypes()[index]).getDescriptor();
          if (_descriptor != null) {
            switch (_descriptor) {
              case "Z":
                this.loadArg(index);
                this.invokeStatic(Boolean.class, "Boolean valueOf(boolean)");
                break;
              case "C":
                this.loadArg(index);
                this.invokeStatic(Character.class, "Character valueOf(char)");
                break;
              case "B":
                this.loadArg(index);
                this.invokeStatic(Byte.class, "Byte valueOf(byte)");
                break;
              case "S":
                this.loadArg(index);
                this.invokeStatic(Short.class, "Short valueOf(short)");
                break;
              case "I":
                this.loadArg(index);
                this.invokeStatic(Integer.class, "Integer valueOf(int)");
                break;
              case "F":
                this.loadArg(index);
                this.invokeStatic(Float.class, "Float valueOf(float)");
                break;
              case "J":
                this.loadArg(index);
                this.invokeStatic(Float.class, "Long valueOf(long)");
                break;
              case "D":
                this.loadArg(index);
                this.invokeStatic(Float.class, "Double valueOf(double)");
                break;
              default:
                this.ga.visitVarInsn(Opcodes.ALOAD, (index + 1));
                break;
            }
          } else {
            this.ga.visitVarInsn(Opcodes.ALOAD, (index + 1));
          }
          this.ga.visitInsn(Opcodes.AASTORE);
        }
      }
    } else {
      this.ga.visitInsn(Opcodes.ACONST_NULL);
    }
    return this;
  }
  
  public BytecodeUtils end() {
    this.tcvisitor.visitEnd();
    return this;
  }
  
  public byte[] toByteArray() {
    return this.cwriter.toByteArray();
  }
  
  @Override
  public Class<?> findClass(final String name) {
    byte[] bytes = this.toByteArray();
    return super.defineClass(name, bytes, 0, bytes.length);
  }
}
