/**
 * Copyright (c) 2022 murenchao
 * taomu framework is licensed under Mulan PubL v2.
 * You can use this software according to the terms and conditions of the Mulan PubL v2.
 * You may obtain a copy of Mulan PubL v2 at:
 *       http://license.coscl.org.cn/MulanPubL-2.0
 * THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
 * EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
 * MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
 * See the Mulan PubL v2 for more details.
 */
package cool.taomu.framework.utils.spi;

import com.google.common.collect.HashBasedTable;
import com.google.common.collect.Table;
import cool.taomu.framework.utils.reflect.ReflectUtils;
import java.io.File;
import java.net.URL;
import java.util.Collection;
import java.util.Enumeration;
import java.util.Objects;
import java.util.Set;
import java.util.function.Consumer;
import org.apache.commons.io.FileUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@SuppressWarnings("all")
public final class ServiceLoader<S extends Object> {
  private static final Logger LOG = LoggerFactory.getLogger(ServiceLoader.class);
  
  private static final String PREFIX = "META-INF/services/";
  
  private final Class<S> service;
  
  private final String name;
  
  private final ClassLoader loader;
  
  private Object[] args;
  
  private static final Table<String, String, Class<?>> providers = HashBasedTable.<String, String, Class<?>>create();
  
  private void loadZlass() {
    try {
      String _name = this.service.getName();
      final String fullName = (ServiceLoader.PREFIX + _name);
      Enumeration<URL> configs = null;
      if ((this.loader == null)) {
        configs = ClassLoader.getSystemResources(fullName);
      } else {
        configs = this.loader.getResources(fullName);
      }
      this.loadZlass(configs);
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  protected void loadZlass(final Enumeration<URL> configs) {
    try {
      while (configs.hasMoreElements()) {
        {
          String path = configs.nextElement().toString();
          ServiceLoader.LOG.info(path);
          String _replace = path.replace("file:", "");
          File _file = new File(_replace);
          String strings = FileUtils.readFileToString(_file, "UTF-8");
          final Consumer<String> _function = (String it) -> {
            Class<?> zlass = null;
            try {
              zlass = Class.forName(it, false, this.loader);
              final Alias alias = zlass.<Alias>getAnnotation(Alias.class);
              if (((alias != null) && (!alias.value().equals("")))) {
                ServiceLoader.providers.put(this.name, alias.value(), zlass);
              } else {
                ServiceLoader.providers.put(this.name, zlass.getSimpleName(), zlass);
              }
            } catch (final Throwable _t) {
              if (_t instanceof ClassNotFoundException) {
                final ClassNotFoundException ex = (ClassNotFoundException)_t;
                ServiceLoader.LOG.info((("类 " + it) + " 不存在"), ex);
              } else {
                throw Exceptions.sneakyThrow(_t);
              }
            }
            boolean _isAssignableFrom = this.service.isAssignableFrom(zlass);
            boolean _not = (!_isAssignableFrom);
            if (_not) {
              ServiceLoader.LOG.info((((("类" + it) + " 不是 ") + this.name) + " subtype"));
            }
          };
          IterableExtensions.<String>filterNull(((Iterable<String>)Conversions.doWrapArray(strings.split("\n")))).forEach(_function);
        }
      }
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  private ServiceLoader(final Class<S> svc, final ClassLoader cl, final boolean isReload, final Object... args) {
    this.service = Objects.<Class<S>>requireNonNull(svc, "class 不能为 null");
    this.name = svc.getName();
    this.loader = ObjectUtils.<ClassLoader>defaultIfNull(cl, ClassLoader.getSystemClassLoader());
    if (isReload) {
      ServiceLoader.providers.row(this.name).clear();
    }
    this.loadZlass();
    this.args = args;
  }
  
  public S get(final String key) {
    Object obj = ReflectUtils.newInstance(this.getZlass(key), this.args);
    return this.service.cast(obj);
  }
  
  public S get(final Integer index) {
    Collection<Class<?>> pro = ServiceLoader.providers.row(this.name).values();
    int _size = pro.size();
    boolean _equals = (_size == 0);
    if (_equals) {
      return null;
    }
    Object obj = ReflectUtils.newInstance(((Class<?>[])Conversions.unwrapArray(ServiceLoader.providers.row(this.name).values(), Class.class))[(index).intValue()], this.args);
    return this.service.cast(obj);
  }
  
  public Iterable<S> get() {
    Set<String> keySet = ServiceLoader.providers.row(this.name).keySet();
    final Function1<String, S> _function = (String it) -> {
      return this.service.cast(ReflectUtils.newInstance(ServiceLoader.providers.get(this.name, it), this.args));
    };
    return IterableExtensions.<String, S>map(keySet, _function);
  }
  
  public Class<?> getZlass(final String key) {
    return ServiceLoader.providers.get(this.name, key);
  }
  
  public S first() {
    return this.get(Integer.valueOf(0));
  }
  
  public static <S extends Object> ServiceLoader<S> load(final Class<S> service, final ClassLoader loader, final Object... args) {
    return new ServiceLoader<S>(service, loader, false, args);
  }
  
  public static <S extends Object> ServiceLoader<S> load(final Class<S> service, final Object... args) {
    return ServiceLoader.<S>load(service, Thread.currentThread().getContextClassLoader(), args);
  }
  
  public static <S extends Object> ServiceLoader<S> load(final Class<S> service) {
    return ServiceLoader.<S>load(service, Thread.currentThread().getContextClassLoader(), null);
  }
  
  public static <S extends Object> ServiceLoader<S> reload(final Class<S> service, final ClassLoader loader, final Object... args) {
    return new ServiceLoader<S>(service, loader, true, args);
  }
  
  public static <S extends Object> ServiceLoader<S> reload(final Class<S> service, final Object... args) {
    return ServiceLoader.<S>reload(service, Thread.currentThread().getContextClassLoader(), args);
  }
}
