/**
 * Copyright (c) 2023 murenchao
 * taomu is licensed under Mulan PubL v2.
 * You can use this software according to the terms and conditions of the Mulan PubL v2.
 * You may obtain a copy of Mulan PubL v2 at:
 *       http://license.coscl.org.cn/MulanPubL-2.0
 * THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
 * EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
 * MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
 * See the Mulan PubL v2 for more details.
 */
package cool.taomu.ssh;

import com.jcraft.jsch.Channel;
import com.jcraft.jsch.ChannelExec;
import com.jcraft.jsch.ChannelSftp;
import com.jcraft.jsch.ChannelShell;
import com.jcraft.jsch.JSch;
import com.jcraft.jsch.Session;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.InputStream;
import java.io.PipedInputStream;
import java.io.PipedOutputStream;
import java.io.PrintWriter;
import java.util.Hashtable;
import java.util.Properties;
import org.apache.commons.io.IOUtils;
import org.eclipse.xtend.lib.annotations.Accessors;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.Pure;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@SuppressWarnings("all")
public class JschUtils implements AutoCloseable {
  public static class Ssh implements AutoCloseable {
    private ChannelExec exec = null;
    
    public Ssh(final Session session) {
      try {
        Channel channel = session.openChannel("exec");
        this.exec = ((ChannelExec) channel);
      } catch (Throwable _e) {
        throw Exceptions.sneakyThrow(_e);
      }
    }
    
    public String exec(final String cmd, final String charset) {
      try {
        this.exec.setCommand(cmd);
        this.exec.connect();
        InputStream input = this.exec.getInputStream();
        return IOUtils.toString(input, charset);
      } catch (Throwable _e) {
        throw Exceptions.sneakyThrow(_e);
      }
    }
    
    @Override
    public void close() throws Exception {
      if ((this.exec != null)) {
        this.exec.disconnect();
      }
    }
  }
  
  @Accessors
  public static class Sftp implements AutoCloseable {
    private ChannelSftp sftp = null;
    
    public Sftp(final Session session) {
      try {
        Channel channel = session.openChannel("sftp");
        this.sftp = ((ChannelSftp) channel);
        this.sftp.connect();
      } catch (Throwable _e) {
        throw Exceptions.sneakyThrow(_e);
      }
    }
    
    @Override
    public void close() throws Exception {
      if ((this.sftp != null)) {
        this.sftp.disconnect();
      }
    }
    
    public byte[] get(final String file) {
      try {
        ByteArrayOutputStream bos = new ByteArrayOutputStream();
        SftpProgresssMonitorImpl _sftpProgresssMonitorImpl = new SftpProgresssMonitorImpl();
        this.sftp.get(file, bos, _sftpProgresssMonitorImpl);
        return bos.toByteArray();
      } catch (Throwable _e) {
        throw Exceptions.sneakyThrow(_e);
      }
    }
    
    public void get(final String src, final String dst) {
      try {
        SftpProgresssMonitorImpl _sftpProgresssMonitorImpl = new SftpProgresssMonitorImpl();
        this.sftp.get(src, dst, _sftpProgresssMonitorImpl);
      } catch (Throwable _e) {
        throw Exceptions.sneakyThrow(_e);
      }
    }
    
    public void put(final String file, final byte[] content) {
      try {
        ByteArrayInputStream bis = new ByteArrayInputStream(content);
        SftpProgresssMonitorImpl _sftpProgresssMonitorImpl = new SftpProgresssMonitorImpl();
        this.sftp.put(bis, file, _sftpProgresssMonitorImpl);
      } catch (Throwable _e) {
        throw Exceptions.sneakyThrow(_e);
      }
    }
    
    public void put(final String src, final String dst) {
      try {
        SftpProgresssMonitorImpl _sftpProgresssMonitorImpl = new SftpProgresssMonitorImpl();
        this.sftp.put(src, dst, _sftpProgresssMonitorImpl);
      } catch (Throwable _e) {
        throw Exceptions.sneakyThrow(_e);
      }
    }
    
    @Pure
    public ChannelSftp getSftp() {
      return this.sftp;
    }
    
    public void setSftp(final ChannelSftp sftp) {
      this.sftp = sftp;
    }
  }
  
  public static class Shell implements AutoCloseable {
    private ChannelShell channel = null;
    
    public Shell(final Session session) {
      try {
        Channel _openChannel = session.openChannel("shell");
        this.channel = ((ChannelShell) _openChannel);
      } catch (Throwable _e) {
        throw Exceptions.sneakyThrow(_e);
      }
    }
    
    public String exec(final String[] content) {
      return this.exec(content, "UTF-8");
    }
    
    public String exec(final String[] content, final String charset) {
      try {
        try (PipedInputStream pis = new PipedInputStream()) {
          try (PipedOutputStream pos = new PipedOutputStream()) {
            try (PrintWriter pw = new PrintWriter(pos)) {
              PipedInputStream _pipedInputStream = new PipedInputStream(pos);
              this.channel.setInputStream(_pipedInputStream);
              PipedOutputStream _pipedOutputStream = new PipedOutputStream(pis);
              this.channel.setOutputStream(_pipedOutputStream);
              this.channel.connect();
              for (final String it : content) {
                pw.println(it);
              }
              pw.flush();
              return IOUtils.toString(pis, charset);
            }
          }
        }
      } catch (Throwable _e) {
        throw Exceptions.sneakyThrow(_e);
      }
    }
    
    @Override
    public void close() throws Exception {
      if ((this.channel != null)) {
        this.channel.disconnect();
      }
    }
  }
  
  private static final Logger LOG = LoggerFactory.getLogger(JschUtils.class);
  
  private Session session = null;
  
  public JschUtils(final String host, final String uname, final String upass, final int port) {
    this(host, uname, upass, port, null);
  }
  
  /**
   * 跳过 Kerberos
   * session.setConfig(
   *     "PreferredAuthentications",
   *     "publickey,keyboard-interactive,password");
   */
  public JschUtils(final String host, final String uname, final String upass, final int port, final Hashtable<String, String> sshconfig) {
    try {
      JSch jsch = new JSch();
      this.session = jsch.getSession(uname, host, port);
      this.session.setPassword(upass);
      if ((sshconfig != null)) {
        this.session.setConfig(sshconfig);
      }
      Properties config = new Properties();
      config.put("StrictHostKeyChecking", "no");
      this.session.setConfig(config);
      this.session.setTimeout(60000000);
      this.session.connect();
      boolean _isConnected = this.session.isConnected();
      if (_isConnected) {
        JschUtils.LOG.info("链接成功");
      } else {
        JschUtils.LOG.info("链接失败");
      }
    } catch (final Throwable _t) {
      if (_t instanceof Exception) {
        final Exception ex = (Exception)_t;
        JschUtils.LOG.info("失败", ex);
      } else {
        throw Exceptions.sneakyThrow(_t);
      }
    }
  }
  
  @Override
  public void close() throws Exception {
    if ((this.session != null)) {
      this.session.disconnect();
    }
  }
  
  public JschUtils.Ssh Ssh() {
    return new JschUtils.Ssh(this.session);
  }
  
  public JschUtils.Sftp Sftp() {
    return new JschUtils.Sftp(this.session);
  }
  
  public JschUtils.Shell Shell() {
    return new JschUtils.Shell(this.session);
  }
}
