/**
 * Copyright (c) 2023 murenchao
 * taomu is licensed under Mulan PubL v2.
 * You can use this software according to the terms and conditions of the Mulan PubL v2.
 * You may obtain a copy of Mulan PubL v2 at:
 *       http://license.coscl.org.cn/MulanPubL-2.0
 * THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
 * EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
 * MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
 * See the Mulan PubL v2 for more details.
 */
package cool.taomu.runtime.groovy;

import cool.taomu.util.FileUtils;
import groovy.lang.GroovyClassLoader;
import groovy.lang.GroovyCodeSource;
import java.io.File;
import java.io.InputStream;
import java.nio.charset.Charset;
import java.util.HashSet;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.Validate;
import org.codehaus.groovy.runtime.IOGroovyMethods;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@SuppressWarnings("all")
public class GroovyScript extends ClassLoader implements AutoCloseable {
  private static final Logger LOG = LoggerFactory.getLogger(GroovyScript.class);
  
  private GroovyClassLoader.InnerLoader il = null;
  
  public GroovyScript() {
  }
  
  public GroovyScript(final String path) throws Exception {
    this(path, Thread.currentThread().getContextClassLoader());
  }
  
  public GroovyScript(final String path, final ClassLoader loader) throws Exception {
    GroovyScript.LOG.info(String.format("加载脚本:%s", path));
    File _file = new File(path);
    HashSet<String> scripts = FileUtils.scanningFile(_file);
    final Function1<String, String> _function = (String it) -> {
      try {
        File _file_1 = new File(it);
        return org.apache.commons.io.FileUtils.readFileToString(_file_1, "UTF-8");
      } catch (Throwable _e) {
        throw Exceptions.sneakyThrow(_e);
      }
    };
    String scriptStr = IterableExtensions.join(IterableExtensions.<String, String>map(IterableExtensions.<String>filterNull(scripts), _function), "\n");
    this.loaderScript(scriptStr, loader);
  }
  
  public void loaderScript(final String src, final ClassLoader loader) {
    try {
      String coding = Validate.<String>notBlank(src, "coding 不能为Blank");
      try (InputStream input = IOUtils.toInputStream(coding, Charset.forName("UTF-8"))) {
        ClassLoader _xifexpression = null;
        if ((loader == null)) {
          _xifexpression = Thread.currentThread().getContextClassLoader();
        } else {
          _xifexpression = loader;
        }
        GroovyClassLoader gl = new GroovyClassLoader(_xifexpression);
        GroovyClassLoader.InnerLoader _innerLoader = new GroovyClassLoader.InnerLoader(gl);
        this.il = _innerLoader;
        String scriptText = IOGroovyMethods.getText(input);
        GroovyCodeSource c = new GroovyCodeSource(scriptText, "./", "/groovy/script");
        this.il.parseClass(c);
      } catch (final Throwable _t) {
        if (_t instanceof Exception) {
          final Exception ex = (Exception)_t;
          throw new Exception(ex);
        } else {
          throw Exceptions.sneakyThrow(_t);
        }
      }
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  @Override
  public Class<?> findClass(final String name) {
    try {
      return this.il.loadClass(name);
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  @Override
  public void close() throws Exception {
    GroovyScript.LOG.debug("Groovy class loader close");
    if ((this.il != null)) {
      this.il.close();
    }
  }
}
