/**
 * Copyright (c) 2023 murenchao
 * taomu is licensed under Mulan PubL v2.
 * You can use this software according to the terms and conditions of the Mulan PubL v2.
 * You may obtain a copy of Mulan PubL v2 at:
 *       http://license.coscl.org.cn/MulanPubL-2.0
 * THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
 * EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
 * MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
 * See the Mulan PubL v2 for more details.
 */
package cool.taomu.mqtt.mqttv3;

import com.google.common.base.Objects;
import com.google.gson.Gson;
import cool.taomu.crypto.Base64;
import java.util.ArrayList;
import java.util.List;
import java.util.UUID;
import java.util.concurrent.ExecutorService;
import java.util.function.Consumer;
import org.eclipse.paho.client.mqttv3.MqttClient;
import org.eclipse.paho.client.mqttv3.MqttConnectOptions;
import org.eclipse.paho.client.mqttv3.MqttMessage;
import org.eclipse.paho.client.mqttv3.persist.MemoryPersistence;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ListExtensions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@SuppressWarnings("all")
public class MqttV3Service {
  private static final Logger LOG = LoggerFactory.getLogger(MqttV3Service.class);
  
  public static void subscriber(final ExecutorService es, final String host, final int port, final String username, final String password, final Class<? extends MqttCallback>... callbacks) {
    final Consumer<Class<? extends MqttCallback>> _function = (Class<? extends MqttCallback> it) -> {
      final Runnable _function_1 = () -> {
        final Topics topics = it.<Topics>getAnnotation(Topics.class);
        if ((topics != null)) {
          final Function1<Topic, Boolean> _function_2 = (Topic it_1) -> {
            Topic.MessageType _messageType = it_1.messageType();
            return Boolean.valueOf(Objects.equal(_messageType, Topic.MessageType.SUBSCRIBER));
          };
          final Consumer<Topic> _function_3 = (Topic topic) -> {
            MqttV3Service.subscriber(topic, host, port, username, password, it);
          };
          IterableExtensions.<Topic>filter(IterableExtensions.<Topic>filterNull(((Iterable<Topic>)Conversions.doWrapArray(topics.value()))), _function_2).forEach(_function_3);
        } else {
          final Topic topic = it.<Topic>getAnnotation(Topic.class);
          MqttV3Service.subscriber(topic, host, port, username, password, it);
        }
      };
      es.execute(_function_1);
    };
    IterableExtensions.<Class<? extends MqttCallback>>filterNull(((Iterable<Class<? extends MqttCallback>>)Conversions.doWrapArray(callbacks))).forEach(_function);
  }
  
  public static void sender(final Topic topic, final String host, final int port, final String username, final String password, final byte[] payload) {
    Topic.MessageType _messageType = topic.messageType();
    boolean _equals = Objects.equal(_messageType, Topic.MessageType.SENDER);
    if (_equals) {
      MqttV3Service.sender(topic.value()[0], topic.timeout(), topic.isSsl(), topic.clientId(), (topic.qos()[0]).ordinal(), 
        topic.retain(), host, port, username, password, payload);
    }
  }
  
  public static void sender(final String topic, final String host, final int port, final String username, final String password, final byte[] payload) {
    MqttV3Service.sender(topic, 6000, false, UUID.randomUUID().toString(), 1, true, host, port, username, password, payload);
  }
  
  public static void sender(final String topic, final int timeout, final boolean isSsl, final String clientId, final int qos, final boolean retain, final String host, final int port, final String username, final String password, final byte[] payload) {
    try (MemoryPersistence memoryPersistence = new MemoryPersistence()) {
      MqttConnectOptions mqttConnectOptions = new MqttConnectOptions();
      mqttConnectOptions.setMqttVersion(MqttConnectOptions.MQTT_VERSION_3_1_1);
      mqttConnectOptions.setCleanSession(true);
      if ((username != null)) {
        mqttConnectOptions.setUserName(username);
        mqttConnectOptions.setPassword(password.toCharArray());
      }
      mqttConnectOptions.setConnectionTimeout(timeout);
      mqttConnectOptions.setKeepAliveInterval(20);
      String _xifexpression = null;
      if (isSsl) {
        _xifexpression = "ssl";
      } else {
        _xifexpression = "tcp";
      }
      String _format = String.format("%s://%s:%d", _xifexpression, host, Integer.valueOf(port));
      MqttClient mqttClient = new MqttClient(_format, clientId, memoryPersistence);
      mqttClient.connect(mqttConnectOptions);
      byte[] _encode = new Base64(payload).encode();
      MqttMessage mqttMessage = new MqttMessage(_encode);
      mqttMessage.setQos(qos);
      mqttMessage.setRetained(retain);
      mqttClient.publish(topic, mqttMessage);
      MqttV3Service.LOG.debug("clientId: {} 发送成功", clientId);
      mqttClient.disconnect();
    } catch (final Throwable _t) {
      if (_t instanceof Exception) {
        final Exception ex = (Exception)_t;
        MqttV3Service.LOG.info("mqtt 异常:", ex);
      } else {
        throw Exceptions.sneakyThrow(_t);
      }
    }
  }
  
  private static void subscriber(final Topic topic, final String host, final int port, final String username, final String password, final Class<? extends MqttCallback> callback) {
    try {
      Topic.MessageType _messageType = topic.messageType();
      boolean _equals = Objects.equal(_messageType, Topic.MessageType.SUBSCRIBER);
      if (_equals) {
        Will will = callback.<Will>getAnnotation(Will.class);
        if ((topic != null)) {
          String _json = new Gson().toJson(topic.value());
          String _plus = ("Topic : " + _json);
          MqttV3Service.LOG.info(_plus);
          final MemoryPersistence memoryPersistence = new MemoryPersistence();
          final MqttConnectOptions mqttConnectOptions = new MqttConnectOptions();
          mqttConnectOptions.setMqttVersion(MqttConnectOptions.MQTT_VERSION_3_1_1);
          mqttConnectOptions.setCleanSession(topic.cleanSession());
          if (((username != null) && (!username.trim().equals("")))) {
            mqttConnectOptions.setUserName(username);
            mqttConnectOptions.setPassword(password.toCharArray());
          }
          mqttConnectOptions.setConnectionTimeout(topic.timeout());
          mqttConnectOptions.setKeepAliveInterval(topic.keepAlive());
          if ((will != null)) {
            mqttConnectOptions.setWill(will.topic(), will.message().getBytes("UTF-8"), will.qos(), will.retain());
          }
          String _xifexpression = null;
          boolean _isSsl = topic.isSsl();
          if (_isSsl) {
            _xifexpression = "ssl";
          } else {
            _xifexpression = "tcp";
          }
          String broker = String.format("%s://%s:%d", _xifexpression, host, Integer.valueOf(port));
          String uuid = UUID.randomUUID().toString();
          boolean _equals_1 = topic.clientId().equals("uuid");
          boolean _not = (!_equals_1);
          if (_not) {
            uuid = topic.clientId();
          }
          final MqttClient mqttClient = new MqttClient(broker, uuid, memoryPersistence);
          MqttCallback mc = callback.newInstance();
          mc.setClient(mqttClient);
          mc.setOptions(mqttConnectOptions);
          mc.setHost(host);
          mc.setPort(port);
          if (((username != null) && (!username.trim().equals("")))) {
            mc.setUsername(username);
            mc.setPassword(password);
          }
          mqttClient.setCallback(mc);
          mqttClient.connect(mqttConnectOptions);
          ArrayList<Integer> qoss = new ArrayList<Integer>();
          final Function1<Topic.QoS, Integer> _function = (Topic.QoS it) -> {
            return Integer.valueOf(it.ordinal());
          };
          qoss.addAll(ListExtensions.<Topic.QoS, Integer>map(((List<Topic.QoS>)Conversions.doWrapArray(topic.qos())), _function));
          for (int i = 0; ((qoss.size() < ((List<String>)Conversions.doWrapArray(topic.value())).size()) && (i < (((List<String>)Conversions.doWrapArray(topic.value())).size() - qoss.size()))); i++) {
            qoss.add(Integer.valueOf(Topic.QoS.AT_MOST_ONCE.ordinal()));
          }
          final ArrayList<Integer> _converted_qoss = (ArrayList<Integer>)qoss;
          mqttClient.subscribe(topic.value(), ((int[])Conversions.unwrapArray(_converted_qoss, int.class)));
          Runtime _runtime = Runtime.getRuntime();
          _runtime.addShutdownHook(new Thread() {
            @Override
            public void run() {
              try {
                mqttClient.unsubscribe(topic.value());
                mqttClient.disconnect();
                memoryPersistence.close();
              } catch (Throwable _e) {
                throw Exceptions.sneakyThrow(_e);
              }
            }
          });
        }
      }
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
}
