/**
 * Copyright (c) 2023 murenchao
 * taomu is licensed under Mulan PubL v2.
 * You can use this software according to the terms and conditions of the Mulan PubL v2.
 * You may obtain a copy of Mulan PubL v2 at:
 *       http://license.coscl.org.cn/MulanPubL-2.0
 * THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
 * EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
 * MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
 * See the Mulan PubL v2 for more details.
 */
package cool.taomu.mqtt.broker.factory;

import cool.taomu.mqtt.broker.entity.MessageEntity;
import cool.taomu.mqtt.broker.impl.PublishObservable;
import cool.taomu.mqtt.broker.impl.Retain;
import cool.taomu.mqtt.broker.impl.RetainObservable;
import cool.taomu.mqtt.broker.utils.MqttUtils;
import cool.taomu.mqtt.broker.utils.impl.DataStorage;
import cool.taomu.mqtt.broker.utils.inter.IObservable;
import cool.taomu.mqtt.broker.utils.inter.IObserver;
import cool.taomu.storage.inter.IStorage;
import io.netty.channel.ChannelHandlerContext;
import io.netty.handler.codec.mqtt.MqttFixedHeader;
import io.netty.handler.codec.mqtt.MqttMessage;
import io.netty.handler.codec.mqtt.MqttMessageIdVariableHeader;
import io.netty.handler.codec.mqtt.MqttMessageType;
import io.netty.handler.codec.mqtt.MqttPubAckMessage;
import io.netty.handler.codec.mqtt.MqttPublishMessage;
import io.netty.handler.codec.mqtt.MqttQoS;
import io.netty.util.ReferenceCountUtil;
import java.util.Collections;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@SuppressWarnings("all")
public class PublishRequest implements IProcess {
  private static final Logger LOG = LoggerFactory.getLogger(PublishRequest.class);
  
  private IStorage cache = new DataStorage();
  
  private static final IObservable<IObserver> retainObservable = RetainObservable.getInstance();
  
  private static final IObservable<IObserver> observable = PublishObservable.getInstance();
  
  @Override
  public void request(final ChannelHandlerContext ctx, final MqttMessage mqttMessage) {
    try {
      if ((!(mqttMessage instanceof MqttPublishMessage))) {
        return;
      }
      MqttPublishMessage publishMessage = ((MqttPublishMessage) mqttMessage);
      MessageEntity message = new MessageEntity();
      message.setSenderId(MqttUtils.getClientId(ctx.channel()));
      String _senderId = message.getSenderId();
      String _plus = ("执行了MQTT Publish 命令 : " + _senderId);
      PublishRequest.LOG.info(_plus);
      MqttQoS qos = publishMessage.fixedHeader().qosLevel();
      message.setQos(qos.ordinal());
      message.setTopic(publishMessage.variableHeader().topicName());
      message.setPayload(((MqttPublishMessage) mqttMessage).payload());
      message.setType(Integer.valueOf(mqttMessage.fixedHeader().messageType().value()));
      message.setDup(publishMessage.fixedHeader().isDup());
      message.setRetain(publishMessage.fixedHeader().isRetain());
      message.setMsgId(publishMessage.variableHeader().packetId());
      message.setSenderChannel(ctx.channel());
      if (qos != null) {
        switch (qos) {
          case EXACTLY_ONCE:
          case AT_MOST_ONCE:
            PublishRequest.LOG.info(String.format("Qos0 and Qos2 message,clientId=%s", message.getSenderId()));
            this.retainMessage(message);
            this.publishMessage(message);
            break;
          case AT_LEAST_ONCE:
            PublishRequest.LOG.info(String.format("Qos1 message,clientId=%s", message.getSenderId()));
            this.retainMessage(message);
            this.publishMessage(message);
            MqttFixedHeader header = new MqttFixedHeader(MqttMessageType.PUBACK, false, MqttQoS.AT_MOST_ONCE, false, 0);
            MqttMessageIdVariableHeader varHeader = MqttMessageIdVariableHeader.from(message.getMsgId());
            MqttPubAckMessage _mqttPubAckMessage = new MqttPubAckMessage(header, varHeader);
            ctx.writeAndFlush(_mqttPubAckMessage);
            break;
          default:
            PublishRequest.LOG.info(String.format("Wrong mqtt message,clientId=%s", message.getSenderId()));
            break;
        }
      } else {
        PublishRequest.LOG.info(String.format("Wrong mqtt message,clientId=%s", message.getSenderId()));
      }
    } catch (final Throwable _t) {
      if (_t instanceof Exception) {
        final Exception ex = (Exception)_t;
        PublishRequest.LOG.debug("执行了MQTT Publish 命令出错了 : ", ex);
      } else {
        throw Exceptions.sneakyThrow(_t);
      }
    } finally {
      ReferenceCountUtil.release(mqttMessage.payload());
    }
  }
  
  public void retainMessage(final MessageEntity message) {
    byte[] _payload = message.getPayload();
    String _string = new String(_payload);
    PublishRequest.LOG.debug(
      "clientId 为 {} 是否存在 Retain 数据 {}, 接受到的数据为 {} ", 
      message.getSenderId(), 
      Boolean.valueOf(message.isRetain()), _string);
    this.cache.put("mqtt-message", message.getSenderId(), message);
    boolean _isRetain = message.isRetain();
    if (_isRetain) {
      int qos = message.getQos();
      byte[] payload = message.getPayload();
      if ((((qos == MqttQoS.AT_MOST_ONCE.ordinal()) || (payload == null)) || (payload.length == 0))) {
        PublishRequest.LOG.info("清空 clientId 为 {} 的Retain数据", message.getSenderId());
        String _senderId = message.getSenderId();
        String _topic = message.getTopic();
        PublishRequest.retainObservable.unregister(IterableExtensions.join(Collections.<String>unmodifiableList(CollectionLiterals.<String>newArrayList(_senderId, _topic)), "-"));
      } else {
        PublishRequest.LOG.info("保存 clientId 为 {} 的Retain数据", message.getSenderId());
        String _senderId_1 = message.getSenderId();
        String _topic_1 = message.getTopic();
        String _join = IterableExtensions.join(Collections.<String>unmodifiableList(CollectionLiterals.<String>newArrayList(_senderId_1, _topic_1)), "-");
        Retain _retain = new Retain(message);
        PublishRequest.retainObservable.register(_join, _retain);
      }
    }
  }
  
  public void publishMessage(final MessageEntity message) {
    PublishRequest.LOG.debug("推送消息");
    PublishRequest.observable.publish(message);
  }
}
