/**
 * Copyright (c) 2023 murenchao
 * taomu is licensed under Mulan PubL v2.
 * You can use this software according to the terms and conditions of the Mulan PubL v2.
 * You may obtain a copy of Mulan PubL v2 at:
 *       http://license.coscl.org.cn/MulanPubL-2.0
 * THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
 * EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
 * MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
 * See the Mulan PubL v2 for more details.
 */
package cool.taomu.mqtt.mqttv3;

import com.google.common.base.Objects;
import cool.taomu.crypto.Base64;
import java.util.List;
import java.util.function.Consumer;
import org.eclipse.paho.client.mqttv3.IMqttDeliveryToken;
import org.eclipse.paho.client.mqttv3.MqttClient;
import org.eclipse.paho.client.mqttv3.MqttConnectOptions;
import org.eclipse.paho.client.mqttv3.MqttMessage;
import org.eclipse.xtend.lib.annotations.Accessors;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.Pure;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Accessors
@SuppressWarnings("all")
public abstract class MqttCallback extends MqttCallbackExtend implements org.eclipse.paho.client.mqttv3.MqttCallback {
  private static final Logger LOG = LoggerFactory.getLogger(MqttCallback.class);
  
  private MqttClient client;
  
  private MqttConnectOptions options;
  
  private String host;
  
  private int port;
  
  private String password;
  
  private String username;
  
  public void setClient(final MqttClient client) {
    this.client = client;
  }
  
  public void setOptions(final MqttConnectOptions options) {
    this.options = options;
  }
  
  @Override
  public void connectionLost(final Throwable cause) {
    MqttCallback.LOG.debug("连接丢失", cause);
  }
  
  @Override
  public void deliveryComplete(final IMqttDeliveryToken token) {
    final Consumer<String> _function = (String it) -> {
      MqttCallback.LOG.debug("Client deliveryComplete : {}", it);
    };
    ((List<String>)Conversions.doWrapArray(token.getTopics())).forEach(_function);
  }
  
  @Override
  public void messageArrived(final String topic, final MqttMessage message) throws Exception {
    byte[] _payload = message.getPayload();
    byte[] base64 = new Base64(_payload).decode();
    final byte[] payload = this.message(topic, base64);
    if ((payload != null)) {
      Topics topics = this.getClass().<Topics>getAnnotation(Topics.class);
      final Function1<Topic, Boolean> _function = (Topic it) -> {
        Topic.MessageType _messageType = it.messageType();
        return Boolean.valueOf(Objects.equal(_messageType, Topic.MessageType.SENDER));
      };
      final Consumer<Topic> _function_1 = (Topic it) -> {
        MqttV3Service.sender(it, this.host, this.port, this.username, this.password, payload);
      };
      IterableExtensions.<Topic>filterNull(IterableExtensions.<Topic>filter(((Iterable<Topic>)Conversions.doWrapArray(topics.value())), _function)).forEach(_function_1);
    }
  }
  
  public abstract byte[] message(final String topic, final byte[] message);
  
  @Pure
  public MqttClient getClient() {
    return this.client;
  }
  
  @Pure
  public MqttConnectOptions getOptions() {
    return this.options;
  }
  
  @Pure
  public String getHost() {
    return this.host;
  }
  
  public void setHost(final String host) {
    this.host = host;
  }
  
  @Pure
  public int getPort() {
    return this.port;
  }
  
  public void setPort(final int port) {
    this.port = port;
  }
  
  @Pure
  public String getPassword() {
    return this.password;
  }
  
  public void setPassword(final String password) {
    this.password = password;
  }
  
  @Pure
  public String getUsername() {
    return this.username;
  }
  
  public void setUsername(final String username) {
    this.username = username;
  }
}
