/**
 * Copyright (c) 2023 murenchao
 * taomu is licensed under Mulan PubL v2.
 * You can use this software according to the terms and conditions of the Mulan PubL v2.
 * You may obtain a copy of Mulan PubL v2 at:
 *       http://license.coscl.org.cn/MulanPubL-2.0
 * THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
 * EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
 * MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
 * See the Mulan PubL v2 for more details.
 */
package cool.taomu.mqtt.broker.factory;

import cool.taomu.mqtt.broker.entity.ClientSessionEntity;
import cool.taomu.mqtt.broker.entity.MessageEntity;
import cool.taomu.mqtt.broker.utils.MqttUtils;
import cool.taomu.mqtt.broker.utils.impl.DataStorage;
import cool.taomu.storage.inter.IStorage;
import io.netty.channel.ChannelFuture;
import io.netty.channel.ChannelHandlerContext;
import io.netty.channel.ChannelPipeline;
import io.netty.handler.codec.mqtt.MqttConnAckMessage;
import io.netty.handler.codec.mqtt.MqttConnAckVariableHeader;
import io.netty.handler.codec.mqtt.MqttConnectMessage;
import io.netty.handler.codec.mqtt.MqttConnectReturnCode;
import io.netty.handler.codec.mqtt.MqttFixedHeader;
import io.netty.handler.codec.mqtt.MqttMessage;
import io.netty.handler.codec.mqtt.MqttMessageType;
import io.netty.handler.codec.mqtt.MqttQoS;
import io.netty.handler.timeout.IdleStateHandler;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@SuppressWarnings("all")
public class ConnectRequest implements IProcess {
  private static final Logger LOG = LoggerFactory.getLogger(ConnectRequest.class);
  
  private IStorage cache = new DataStorage();
  
  private MqttConnectReturnCode code;
  
  private boolean sessionPresent;
  
  @Override
  public void request(final ChannelHandlerContext ctx, final MqttMessage mqttMessage) {
    if ((!(mqttMessage instanceof MqttConnectMessage))) {
      return;
    }
    ConnectRequest.LOG.info("执行 Connect 操作");
    boolean sessionPresent = false;
    try {
      MqttConnectMessage connect = ((MqttConnectMessage) mqttMessage);
      int version = connect.variableHeader().version();
      ConnectRequest.LOG.info("mqtt version:{}", Integer.valueOf(version));
      boolean isCleanSession = connect.variableHeader().isCleanSession();
      String clientId = connect.payload().clientIdentifier();
      MqttUtils.setClientId(ctx.channel(), clientId);
      String uname = connect.payload().userName();
      byte[] passwd = connect.payload().passwordInBytes();
      int heartbeatSec = connect.variableHeader().keepAliveTimeSeconds();
      ConnectRequest.LOG.info("clientId:{},cleanSession:{}", clientId, Boolean.valueOf(isCleanSession));
      boolean _version = CheckConnect.version(version);
      boolean _not = (!_version);
      if (_not) {
        this.code = MqttConnectReturnCode.CONNECTION_REFUSED_UNACCEPTABLE_PROTOCOL_VERSION;
        this.sessionPresent = sessionPresent;
      } else {
        boolean _clientId = CheckConnect.clientId(ctx.channel(), clientId);
        boolean _not_1 = (!_clientId);
        if (_not_1) {
          this.code = MqttConnectReturnCode.CONNECTION_REFUSED_IDENTIFIER_REJECTED;
          this.sessionPresent = sessionPresent;
        } else {
          boolean _authorized = CheckConnect.authorized(MqttUtils.getRemoteAddr(ctx.channel()), clientId);
          boolean _not_2 = (!_authorized);
          if (_not_2) {
            this.code = MqttConnectReturnCode.CONNECTION_REFUSED_NOT_AUTHORIZED;
            this.sessionPresent = sessionPresent;
          } else {
            boolean _userAuth = CheckConnect.userAuth(clientId, uname, passwd);
            boolean _not_3 = (!_userAuth);
            if (_not_3) {
              this.code = MqttConnectReturnCode.CONNECTION_REFUSED_BAD_USER_NAME_OR_PASSWORD;
              this.sessionPresent = sessionPresent;
            } else {
              boolean _keepAlive = this.keepAlive(clientId, ctx, heartbeatSec);
              boolean _not_4 = (!_keepAlive);
              if (_not_4) {
                String failure = String.format("set heartbeat failure clientId:%s,heartbeatSec:%d", clientId, Integer.valueOf(heartbeatSec));
                throw new Exception(failure);
              }
              this.code = MqttConnectReturnCode.CONNECTION_ACCEPTED;
              this.sessionPresent = (this.createSession(clientId, ctx, isCleanSession)).booleanValue();
              this.storeWill(connect, clientId);
            }
          }
        }
      }
    } catch (final Throwable _t) {
      if (_t instanceof Exception) {
        final Exception ex = (Exception)_t;
        ConnectRequest.LOG.info("服务不可用 :", ex);
        this.code = MqttConnectReturnCode.CONNECTION_REFUSED_SERVER_UNAVAILABLE;
        this.sessionPresent = sessionPresent;
      } else {
        throw Exceptions.sneakyThrow(_t);
      }
    } finally {
      this.response(ctx, mqttMessage);
    }
  }
  
  public ChannelFuture response(final ChannelHandlerContext ctx, final MqttMessage mqttMessage) {
    ChannelFuture _xblockexpression = null;
    {
      MqttFixedHeader header = new MqttFixedHeader(MqttMessageType.CONNACK, false, MqttQoS.AT_MOST_ONCE, false, 0);
      MqttConnAckVariableHeader varHeader = new MqttConnAckVariableHeader(this.code, this.sessionPresent);
      MqttConnAckMessage _mqttConnAckMessage = new MqttConnAckMessage(header, varHeader);
      _xblockexpression = ctx.writeAndFlush(_mqttConnAckMessage);
    }
    return _xblockexpression;
  }
  
  protected Boolean createSession(final String clientId, final ChannelHandlerContext ctx, final boolean cleanSession) {
    ConnectRequest.LOG.info("记录用户session：{}", clientId);
    ClientSessionEntity mqttSession = new ClientSessionEntity();
    boolean sessionPresent = false;
    ClientSessionEntity _clientSessionEntity = new ClientSessionEntity();
    mqttSession = _clientSessionEntity;
    mqttSession.setClientId(clientId);
    mqttSession.setCtx(ctx);
    mqttSession.setCleanStatus(cleanSession);
    if ((!cleanSession)) {
      sessionPresent = true;
    }
    this.cache.put("mqtt-session", clientId, mqttSession);
    return Boolean.valueOf(sessionPresent);
  }
  
  protected int storeWill(final MqttConnectMessage connectMessage, final String clientId) {
    int _xifexpression = (int) 0;
    boolean _isWillFlag = connectMessage.variableHeader().isWillFlag();
    if (_isWillFlag) {
      int _xblockexpression = (int) 0;
      {
        ConnectRequest.LOG.info("保存遗嘱消息 ： clientId:{}", clientId);
        MessageEntity will = new MessageEntity();
        will.setSenderId(clientId);
        will.setRetain(connectMessage.variableHeader().isWillRetain());
        will.setQos(connectMessage.variableHeader().willQos());
        will.setTopic(connectMessage.payload().willTopic());
        will.setPayload(connectMessage.payload().willMessageInBytes());
        _xblockexpression = this.cache.put("mqtt-will", clientId, will);
      }
      _xifexpression = _xblockexpression;
    }
    return _xifexpression;
  }
  
  public boolean keepAlive(final String clientId, final ChannelHandlerContext ctx, final int heatbeatSec) {
    ConnectRequest.LOG.info("设置keep alive");
    int keepAlive = ((int) (heatbeatSec * 1.5f));
    boolean _contains = ctx.pipeline().names().contains("idleStateHandler");
    if (_contains) {
      ctx.pipeline().remove("idleStateHandler");
    }
    ChannelPipeline _pipeline = ctx.pipeline();
    IdleStateHandler _idleStateHandler = new IdleStateHandler(keepAlive, 0, 0);
    _pipeline.addFirst("idleStateHandler", _idleStateHandler);
    return true;
  }
}
