/**
 * Copyright (c) 2023 murenchao
 * taomu is licensed under Mulan PubL v2.
 * You can use this software according to the terms and conditions of the Mulan PubL v2.
 * You may obtain a copy of Mulan PubL v2 at:
 *       http://license.coscl.org.cn/MulanPubL-2.0
 * THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
 * EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
 * MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
 * See the Mulan PubL v2 for more details.
 */
package cool.taomu.mqtt.broker.impl;

import cool.taomu.mqtt.broker.entity.ClientSessionEntity;
import cool.taomu.mqtt.broker.entity.MessageEntity;
import cool.taomu.mqtt.broker.entity.PublishEntity;
import cool.taomu.mqtt.broker.entity.TopicEntity;
import cool.taomu.mqtt.broker.utils.MqttUtils;
import cool.taomu.mqtt.broker.utils.impl.DataStorage;
import cool.taomu.mqtt.broker.utils.inter.IObservable;
import cool.taomu.mqtt.broker.utils.inter.IObserver;
import cool.taomu.storage.inter.IStorage;
import io.netty.buffer.ByteBuf;
import io.netty.buffer.Unpooled;
import io.netty.handler.codec.mqtt.MqttFixedHeader;
import io.netty.handler.codec.mqtt.MqttMessageType;
import io.netty.handler.codec.mqtt.MqttPublishMessage;
import io.netty.handler.codec.mqtt.MqttPublishVariableHeader;
import io.netty.handler.codec.mqtt.MqttQoS;
import java.io.Serializable;
import java.util.concurrent.atomic.AtomicInteger;
import org.apache.commons.lang3.SerializationUtils;
import org.apache.oro.text.perl.Perl5Util;
import org.eclipse.xtend.lib.annotations.Accessors;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.Pure;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Accessors
@SuppressWarnings("all")
public class Retain implements IObserver {
  private final Logger LOG = LoggerFactory.getLogger(Retain.class);
  
  private IStorage cache = new DataStorage();
  
  private MessageEntity msg;
  
  private static AtomicInteger count = new AtomicInteger(0);
  
  private int number = 0;
  
  public Retain(final MessageEntity msg) {
    this.msg = msg;
  }
  
  public Retain() {
    Retain.count.incrementAndGet();
    this.number = Retain.count.intValue();
  }
  
  private void publishMessage(final TopicEntity topic) {
    try {
      Perl5Util p5 = new Perl5Util();
      String subTopicName = topic.getName().replace("/+", "/[a-zA-Z]?[a-zA-Z0-9]+").replace("/#", 
        "/[a-zA-Z]?([a-zA-Z0-9/]*)").replace("/", "\\/");
      boolean _match = p5.match((("/" + subTopicName) + "/"), this.msg.getTopic());
      if (_match) {
        this.LOG.info("发送者id : {},  Topic : {}", this.msg.getSenderId(), this.msg.getTopic());
        int minQos = MqttUtils.getQos(this.msg.getQos(), topic.getQos());
        if ((minQos == 2)) {
          MessageEntity cloneMsg = SerializationUtils.<MessageEntity>clone(this.msg);
          cloneMsg.setSenderChannel(this.msg.getSenderChannel());
          this.cache.put("mqtt-qos2-message", topic.getClientId(), cloneMsg);
        }
        this.LOG.info("cache is null:{} client id:{}", Boolean.valueOf((this.cache == null)), topic.getClientId());
        Serializable _get = this.cache.get("mqtt-session", topic.getClientId());
        ClientSessionEntity clientSession = ((ClientSessionEntity) _get);
        this.LOG.info("订阅者id : {},  Topic : {}, 发送内容长度： {}", topic.getClientId(), Integer.valueOf(this.msg.getPayload().length));
        MqttQoS _valueOf = MqttQoS.valueOf(minQos);
        String _name = topic.getName();
        int _generateMessageId = clientSession.generateMessageId();
        byte[] _payload = this.msg.getPayload();
        PublishEntity entity = new PublishEntity(_valueOf, _name, 
          ((Integer) Integer.valueOf(_generateMessageId)), _payload, Boolean.valueOf(false));
        clientSession.getCtx().writeAndFlush(this.response(entity));
      }
    } catch (final Throwable _t) {
      if (_t instanceof Exception) {
        final Exception ex = (Exception)_t;
        this.LOG.debug("publishMessage 方法出现错误 : ", ex);
      } else {
        throw Exceptions.sneakyThrow(_t);
      }
    }
  }
  
  private MqttPublishMessage response(final PublishEntity entity) {
    Boolean _dup = entity.getDup();
    MqttQoS _qos = entity.getQos();
    MqttFixedHeader header = new MqttFixedHeader(MqttMessageType.PUBLISH, (_dup).booleanValue(), _qos, false, 0);
    String _topicName = entity.getTopicName();
    Integer _messageId = entity.getMessageId();
    MqttPublishVariableHeader varHeader = new MqttPublishVariableHeader(_topicName, (_messageId).intValue());
    ByteBuf heapBuf = null;
    byte[] _payload = entity.getPayload();
    boolean _tripleEquals = (_payload == null);
    if (_tripleEquals) {
      heapBuf = Unpooled.EMPTY_BUFFER;
    } else {
      try {
        heapBuf = Unpooled.wrappedBuffer(entity.getPayload());
      } catch (final Throwable _t) {
        if (_t instanceof IllegalArgumentException) {
          final IllegalArgumentException e = (IllegalArgumentException)_t;
          e.printStackTrace();
        } else {
          throw Exceptions.sneakyThrow(_t);
        }
      }
    }
    return new MqttPublishMessage(header, varHeader, heapBuf);
  }
  
  @Override
  public void publish(final IObservable<?> o, final Object arg) {
    if ((arg instanceof TopicEntity)) {
      this.publishMessage(((TopicEntity)arg));
    }
  }
  
  @Pure
  public Logger getLOG() {
    return this.LOG;
  }
  
  @Pure
  public IStorage getCache() {
    return this.cache;
  }
  
  public void setCache(final IStorage cache) {
    this.cache = cache;
  }
  
  @Pure
  public MessageEntity getMsg() {
    return this.msg;
  }
  
  public void setMsg(final MessageEntity msg) {
    this.msg = msg;
  }
  
  @Pure
  public int getNumber() {
    return this.number;
  }
  
  public void setNumber(final int number) {
    this.number = number;
  }
}
