/**
 * Copyright (c) 2023 murenchao
 * taomu is licensed under Mulan PubL v2.
 * You can use this software according to the terms and conditions of the Mulan PubL v2.
 * You may obtain a copy of Mulan PubL v2 at:
 *       http://license.coscl.org.cn/MulanPubL-2.0
 * THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
 * EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
 * MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
 * See the Mulan PubL v2 for more details.
 */
package cool.taomu.mqtt.broker.impl;

import cool.taomu.mqtt.broker.entity.ClientSessionEntity;
import cool.taomu.mqtt.broker.entity.MessageEntity;
import cool.taomu.mqtt.broker.entity.PublishEntity;
import cool.taomu.mqtt.broker.entity.TopicEntity;
import cool.taomu.mqtt.broker.utils.MqttUtils;
import cool.taomu.mqtt.broker.utils.impl.DataStorage;
import cool.taomu.mqtt.broker.utils.inter.IObservable;
import cool.taomu.mqtt.broker.utils.inter.IObserver;
import cool.taomu.storage.inter.IStorage;
import io.netty.buffer.ByteBuf;
import io.netty.buffer.Unpooled;
import io.netty.handler.codec.mqtt.MqttFixedHeader;
import io.netty.handler.codec.mqtt.MqttMessageType;
import io.netty.handler.codec.mqtt.MqttPublishMessage;
import io.netty.handler.codec.mqtt.MqttPublishVariableHeader;
import io.netty.handler.codec.mqtt.MqttQoS;
import java.io.Serializable;
import java.util.concurrent.atomic.AtomicInteger;
import org.apache.commons.lang3.SerializationUtils;
import org.apache.oro.text.perl.Perl5Util;
import org.eclipse.xtend.lib.annotations.Accessors;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.Pure;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Accessors
@SuppressWarnings("all")
public class Publish implements IObserver {
  private final Logger LOG = LoggerFactory.getLogger(Publish.class);
  
  private static final IStorage cache = new DataStorage();
  
  private TopicEntity topic;
  
  private static AtomicInteger count = new AtomicInteger(0);
  
  private int number = 0;
  
  public Publish(final TopicEntity topic) {
    this.topic = topic;
  }
  
  public Publish() {
    Publish.count.incrementAndGet();
    this.number = Publish.count.intValue();
  }
  
  private void publishMessage(final MessageEntity msg) {
    try {
      Perl5Util p5 = new Perl5Util();
      String subTopicName = this.topic.getName().replace("/+", "/[a-zA-Z]?[a-zA-Z0-9]+").replace("/#", 
        "/[a-zA-Z]?([a-zA-Z0-9/]*)").replace("/", "\\/");
      boolean _match = p5.match((("/" + subTopicName) + "/"), msg.getTopic());
      if (_match) {
        this.LOG.info("发送者id : {},  Topic : {}", msg.getSenderId(), msg.getTopic());
        int minQos = MqttUtils.getQos(msg.getQos(), this.topic.getQos());
        if ((minQos == 2)) {
          MessageEntity cloneMsg = SerializationUtils.<MessageEntity>clone(msg);
          cloneMsg.setSenderChannel(msg.getSenderChannel());
          Publish.cache.put("mqtt-qos2-message", this.topic.getClientId(), cloneMsg);
        }
        Serializable _get = Publish.cache.get("mqtt-session", this.topic.getClientId());
        ClientSessionEntity clientSession = ((ClientSessionEntity) _get);
        this.LOG.info("clientSession is not null {}", Boolean.valueOf((clientSession != null)));
        int _generateMessageId = clientSession.generateMessageId();
        Integer msgId = ((Integer) Integer.valueOf(_generateMessageId));
        this.LOG.info("订阅者id : {},  Topic : {}, 发送内容长度： {} messageId : {}", this.topic.getClientId(), this.topic.getName(), 
          Integer.valueOf(msg.getPayload().length), msgId);
        MqttQoS _valueOf = MqttQoS.valueOf(minQos);
        String _name = this.topic.getName();
        byte[] _payload = msg.getPayload();
        PublishEntity entity = new PublishEntity(_valueOf, _name, msgId, _payload, Boolean.valueOf(false));
        clientSession.getCtx().writeAndFlush(this.response(entity));
      }
    } catch (final Throwable _t) {
      if (_t instanceof Exception) {
        final Exception ex = (Exception)_t;
        this.LOG.debug("publishMessage 方法出现错误 : ", ex);
      } else {
        throw Exceptions.sneakyThrow(_t);
      }
    }
  }
  
  private MqttPublishMessage response(final PublishEntity entity) {
    this.LOG.debug("------<>{}", entity.getMessageId());
    Boolean _dup = entity.getDup();
    MqttQoS _qos = entity.getQos();
    MqttFixedHeader header = new MqttFixedHeader(MqttMessageType.PUBLISH, (_dup).booleanValue(), _qos, false, 0);
    String _topicName = entity.getTopicName();
    Integer _messageId = entity.getMessageId();
    MqttPublishVariableHeader varHeader = new MqttPublishVariableHeader(_topicName, (_messageId).intValue());
    ByteBuf heapBuf = null;
    byte[] _payload = entity.getPayload();
    boolean _tripleEquals = (_payload == null);
    if (_tripleEquals) {
      heapBuf = Unpooled.EMPTY_BUFFER;
    } else {
      try {
        heapBuf = Unpooled.wrappedBuffer(entity.getPayload());
      } catch (final Throwable _t) {
        if (_t instanceof IllegalArgumentException) {
          final IllegalArgumentException e = (IllegalArgumentException)_t;
          e.printStackTrace();
        } else {
          throw Exceptions.sneakyThrow(_t);
        }
      }
    }
    return new MqttPublishMessage(header, varHeader, heapBuf);
  }
  
  @Override
  public void publish(final IObservable<?> o, final Object arg) {
    if ((arg instanceof MessageEntity)) {
      this.publishMessage(((MessageEntity)arg));
    }
  }
  
  @Pure
  public Logger getLOG() {
    return this.LOG;
  }
  
  @Pure
  public TopicEntity getTopic() {
    return this.topic;
  }
  
  public void setTopic(final TopicEntity topic) {
    this.topic = topic;
  }
  
  @Pure
  public int getNumber() {
    return this.number;
  }
  
  public void setNumber(final int number) {
    this.number = number;
  }
}
