/**
 * Copyright (c) 2023 murenchao
 * taomu is licensed under Mulan PubL v2.
 * You can use this software according to the terms and conditions of the Mulan PubL v2.
 * You may obtain a copy of Mulan PubL v2 at:
 *       http://license.coscl.org.cn/MulanPubL-2.0
 * THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
 * EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
 * MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
 * See the Mulan PubL v2 for more details.
 */
package cool.taomu.thrift;

import com.google.auto.service.AutoService;
import cool.taomu.core.TaomuTask;
import cool.taomu.core.entity.TaskEntity;
import cool.taomu.core.inter.ITaomuService;
import cool.taomu.thrift.NetworkThread;
import cool.taomu.thrift.entity.ThriftConfig;
import cool.taomu.util.ThriftUtils;
import cool.taomu.util.YamlUtils;
import java.net.InetSocketAddress;
import java.nio.ByteBuffer;
import java.util.Arrays;
import org.apache.commons.lang3.SerializationException;
import org.apache.commons.lang3.SerializationUtils;
import org.apache.thrift.TException;
import org.apache.thrift.TProcessor;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@AutoService(ITaomuService.class)
@SuppressWarnings("all")
public class NetworkThreadServer implements NetworkThread.Iface, ITaomuService {
  private final Logger LOG = LoggerFactory.getLogger(NetworkThreadServer.class);
  
  private static TaomuTask taomuTask = TaomuTask.getInstance();
  
  @Override
  public void execute(final ByteBuffer data) throws TException {
    try {
      byte[] ser = Arrays.copyOfRange(data.array(), data.position(), data.limit());
      Object _deserialize = SerializationUtils.<Object>deserialize(ser);
      TaskEntity task = ((TaskEntity) _deserialize);
      if ((NetworkThreadServer.taomuTask != null)) {
        NetworkThreadServer.taomuTask.add(task);
      } else {
        throw new NullPointerException();
      }
    } catch (final Throwable _t) {
      if (_t instanceof SerializationException) {
        final SerializationException ex = (SerializationException)_t;
        this.LOG.info("反序列化失败", ex);
      } else if (_t instanceof Exception) {
        final Exception ex_1 = (Exception)_t;
        this.LOG.info("Exception:", ex_1);
      } else {
        throw Exceptions.sneakyThrow(_t);
      }
    }
  }
  
  /**
   * def getObjectInputStreamAware(InputStream bais, AbsWrapperClassLoader dest) {
   * 	return new ObjectInputStreamAware(bais, dest);
   * }
   */
  public static void start(final String host, final int port, final int max, final int min) {
    NetworkThreadServer _networkThreadServer = new NetworkThreadServer();
    TProcessor tprocessor = new NetworkThread.Processor<NetworkThreadServer>(_networkThreadServer);
    InetSocketAddress isa = new InetSocketAddress(host, port);
    ThriftUtils.startServer(tprocessor, isa, max, min);
  }
  
  public static void start(final String host, final int port) {
    NetworkThreadServer _networkThreadServer = new NetworkThreadServer();
    TProcessor tprocessor = new NetworkThread.Processor<NetworkThreadServer>(_networkThreadServer);
    InetSocketAddress isa = new InetSocketAddress(host, port);
    ThriftUtils.startServer(tprocessor, isa);
  }
  
  @Override
  public void service(final TaomuTask factory) {
    NetworkThreadServer.taomuTask = factory;
    ThriftConfig config = YamlUtils.<ThriftConfig>read("./config/thrift.yml", ThriftConfig.class);
    NetworkThreadServer.start(config.getHost(), config.getPort(), config.getMaxThreads(), config.getMinThreads());
  }
}
