package cz.applifting.appgraph.charts.barChart

import androidx.compose.foundation.Canvas
import androidx.compose.foundation.background
import androidx.compose.foundation.layout.*
import androidx.compose.runtime.Composable
import androidx.compose.runtime.MutableState
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.ui.Modifier
import androidx.compose.ui.geometry.CornerRadius
import androidx.compose.ui.geometry.Offset
import androidx.compose.ui.geometry.Size
import androidx.compose.ui.graphics.*
import androidx.compose.ui.graphics.drawscope.DrawScope
import androidx.compose.ui.unit.LayoutDirection
import androidx.compose.ui.unit.dp
import cz.applifting.appgraph.charts.utils.chartXToCanvasX
import cz.applifting.appgraph.charts.utils.chartYtoCanvasY
import cz.applifting.appgraph.charts.common.BasicChartDrawer
import cz.applifting.appgraph.charts.common.YScale
import cz.applifting.appgraph.charts.utils.drawPaddings
import cz.applifting.appgraph.decorations.CanvasDrawable
import cz.applifting.appgraph.decorations.XAxisLabels
import cz.applifting.appgraph.decorations.XAxisLabelsPosition
import cz.applifting.appgraph.decorations.YAxisLabels
import cz.applifting.appgraph.decorations.YAxisLabelsPosition

import cz.applifting.appgraph.accessibility.ChartAccessibilityContainer

/**
 * Composable that displays bar chart
 *
 * @param data - list of data to display
 * @param style - styling of the chart such as padding, canvasPadding, defaultDataPointStyle and axes label visibility
 * @param xAxisLabels - label of the xAxis to display. Must have the same size as data.
 * @param yAxisLabels - labels of the yAxis to display.
 * @param yScale - where the chart begins and ends on the yAxis
 * @param header - composable to display above the chart
 * @param decorations - list of decorations to draw into the chart such as [com.jaikeerthick.composable_graphs.decorations.BackgroundHighlight]
 * @param dataPointsStyles - map that allows to override [BarChartStyle.defaultDataPointStyle]
 * @param basicChartContentDescription - part of content description when the chart as a whole is selected. Should be something like:
 * This is bar chart of...
 * @param dataSizeContentDescription - part of content description when the chart as a whole is selected. Should inform the user
 * how many records are on the chart
 */
@Composable
fun BarChart(
    data: List<Number>,
    style: BarChartStyle = BarChartStyle(),
    xAxisLabels: XAxisLabels? = null,
    yAxisLabels: YAxisLabels = YAxisLabels.fromGraphInputs(data, style.yAxisTextColor, YAxisLabelsPosition.LEFT),
    yScale: YScale = YScale.ZeroToMaxScale(),
    header: @Composable() () -> Unit = {},
    decorations: List<CanvasDrawable> = emptyList<CanvasDrawable>(),
    dataPointsStyles: Map<Int, BarChartDataPointStyle> = emptyMap(),
    basicChartContentDescription: String = "",
    dataSizeContentDescription: String = "",
    onBarClicked: (value: Any) -> Unit = {},
) {
    val barOffsetList = remember { mutableListOf<Pair<Offset, Offset>>() }
    val clickedBar: MutableState<Offset?> = remember { mutableStateOf(null) }

    val presentXAxisLabels = xAxisLabels?: XAxisLabels.createDefault(data, XAxisLabelsPosition.BOTTOM, style.xAxisTextColor)

    Column(
        modifier = Modifier
            .background(
                color = style.backgroundColor //            Color.LightGray
            )
            .fillMaxWidth()
            .padding(style.paddingValues)
            .wrapContentHeight()
    ) {
        if (style.isHeaderVisible){
            header()
        }
        ChartAccessibilityContainer(
            data.map { it.toString() },
            presentXAxisLabels,
            yAxisLabels,
            style.canvasPaddingValues,
            modifier = Modifier
                .fillMaxWidth()
                .height(style.height)
                .padding(horizontal = 1.dp)
                .padding(top = 12.dp),
            basicChartContentDescription = basicChartContentDescription,
            dataSizeContentDescription = dataSizeContentDescription,
        ) {
            Canvas(
                modifier = Modifier
                    .fillMaxSize()
            ) {

                yScale.setupValuesFromData(data)
                val basicDrawer = BasicChartDrawer(
                    this,
                    size,
                    style.canvasPaddingValues.calculateLeftPadding(LayoutDirection.Ltr).toPx(),
                    style.canvasPaddingValues.calculateRightPadding(LayoutDirection.Ltr).toPx(),
                    style.canvasPaddingValues.calculateTopPadding().toPx(),
                    style.canvasPaddingValues.calculateBottomPadding().toPx(),
                    presentXAxisLabels,
                    yAxisLabels,
                    data,
                    yScale,
                    style.backgroundColor,
                    customXDataOffset = 0f
                )

                if (style.drawCanvasPaddings) drawPaddings(basicDrawer)
                if (style.isXAxisLabelVisible) presentXAxisLabels.drawToCanvas(basicDrawer)
                if (style.isYAxisLabelVisible) yAxisLabels.drawToCanvas(basicDrawer)

                decorations.forEach { it.drawToCanvas(basicDrawer) }

                constructGraph(
                    this,
                    data,
                    barOffsetList,
                    basicDrawer,
                    dataPointsStyles,
                    style.defaultDataPointStyle,
                )

                drawClickedRect(this, clickedBar, style.clickHighlightColor, basicDrawer)
            }
        }
    }
}

private fun constructGraph(
    scope: DrawScope,
    dataList: List<Number>,
    barOffsetList: MutableList<Pair<Offset, Offset>>,
    basicChartDrawer: BasicChartDrawer,
    dataPointsStyles: Map<Int, BarChartDataPointStyle>,
    defaultDataPointStyle: BarChartDataPointStyle,
) {
    barOffsetList.clear()
    for (i in dataList.indices) {

        val x1 = chartXToCanvasX(i.toFloat(), basicChartDrawer)
        val y1 = chartYtoCanvasY(dataList[i].toFloat(), basicChartDrawer)

        barOffsetList.add(
            Pair(
                first = Offset(
                    x = x1,
                    y = y1
                ),
                second = Offset(
                    x = chartXToCanvasX((i + 1).toFloat(), basicChartDrawer),
                    y = y1
                ),
            )
        )

        val style = dataPointsStyles.getOrElse(i) {defaultDataPointStyle}

        val x = style.barWidth.getLeftSideXCoordinate(
            chartXToCanvasX((i).toFloat(), basicChartDrawer),
            chartXToCanvasX((i + 1).toFloat(), basicChartDrawer),
            scope
        )
        val width = style.barWidth.getSize(
            chartXToCanvasX((i).toFloat(), basicChartDrawer),
            chartXToCanvasX((i + 1).toFloat(), basicChartDrawer),
            scope
        )

        with (scope) {
            scope.drawRoundRect(
                brush = style.fillGradient,
                topLeft = Offset(
                    x = x,
                    y = y1
                ),
                size = Size(
                    width = width,
                    height = basicChartDrawer.paddingTopPx +  basicChartDrawer.gridHeight - y1
                ),
                cornerRadius = CornerRadius(style.cornerRadius.toPx(), style.cornerRadius.toPx())
            )
        }
    }
}

private fun drawClickedRect(
    scope: DrawScope, clickedBar: MutableState<Offset?>, highlightColor: Color, basicChartDrawer: BasicChartDrawer
) {
    // click action
    clickedBar.value?.let{
        scope.drawRect(
            color = highlightColor,
            topLeft = Offset(
                x = it.x,
                y = it.y
            ),
            size = Size(
                width = basicChartDrawer.xItemSpacing,
                height = basicChartDrawer.paddingTopPx + basicChartDrawer.gridHeight - it.y
            )
        )
    }
}


