package cz.applifting.appgraph.charts.barChart

import androidx.compose.ui.graphics.Brush
import androidx.compose.ui.graphics.drawscope.DrawScope
import androidx.compose.ui.unit.Dp
import androidx.compose.ui.unit.dp
import cz.applifting.appgraph.color.Gradient1
import cz.applifting.appgraph.color.Gradient2

/**
 * Class that allows to style individual bars in barchart
 *
 * @property fillGradient - color of the bar. For monocolored bar use [Brush.verticalGradient] with listof(color,color)
 * @property barWidth how much space should the bar occupy
 * @property cornerRadius - corner radius of the bar applied to all four corners.
 */
data class BarChartDataPointStyle(

    val fillGradient: Brush = Brush.verticalGradient(
        listOf(Gradient1, Gradient2)
    ),
    val barWidth: BarWidth = BarWidth.FullWidth(),
    val cornerRadius: Dp = 0.dp
)

/**
 * Class to compute intended bar width. See [BarWidth.FullWidth], [BarWidth.DpWidth] and [BarWidth.PercentageWidth]
 */
sealed class BarWidth() {

    abstract fun getLeftSideXCoordinate(minX: Float, maxX: Float, scope: DrawScope): Float
    abstract fun getSize(minX: Float, maxX: Float, scope: DrawScope): Float

    /**
     * The bar will occupy all available space
     */
    class FullWidth(): BarWidth() {
        override fun getLeftSideXCoordinate(minX: Float, maxX: Float, scope: DrawScope): Float {
            return minX
        }

        override fun getSize(minX: Float, maxX: Float, scope: DrawScope): Float {
            return maxX - minX
        }
    }

    /**
     * The bar will occupy percentage of the available space.
     * @property percentage of the space to occupy. Number between 0 and 1.
     */
    class PercentageWidth(
        private val percentage: Float
    ): BarWidth() {
        override fun getLeftSideXCoordinate(minX: Float, maxX: Float, scope: DrawScope): Float {
            val maxSize = maxX - minX
            val actualSize = getSize(minX, maxX, scope)
            return minX + ((maxSize - actualSize) / 2f)
        }

        override fun getSize(minX: Float, maxX: Float, scope: DrawScope): Float {
            return (maxX - minX) * percentage
        }
    }

    /**
     * The bar will have fixed width in dp. It is up to the developer to make sure there is enough space for it.
     * @property width - width of the bar in dp.
     */
    class DpWidth(
        private val width: Dp
    ): BarWidth() {
        override fun getLeftSideXCoordinate(minX: Float, maxX: Float, scope: DrawScope): Float {
            val maxSize = maxX - minX
            val actualSize = getSize(minX, maxX, scope)
            return minX + ((maxSize - actualSize) / 2f)
        }

        override fun getSize(minX: Float, maxX: Float, scope: DrawScope): Float {
            with (scope) {
                return width.toPx()
            }
        }
    }
}
