package cz.applifting.appgraph.charts.common

import androidx.compose.ui.geometry.Size
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.graphics.drawscope.DrawScope
import cz.applifting.appgraph.decorations.XAxisLabels
import cz.applifting.appgraph.decorations.YAxisLabels

/**
 * GOD class that holds all the references that might be needed when drawing ALL the stuff to the canvas
 *
 * @property scope - object that is supplied by [androidx.compose.ui.graphics.Canvas] that allows to draw onto it.
 * @property canvasSize - holds width  and height of the canvas as a whole
 * @property paddingLeftPx - padding on the canvas from the left side
 * @property paddingRightPx - padding on the canvas from the right side
 * @property paddingTopPx - padding on the canvas from the top side
 * @property paddingBottomPx - padding on the canvas from the bottom side
 * @property xAxisLabels - reference to the [XAxisLabels] so we know how many vertical grid lines to draw
 * @property yAxisLabels - reference to the [YAxisLabels] so we know how many horizontal grid lines to draw
 * @property dataList
 * @property yScale - reference to [YScale] so we can translate y coordinates in various components.
 * @property canvasBgColor - color that is used to erase elements on the canvas when using [PointStyle]
 * @property customXDataOffset - offset in px of the x coordinate. Used to center data points
 * @property customXLabelOffset - offset in px of the x coordinate. Used to center x labels
 * @property gridWidth - width of the drawing area without paddings. In px
 * @property gridHeight - height of the drawing area without paddings. In px
 * @property xItemSpacing - amount of space between items on the x axis. in px
 * @property xLabelOffset - see [customXLabelOffset]
 * @property xDataOffset see [customXDataOffset]
 */
open class BasicChartDrawer(
    val scope: DrawScope,
    val canvasSize: Size,
    val paddingLeftPx: Float,
    val paddingRightPx: Float,
    val paddingTopPx: Float,
    val paddingBottomPx: Float,
    val xAxisLabels: XAxisLabels,
    val yAxisLabels: YAxisLabels,
    val dataList: List<Number>,
    val yScale: YScale,
    val canvasBgColor: Color,
    val customXDataOffset: Float? = null,
    val customXLabelOffset: Float? = null,
) {

    val gridWidth: Float
        get() = canvasSize.width - paddingLeftPx - paddingRightPx

    val gridHeight: Float
        get() = canvasSize.height - paddingTopPx - paddingBottomPx

    val xItemSpacing: Float
        get() = gridWidth / if (dataList.isNotEmpty()) dataList.size else 1

    val xLabelOffset: Float
        get() = customXLabelOffset ?: (xItemSpacing / 2)

    val xDataOffset: Float
        get() = customXDataOffset ?: (xItemSpacing / 2)
}