package cz.applifting.appgraph.charts.common

import androidx.compose.ui.geometry.Offset
import androidx.compose.ui.geometry.Size
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.graphics.ColorFilter
import androidx.compose.ui.graphics.drawscope.Stroke
import androidx.compose.ui.graphics.drawscope.translate
import androidx.compose.ui.graphics.painter.Painter
import androidx.compose.ui.unit.Dp
import cz.applifting.appgraph.color.GraphAccent

/**
 * Class for styling the point circles. See [PointStyle.FilledPoint], [PointStyle.OutlinedPoint], [PointStyle.ImagePointStyle]
 */
sealed class PointStyle {

    abstract fun drawToCanvas(x: Float, y: Float, basicChartDrawer: BasicChartDrawer)

    /**
     * Class that draws filled point to the canvas
     * @property radius - point radius
     * @property color - fill color of the point
     */
    class FilledPoint(val radius: Dp, val color: Color = GraphAccent): PointStyle() {
        override fun drawToCanvas(x: Float, y: Float, basicChartDrawer: BasicChartDrawer) {

            with(basicChartDrawer.scope) {
                drawCircle(
                    color = color,
                    radius = radius.toPx(),
                    center = Offset(x, y)
                )
            }
        }
    }

    /**
     * Class that draws outlined point to the canvas. [BasicChartStyle.backgroundColor] needs to be set for this to work properly.
     * @property radius - radius of the point outline
     * @property color - color of the point stroke
     * @property strokeWidth - width of the outline stroke
     */
    class OutlinedPoint(val radius: Dp, val color: Color = GraphAccent, val strokeWidth: Dp): PointStyle() {
        override fun drawToCanvas(x: Float, y: Float, basicChartDrawer: BasicChartDrawer) {

            with(basicChartDrawer.scope) {
                drawCircle(
                    color = basicChartDrawer.canvasBgColor,
                    radius = radius.toPx(),
                    center = Offset(x, y)
                )
                drawCircle(
                    color = color,
                    radius = radius.toPx(),
                    center = Offset(x, y),
                    style = Stroke(strokeWidth.toPx())
                )
            }
        }
    }

    /**
     * Draws image instead of the point with [Painter]. For vector image use rememberVectorPainter or painterResource for resources.
     * Needs to have [BasicChartStyle.backgroundColor] set to work properly.
     * @property image - to be drawn
     * @property width - width of the image to be drawn
     * @property height - height of the image to be drawn
     * @property tintColor - tint of the image to be applied
     */
    class ImagePointStyle(
        val image: Painter,
        val width: Dp,
        val height: Dp,
        val tintColor: Color = Color.Black
    ): PointStyle() {
        override fun drawToCanvas(x: Float, y: Float, basicChartDrawer: BasicChartDrawer) {

            var widthPx: Float
            var heightPx: Float

            with (basicChartDrawer.scope) {
                widthPx = width.toPx()
                heightPx = height.toPx()
            }

            basicChartDrawer.scope.drawRect(basicChartDrawer.canvasBgColor, Offset(x - widthPx / 2, y - heightPx / 2), Size(widthPx, heightPx))

            with(image) {
                basicChartDrawer.scope.translate(x - widthPx / 2, y - heightPx / 2) {
                    basicChartDrawer.scope.draw(Size(width.toPx(), height.toPx()), colorFilter = ColorFilter.tint(tintColor))
                }
            }
        }
    }
}
