package cz.applifting.appgraph.charts.doublePointChart

import androidx.compose.foundation.Canvas
import androidx.compose.foundation.background
import androidx.compose.foundation.layout.*
import androidx.compose.runtime.*
import androidx.compose.ui.Modifier
import androidx.compose.ui.geometry.Offset
import androidx.compose.ui.graphics.drawscope.DrawScope
import androidx.compose.ui.platform.LocalDensity
import androidx.compose.ui.unit.Density
import androidx.compose.ui.unit.LayoutDirection
import androidx.compose.ui.unit.dp
import cz.applifting.appgraph.accessibility.ChartAccessibilityContainer
import cz.applifting.appgraph.charts.utils.chartXToCanvasX
import cz.applifting.appgraph.charts.utils.chartYtoCanvasY
import cz.applifting.appgraph.charts.common.BasicChartDrawer
import cz.applifting.appgraph.charts.common.YScale
import cz.applifting.appgraph.charts.utils.drawPaddings
import cz.applifting.appgraph.decorations.CanvasDrawable
import cz.applifting.appgraph.decorations.XAxisLabels
import cz.applifting.appgraph.decorations.XAxisLabelsPosition
import cz.applifting.appgraph.decorations.YAxisLabels
import cz.applifting.appgraph.decorations.YAxisLabelsPosition

/**
 * Composable that displays double point chart
 *
 * @param data - list of data to display. [Pair.first] should be min. [Pair.second] should be max.
 * @param style - styling of the chart such as padding, canvasPadding, defaultDataPointStyle and axes label visibility
 * @param xAxisLabels - label of the xAxis to display. Must have the same size as data.
 * @param yAxisLabels - labels of the yAxis to display.
 * @param yScale - where the chart begins and ends on the yAxis
 * @param header - composable to display above the chart
 * @param decorations - list of decorations to draw into the chart such as [com.jaikeerthick.composable_graphs.decorations.BackgroundHighlight]
 * @param dataPointStyles - map that allows to override [DoublePointChartStyle.defaultDataPointStyle]
 * @param basicChartContentDescription - part of content description when the chart as a whole is selected. Should be something like:
 * This is double point chart of...
 * @param dataSizeContentDescription - part of content description when the chart as a whole is selected. Should inform the user
 * how many records are on the chart
 * @param dataPointMinPrefixContentDescription - content description read before min of data point. Should be something
 * like Minimum of. Also [YAxisLabels.yAxisPointDescriptionSuffix] should be empty.
 * @param dataPointMaxPrefixContentDescription - content description read before max of data point. Should be something
 * like Maximum of. Also [YAxisLabels.yAxisPointDescriptionSuffix] should be empty.
 * @param dataPointSuffixContentDescription - content description read after min and max of data point. Should be a unit like
 * steps. Also should be used instead of [YAxisLabels.yAxisPointDescriptionSuffix] which should be empty.
 */
@Composable
fun DoublePointChart(
    data: List<Pair<Number, Number>>,
    style: DoublePointChartStyle = DoublePointChartStyle(),
    xAxisLabels: XAxisLabels? = null,
    yAxisLabels: YAxisLabels = YAxisLabels.fromGraphInputs(data.map { it.first } + data.map { it.second }, style.yAxisTextColor, YAxisLabelsPosition.LEFT),
    yScale: YScale = YScale.ZeroToMaxScale(),
    header: @Composable() () -> Unit = {},

    decorations: List<CanvasDrawable> = emptyList<CanvasDrawable>(),
    dataPointStyles: Map<Int, DoublePointChartDataPointStyle> = emptyMap(),
    basicChartContentDescription: String = "",
    dataSizeContentDescription: String = "",
    dataPointMinPrefixContentDescription: String = "",
    dataPointMaxPrefixContentDescription: String = "",
    dataPointSuffixContentDescription: String = "",
) {

    var offsetList by remember { mutableStateOf(mutableListOf<Pair<Offset, Offset>>()) }

    val currentDensity = LocalDensity.current
    val maxList = data.map { it.second }
    val presentXAxisLabels = xAxisLabels ?: XAxisLabels.createDefault(maxList, XAxisLabelsPosition.BOTTOM, style.xAxisTextColor)

    Column(
        modifier = Modifier
            .background(
                color = style.backgroundColor
            )
            .fillMaxWidth()
            .padding(style.paddingValues)
            .wrapContentHeight()
    ) {

        if (style.isHeaderVisible) {
            header()
        }

        ChartAccessibilityContainer(
            dataContentDescriptionList = data.map { "$dataPointMinPrefixContentDescription ${it.first} $dataPointSuffixContentDescription $dataPointMaxPrefixContentDescription ${it.second} $dataPointSuffixContentDescription" },
            xAxisLabels = presentXAxisLabels,
            yAxisLabels = yAxisLabels,
            canvasPaddingValues = style.canvasPaddingValues,
            modifier = Modifier
                .fillMaxWidth()
                .height(style.height)
                .padding(horizontal = 1.dp)
                .padding(top = 12.dp),
            basicChartContentDescription = basicChartContentDescription,
            dataSizeContentDescription = dataSizeContentDescription,
        ) {
            Canvas(
                modifier = Modifier.fillMaxSize(),
            ) {
                yScale.setupValuesFromData(maxList)
                val basicDrawer = BasicChartDrawer(
                    this,
                    size,
                    style.canvasPaddingValues.calculateLeftPadding(LayoutDirection.Ltr).toPx(),
                    style.canvasPaddingValues.calculateLeftPadding(LayoutDirection.Ltr).toPx(),
                    style.canvasPaddingValues.calculateTopPadding().toPx(),
                    style.canvasPaddingValues.calculateBottomPadding().toPx(),
                    presentXAxisLabels,
                    yAxisLabels,
                    maxList,
                    yScale,
                    style.backgroundColor
                )

                if (style.drawCanvasPaddings) drawPaddings(basicDrawer)
                if (style.isXAxisLabelVisible) presentXAxisLabels.drawToCanvas(basicDrawer)
                if (style.isYAxisLabelVisible) yAxisLabels.drawToCanvas(basicDrawer)

                decorations.forEach { it.drawToCanvas(basicDrawer) }

                offsetList =  constructOffsetList(
                    data, basicDrawer
                )

                drawLineConnectingPoints(this, offsetList, dataPointStyles, style.defaultDataPointStyle, currentDensity)
                drawPoints(basicDrawer, offsetList, dataPointStyles, style.defaultDataPointStyle, currentDensity)
            }
        }
    }
}

private fun constructOffsetList(
    data: List<Pair<Number, Number>>,
    basicChartDrawer: BasicChartDrawer,
): MutableList<Pair<Offset, Offset>> {

    val offsetList: MutableList<Pair<Offset, Offset>> = mutableListOf()

    for (i in data.indices) {

        val x1 = chartXToCanvasX(i.toFloat(), basicChartDrawer)
        val y1 = chartYtoCanvasY(data[i].first.toFloat(), basicChartDrawer)
        val y2 = chartYtoCanvasY(data[i].second.toFloat(), basicChartDrawer)

        offsetList.add(
            Pair(Offset(x1, y1), Offset(x1, y2))
        )
    }
    return offsetList
}

private fun drawPoints(
    basicChartDrawer: BasicChartDrawer,
    offsetList: List<Pair<Offset, Offset>>,
    dataPointStyles: Map<Int, DoublePointChartDataPointStyle>,
    defaultStyle: DoublePointChartDataPointStyle,
    currentDensity: Density
) {
    offsetList.forEachIndexed { idx, offset ->
        val style = dataPointStyles.getOrElse(idx) { defaultStyle }

        style.bottomPointStyle.drawToCanvas(offset.first.x, offset.first.y, basicChartDrawer)
        style.topPointStyle.drawToCanvas(offset.second.x, offset.second.y, basicChartDrawer)
    }
}

private fun drawLineConnectingPoints(
    scope: DrawScope,
    offsetList: List<Pair<Offset, Offset>>,
    dataPointStyles: Map<Int, DoublePointChartDataPointStyle>,
    defaultStyle: DoublePointChartDataPointStyle,
    currentDensity: Density
) {

    offsetList.forEachIndexed { idx, points ->

        val style = dataPointStyles.getOrElse(idx) {defaultStyle}

        val lineWidthPx = with(currentDensity) { style.lineWidth.toPx() }

        scope.drawLine(color = style.lineColor, start = points.first, end = points.second, strokeWidth = lineWidthPx)
    }
}




