package cz.applifting.appgraph.charts.lineChart

import androidx.compose.foundation.Canvas
import androidx.compose.foundation.background
import androidx.compose.foundation.layout.*
import androidx.compose.runtime.*
import androidx.compose.ui.Modifier
import androidx.compose.ui.geometry.Offset
import androidx.compose.ui.graphics.*
import androidx.compose.ui.graphics.drawscope.DrawScope
import androidx.compose.ui.platform.LocalDensity
import androidx.compose.ui.unit.Density
import androidx.compose.ui.unit.LayoutDirection
import androidx.compose.ui.unit.dp
import cz.applifting.appgraph.accessibility.ChartAccessibilityContainer
import cz.applifting.appgraph.charts.utils.chartXToCanvasX
import cz.applifting.appgraph.charts.utils.chartYtoCanvasY
import cz.applifting.appgraph.charts.common.BasicChartDrawer
import cz.applifting.appgraph.charts.common.YScale
import cz.applifting.appgraph.charts.utils.drawPaddings
import cz.applifting.appgraph.decorations.CanvasDrawable
import cz.applifting.appgraph.decorations.XAxisLabels
import cz.applifting.appgraph.decorations.XAxisLabelsPosition
import cz.applifting.appgraph.decorations.YAxisLabels
import cz.applifting.appgraph.decorations.YAxisLabelsPosition

/**
 * Composable that displays line chart
 *
 * @param data - list of data to display
 * @param style - styling of the chart such as padding, canvasPadding, defaultDataPointStyle and axes label visibility
 * @param xAxisLabels - label of the xAxis to display. Must have the same size as data.
 * @param yAxisLabels - labels of the yAxis to display.
 * @param yScale - where the chart begins and ends on the yAxis
 * @param header - composable to display above the chart
 * @param decorations - list of decorations to draw into the chart such as [com.jaikeerthick.composable_graphs.decorations.BackgroundHighlight]
 * @param dataPointStyles - map that allows to override [LineChartStyle.defaultDataPointStyle]
 * @param basicChartContentDescription - part of content description when the chart as a whole is selected. Should be something like:
 * This is line chart of...
 * @param dataSizeContentDescription - part of content description when the chart as a whole is selected. Should inform the user
 * how many records are on the chart
 */
@Composable
fun LineChart(
    data: List<Number>,
    style: LineChartStyle = LineChartStyle(),
    xAxisLabels: XAxisLabels? = null,
    yAxisLabels: YAxisLabels = YAxisLabels.fromGraphInputs(data, Color.Black.toArgb(), YAxisLabelsPosition.LEFT),
    yScale: YScale = YScale.ZeroToMaxScale(),
    header: @Composable() () -> Unit = {},
    decorations: List<CanvasDrawable> = emptyList<CanvasDrawable>(),
    dataPointStyles: Map<Int, LineChartDataPointStyle> = emptyMap(),
    basicChartContentDescription: String = "",
    dataSizeContentDescription: String = "",
    onPointClicked: (pair: Pair<Any,Any>) -> Unit = {},
) {
    var offsetList by remember{ mutableStateOf(mutableListOf<Offset>()) }
    val isPointClicked = remember { mutableStateOf(false) }
    val clickedPoint: MutableState<Offset?> = remember { mutableStateOf(null) }
    val presentXAxisLabels: XAxisLabels = xAxisLabels ?: XAxisLabels.createDefault(data, XAxisLabelsPosition.BOTTOM, style.xAxisTextColor)

    val currentDensity = LocalDensity.current

    Column(
        modifier = Modifier
            .background(
                color = style.backgroundColor
            )
            .fillMaxWidth()
            .padding(style.paddingValues),
        verticalArrangement = Arrangement.spacedBy(16.dp)
    ) {
        if (style.isHeaderVisible) {
            header()
        }

        ChartAccessibilityContainer(
            dataContentDescriptionList = data.map { it.toString() },
            xAxisLabels = presentXAxisLabels,
            yAxisLabels = yAxisLabels,
            canvasPaddingValues = style.canvasPaddingValues,
            modifier = Modifier.fillMaxWidth().height(style.height).padding(horizontal = 1.dp).padding(top = 12.dp),
            basicChartContentDescription = basicChartContentDescription,
            dataSizeContentDescription = dataSizeContentDescription,
        ) {
            Canvas(
                modifier = Modifier.fillMaxSize(),
            ) {
                yScale.setupValuesFromData(data)
                val basicDrawer = BasicChartDrawer(
                    this,
                    size,
                    style.canvasPaddingValues.calculateLeftPadding(LayoutDirection.Ltr).toPx(),
                    style.canvasPaddingValues.calculateLeftPadding(LayoutDirection.Ltr).toPx(),
                    style.canvasPaddingValues.calculateTopPadding().toPx(),
                    style.canvasPaddingValues.calculateBottomPadding().toPx(),
                    presentXAxisLabels,
                    yAxisLabels,
                    data,
                    yScale,
                    style.backgroundColor
                )

                if (style.drawCanvasPaddings) drawPaddings(basicDrawer)
                if (style.isXAxisLabelVisible) presentXAxisLabels.drawToCanvas(basicDrawer)
                if (style.isYAxisLabelVisible) yAxisLabels.drawToCanvas(basicDrawer)

                decorations.forEach { it.drawToCanvas(basicDrawer) }

                offsetList = constructOffsetList(
                    data, basicDrawer
                )

                paintGradientUnderTheGraphLine(
                    this, offsetList, data, basicDrawer, style.fillGradient
                )

                drawLineConnectingPoints(this, offsetList, style.lineColor, style.lineWidth.toPx())

                drawPoints(basicDrawer, offsetList, dataPointStyles, style.defaultDataPointStyle, currentDensity)

                drawHighlightedPointAndCrossHair(
                    this,
                    clickedPoint,
                    style.clickHighlightColor,
                    style.clickHighlightRadius.toPx(),
                    style.isCrossHairVisible,
                    style.crossHairColor,
                    style.crossHairLineWidth.toPx(),
                    basicDrawer.gridHeight
                )
            }
        }
    }
}

private fun constructOffsetList(
    data: List<Number>,
    basicChartDrawer: BasicChartDrawer,
): MutableList<Offset> {
    val offsetList = mutableListOf<Offset>()

    data.forEachIndexed { idx, dataPoint ->
        val x1 = chartXToCanvasX(idx.toFloat(), basicChartDrawer)
        val y1 = chartYtoCanvasY(dataPoint.toFloat(), basicChartDrawer)

        offsetList.add(
            Offset(
                x = x1,
                y = y1
            )
        )
    }
    return offsetList
}

private fun drawPoints(
    basicChartDrawer: BasicChartDrawer,
    offsetList: MutableList<Offset>,
    dataPointStyles: Map<Int, LineChartDataPointStyle>,
    defaultStyle: LineChartDataPointStyle,
    currentDensity: Density,
) {
    offsetList.forEachIndexed { idx, offset ->
        val style = dataPointStyles.getOrElse(idx) {defaultStyle}

        style.pointStyle.drawToCanvas(offset.x, offset.y, basicChartDrawer)
    }
}

private fun paintGradientUnderTheGraphLine(
    scope: DrawScope,
    offsetList: MutableList<Offset>,
    yAxisData: List<Number>,
    basicChartDrawer: BasicChartDrawer,
    fillBrush: Brush
) {
    /**
     * Drawing Gradient fill for the plotted points
     * Create Path from the offset list with start and end point to complete the path
     * then draw path using brush
     */
    val path = Path().apply {
        // starting point for gradient
        moveTo(
            x = chartXToCanvasX(0f, basicChartDrawer),
            y = chartYtoCanvasY(0f, basicChartDrawer)
        )

        offsetList.forEach { offset -> lineTo(offset.x, offset.y) }

//         ending point for gradient
        lineTo(
            x = offsetList.last().x,
            y = chartYtoCanvasY(0f, basicChartDrawer)
        )
    }

    scope.drawPath(
        path = path,
        brush = fillBrush
    )
}

private fun drawLineConnectingPoints(scope: DrawScope, offsetList: List<Offset>, lineColor: Color, lineWidth: Float) {

    scope.drawPoints(
        points = offsetList,
        color = lineColor,
        // Polygon mode draws the line that connects the points
        pointMode = PointMode.Polygon,
        strokeWidth = lineWidth,
    )
}

private fun drawHighlightedPointAndCrossHair(
    scope: DrawScope,
    clickedPoint: MutableState<Offset?>,
    clickHighlightColor: Color,
    clickHighlightRadius: Float,
    drawCrossHair: Boolean,
    crossHairColor: Color,
    crossHairLineWidth: Float,
    gridHeight: Float
) {
    clickedPoint.value?.let {
        scope.drawCircle(
            color = clickHighlightColor,
            center = it,
            radius = clickHighlightRadius
        )
        if (drawCrossHair) {
            scope.drawLine(
                color = crossHairColor,
                start = Offset(it.x, 0f),
                end = Offset(it.x, gridHeight),
                strokeWidth = crossHairLineWidth,
                pathEffect = PathEffect.dashPathEffect(
                    intervals = floatArrayOf(15f, 15f)
                )
            )
        }
    }
}


