package cz.applifting.appgraph.charts.common

import android.graphics.Paint
import android.graphics.Rect
import androidx.compose.ui.geometry.Offset
import androidx.compose.ui.geometry.Size
import androidx.compose.ui.graphics.PathEffect
import androidx.compose.ui.graphics.drawscope.DrawScope
import androidx.compose.ui.graphics.nativeCanvas
import androidx.compose.ui.unit.LayoutDirection
import androidx.compose.ui.unit.sp
import cz.applifting.appgraph.charts.utils.chartYtoCanvasY
import cz.applifting.appgraph.decorations.BackgroundHighlight
import cz.applifting.appgraph.decorations.HorizontalGridLines
import cz.applifting.appgraph.decorations.HorizontalLine
import cz.applifting.appgraph.decorations.HorizontalLineStyle
import cz.applifting.appgraph.decorations.Label
import cz.applifting.appgraph.decorations.LabelXPosition
import cz.applifting.appgraph.decorations.PaddingPositionType
import cz.applifting.appgraph.decorations.VerticalGridLines
import cz.applifting.appgraph.decorations.XAxisLabels
import cz.applifting.appgraph.decorations.XAxisLabelsPosition
import cz.applifting.appgraph.decorations.YAxisLabels
import cz.applifting.appgraph.decorations.YAxisLabelsPosition

fun DrawScope.drawBackgroundHighlight(
    highlight: BackgroundHighlight, basicChartDrawer: BasicChartDrawer
) {
    drawRect(
        color = highlight.color,
        topLeft = Offset(basicChartDrawer.paddingLeftPx, chartYtoCanvasY(highlight.yEnd, basicChartDrawer)),
        size = Size(basicChartDrawer.gridWidth, chartYtoCanvasY(highlight.yStart, basicChartDrawer) - chartYtoCanvasY(highlight.yEnd, basicChartDrawer))
    )
}

fun DrawScope.drawHorizontalGridLines(gridLines: HorizontalGridLines, basicChartDrawer: BasicChartDrawer) {
    for (label in basicChartDrawer.yAxisLabels.labels) {
        drawLine(
            color = gridLines.color,
            start = Offset(basicChartDrawer.paddingLeftPx, chartYtoCanvasY(label.chartValue, basicChartDrawer)),
            end = Offset(basicChartDrawer.paddingLeftPx + basicChartDrawer.gridWidth , chartYtoCanvasY(label.chartValue, basicChartDrawer)),
            strokeWidth = gridLines.heightPx.toFloat()
        )
    }
}

fun DrawScope.drawHorizontalLine(
    line: HorizontalLine, basicChartDrawer: BasicChartDrawer
) {
    val pathEffect = when (line.style) {
        HorizontalLineStyle.DASHED -> PathEffect.dashPathEffect(floatArrayOf(20f, 10f))
        else -> null
    }

    drawLine(
        color = line.color,
        start = Offset(basicChartDrawer.paddingLeftPx, chartYtoCanvasY(line.y, basicChartDrawer)),
        end = Offset(basicChartDrawer.paddingLeftPx + basicChartDrawer.gridWidth, chartYtoCanvasY(line.y, basicChartDrawer)),
        strokeWidth = line.widthDp.toPx(),
        pathEffect = pathEffect
    )
}

fun DrawScope.drawLabel(label: Label, basicChartDrawer: BasicChartDrawer) {
    val textPaint = Paint()

    val align = when(label.xPosition) {
        is LabelXPosition.PaddingPosition -> when(label.xPosition.type) {
            PaddingPositionType.LEFT -> Paint.Align.LEFT
            else -> Paint.Align.RIGHT
        }
        else -> Paint.Align.CENTER
    }

    textPaint.apply {
        color = label.textColor
        textAlign = align
        textSize = label.textSize.toPx()
    }

    val textBounds = Rect()
    textPaint.getTextBounds(label.text, 0, label.text.length, textBounds)

    val rectTop = label.yPosition.getYValue(basicChartDrawer) - textBounds.height() / 2 - label.padding.calculateTopPadding().toPx()
    val rectLeft = when(label.xPosition) {
        is LabelXPosition.PaddingPosition -> when(label.xPosition.type) {
            PaddingPositionType.LEFT -> label.xPosition.getXValue(basicChartDrawer) - label.padding.calculateLeftPadding(
                LayoutDirection.Ltr).toPx()
            else -> label.xPosition.getXValue(basicChartDrawer) - textBounds.width() - label.padding.calculateLeftPadding(
                LayoutDirection.Ltr).toPx()
        }
        else -> label.xPosition.getXValue(basicChartDrawer) - textBounds.exactCenterX() - label.padding.calculateLeftPadding(
            LayoutDirection.Ltr).toPx()
    }

    drawRect(
        label.backgroundColor,
        Offset(rectLeft, rectTop),
        Size(
            textBounds.width().toFloat()
                    + label.padding.calculateLeftPadding(LayoutDirection.Ltr).toPx()
                    + label.padding.calculateRightPadding(LayoutDirection.Ltr).toPx(),
            textBounds.height().toFloat()
                    + label.padding.calculateTopPadding().toPx()
                    + label.padding.calculateBottomPadding().toPx()
        )
    )

    drawContext.canvas.nativeCanvas.drawText(
        label.text,
        label.xPosition.getXValue(basicChartDrawer),
        label.yPosition.getYValue(basicChartDrawer) - textBounds.exactCenterY(),
        textPaint
    )
}

fun DrawScope.drawVerticalGridLines(gridLines: VerticalGridLines, basicChartDrawer: BasicChartDrawer) {

    for (i in 0 until basicChartDrawer.xAxisLabels.labels.size) {

        val x = basicChartDrawer.paddingLeftPx + (basicChartDrawer.xItemSpacing * i) + basicChartDrawer.xItemSpacing / 2

        drawLine(
            color = gridLines.color,
            start = Offset( x, basicChartDrawer.paddingTopPx),
            end = Offset(x, basicChartDrawer.paddingTopPx + basicChartDrawer.gridHeight),
            strokeWidth = gridLines.widthPx.toFloat()
        )
    }
}

fun DrawScope.drawXAxisLabels(labels: XAxisLabels, basicChartDrawer: BasicChartDrawer) {

    labels.labels.forEachIndexed {idx, label ->

        val y = if (labels.position == XAxisLabelsPosition.BOTTOM) basicChartDrawer.canvasSize.height.toFloat() else 0f

        drawContext.canvas.nativeCanvas.drawText(
            label.label,
            (basicChartDrawer.xItemSpacing * (idx)) + basicChartDrawer.xLabelOffset + basicChartDrawer.paddingLeftPx, // x
            y, // y
            Paint().apply {
                color = labels.color
                textAlign = Paint.Align.CENTER
                textSize = 12.sp.toPx()
            }
        )
    }
}

fun DrawScope.drawYAxisLabels(labels: YAxisLabels, basicChartDrawer: BasicChartDrawer) {

    val x = if (labels.position == YAxisLabelsPosition.LEFT) 0 else basicChartDrawer.canvasSize.width
    val align = if (labels.position == YAxisLabelsPosition.LEFT) Paint.Align.LEFT else Paint.Align.RIGHT

    val textPaint = Paint()
    textPaint.apply {
        color = labels.color
        textAlign = align
        textSize = 12.sp.toPx()
    }

    labels.labels.forEachIndexed {idx, label ->

        val textBounds = Rect()
        textPaint.getTextBounds(label.label, 0, label.label?.length ?: 0, textBounds)

        drawContext.canvas.nativeCanvas.drawText(
            label.label?: "",
            x.toFloat(),
            chartYtoCanvasY(label.chartValue, basicChartDrawer) - textBounds.exactCenterY(), //y
            textPaint
        )
    }
}