@file:OptIn(ExperimentalUnitApi::class)

package cz.applifting.appgraph.decorations

import android.graphics.Paint
import android.graphics.Rect
import androidx.compose.foundation.layout.PaddingValues
import androidx.compose.ui.geometry.Offset
import androidx.compose.ui.geometry.Size
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.graphics.drawscope.DrawScope
import androidx.compose.ui.graphics.nativeCanvas
import androidx.compose.ui.graphics.toArgb
import androidx.compose.ui.unit.Dp
import androidx.compose.ui.unit.ExperimentalUnitApi
import androidx.compose.ui.unit.LayoutDirection
import androidx.compose.ui.unit.TextUnit
import androidx.compose.ui.unit.TextUnitType
import cz.applifting.appgraph.charts.utils.chartXToCanvasX
import cz.applifting.appgraph.charts.utils.chartYtoCanvasY
import cz.applifting.appgraph.charts.common.BasicChartDrawer
import cz.applifting.appgraph.charts.common.drawLabel

/**
 * Component that draws text label with color background to the canvas.
 *
 * @property text
 * @property xPosition - see [LabelXPosition.CanvasPosition], [LabelXPosition.ChartPosition], [LabelXPosition.PaddingPosition]
 * @property yPosition - see [LabelYPosition.CanvasPosition], [LabelYPosition.ChartPosition], [LabelYPosition.PaddingPosition]
 * @property backgroundColor
 * @property padding - Padding between text and bgColor
 * @property textColor - Color int of the text. Use [Color.toArgb]
 * @property textSize
 */
class Label(
    val text: String,
    val xPosition: LabelXPosition,
    val yPosition: LabelYPosition,
    val backgroundColor: Color = Color.Transparent,
    val padding: PaddingValues = PaddingValues(Dp(4f)),
    val textColor: Int = Color.Black.toArgb(),
    val textSize: TextUnit = TextUnit(12f, TextUnitType.Sp),
): CanvasDrawable
{
    override fun drawToCanvas(basicChartDrawer: BasicChartDrawer) {
        basicChartDrawer.scope.drawLabel(this, basicChartDrawer)
    }
}



/**
 * Class that translates x coordinate for the label.
 */
sealed class LabelXPosition() {

    abstract fun getXValue(basicChartDrawer: BasicChartDrawer): Float

    /**
     * Makes no translation. 0 is left corner. Right corner is canvas width
     */
    class CanvasPosition(val x: Float): LabelXPosition() {
        override fun getXValue(basicChartDrawer: BasicChartDrawer): Float {
            return x
        }
    }

    /**
     * Depends how chart is drawn. In LineChart and DoublePointChart 1 is above first data point. Barchart 1 is beggining of the first bar space.
     * 2 is the end of the first bar space
     */
    class ChartPosition(val x: Float): LabelXPosition() {
        override fun getXValue(basicChartDrawer: BasicChartDrawer): Float {
            return chartXToCanvasX(x - 1, basicChartDrawer)
        }
    }

    /**
     * Positions either in LEFT or RIGHT canvas padding.
     */
    class PaddingPosition(val type: PaddingPositionType): LabelXPosition() {
        override fun getXValue(basicChartDrawer: BasicChartDrawer): Float {
            return when(type) {
                PaddingPositionType.LEFT -> 0f
                else -> basicChartDrawer.canvasSize.width
            }
        }
    }
}

sealed class LabelYPosition() {

    abstract fun getYValue(basicChartDrawer: BasicChartDrawer): Float

    /**
     * Makes no translation 0 is top corner. Bottom is canvas height
     */
    class CanvasPosition(val y: Float): LabelYPosition() {
        override fun getYValue(basicChartDrawer: BasicChartDrawer): Float {
            return y
        }
    }

    /**
     * The value corresponds to y-value according to current chart scale. The same as data are drawn
     */
    class ChartPosition(val y: Float): LabelYPosition() {
        override fun getYValue(basicChartDrawer: BasicChartDrawer): Float {
            return chartYtoCanvasY(y, basicChartDrawer)
        }
    }

    /**
     * TOP or BOTTOM canvas padding position
     */
    class PaddingPosition(val type: PaddingPositionType): LabelYPosition() {
        override fun getYValue(basicChartDrawer: BasicChartDrawer): Float {
            return when(type) {
                PaddingPositionType.TOP -> 0f
                else -> basicChartDrawer.canvasSize.height
            }
        }
    }
}

enum class PaddingPositionType {LEFT, RIGHT, TOP, BOTTOM}