package cz.applifting.appgraph.decorations

import android.graphics.Paint
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.graphics.drawscope.DrawScope
import androidx.compose.ui.graphics.nativeCanvas
import androidx.compose.ui.graphics.toArgb
import androidx.compose.ui.unit.sp
import cz.applifting.appgraph.charts.common.BasicChartDrawer
import cz.applifting.appgraph.charts.common.XAxisLabelItem
import cz.applifting.appgraph.charts.common.drawXAxisLabels

/**
 * Component that draw X-Axis labels to Canvas.
 *
 * @property labels
 * @property position - TOP or BOTTOM
 * @property color - Color int. Use [Color#toArgb] to convert.
 * @property xAxisContentDescription - Description that is read when the chart is selected. E.g. On the x axis there are days of the week.
 * @property xAxisPointPrefixDescription - Description that is read before each data point on the x axis. E.g If you have days of week the prefix would be "On" and it would be read
 * as On monday, On Tuesday...
 * The whole content description for each data point is as follows: [xAxisPointPrefixDescription] + [XAxisLabelItem.label] + [xAxisPointSuffixDescription] + yValue + [YAxisLabels.yAxisPointDescriptionSuffix]
 * @property xAxisPointSuffixDescription - Description that is read after each data point on the x axis. E.g. you have, there are...
 * The whole content description for each data point is as follows: [xAxisPointPrefixDescription] + [XAxisLabelItem.label] + [xAxisPointSuffixDescription] + yValue + [YAxisLabels.yAxisPointDescriptionSuffix]
 */
data class XAxisLabels(
    val labels: List<XAxisLabelItem>,
    val position: XAxisLabelsPosition = XAxisLabelsPosition.BOTTOM,
    val color: Int = Color.Black.toArgb(),
    val xAxisContentDescription: String = "",
    val xAxisPointPrefixDescription: String = "",
    val xAxisPointSuffixDescription: String = ""
): CanvasDrawable {

    companion object {
        fun createDefault(data: List<Number>, position: XAxisLabelsPosition, color: Int): XAxisLabels {
            return XAxisLabels(data.mapIndexed {idx, _ -> XAxisLabelItem(idx.toString()) }, position, color)
        }
    }

    override fun drawToCanvas(basicChartDrawer: BasicChartDrawer) {
        basicChartDrawer.scope.drawXAxisLabels(
            this, basicChartDrawer
        )
    }
}

enum class XAxisLabelsPosition {
    TOP, BOTTOM
}