package cz.applifting.appgraph.decorations

import android.graphics.Paint
import android.graphics.Rect
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.graphics.drawscope.DrawScope
import androidx.compose.ui.graphics.nativeCanvas
import androidx.compose.ui.graphics.toArgb
import androidx.compose.ui.unit.sp
import cz.applifting.appgraph.charts.common.BasicChartDrawer
import cz.applifting.appgraph.charts.common.drawYAxisLabels
import cz.applifting.appgraph.charts.utils.chartYtoCanvasY
import cz.applifting.appgraph.charts.utils.GraphHelper

/**
 * Component that draws Y-Axis labels to canvas
 *
 * @property labels
 * @property color - Color int the text should have. Use [Color.toArgb] to convert compose color
 * @property position - LEFT, RIGHT
 * @property yAxisContentDescription - Content description of what is displayed on the y-axis. Read when chart is focusesd.
 * E.g. On the vertical axis there is number of steps you have made.
 * @property yAxisPointDescriptionSuffix - Content description read after each data point. Should be the unit. E.g. Steps, meals...
 */
data class YAxisLabels(
    val labels: List<AxisLabelItem>,
    val color: Int = Color.Black.toArgb(),
    val position: YAxisLabelsPosition = YAxisLabelsPosition.LEFT,
    val yAxisContentDescription: String = "",
    val yAxisPointDescriptionSuffix: String = "",
): CanvasDrawable {
    companion object {
        fun fromGraphInputs(inputs: List<Number>, color: Int, position: YAxisLabelsPosition): YAxisLabels {
            val absMaxY = GraphHelper.getAbsoluteMax(inputs)
            val verticalStep = absMaxY.toInt() / inputs.size.toFloat()

            val yAxisLabelList = mutableListOf<AxisLabelItem>()

            for (i in 0..inputs.size) {
                yAxisLabelList.add(AxisLabelItem(verticalStep * i))
            }
            return YAxisLabels(yAxisLabelList, color, position)
        }
    }

    override fun drawToCanvas(basicChartDrawer: BasicChartDrawer) {
        basicChartDrawer.scope.drawYAxisLabels(
            this, basicChartDrawer
        )
    }
}

enum class YAxisLabelsPosition {LEFT, RIGHT}

