package cz.applifting.firebasewrappersdk

import android.app.NotificationManager
import android.content.Context
import android.util.Log
import androidx.core.app.NotificationCompat
import androidx.core.content.ContextCompat
import com.google.firebase.ktx.Firebase
import com.google.firebase.messaging.FirebaseMessagingService
import com.google.firebase.messaging.RemoteMessage
import com.google.firebase.messaging.ktx.messaging

private const val NOTIFICATION_ID = 0
private const val TAG = "FirebaseCloudMessagingService"

internal open class FirebaseCloudMessagingService : FirebaseMessagingService() {

    override fun onMessageReceived(message: RemoteMessage) {
        message.data.let {
            Log.d(TAG, "onMessageReceived: $it")
        }
        message.notification?.let {
            sendNotification(this, it)
        }
    }

    override fun onNewToken(token: String) {
        Log.d(TAG, "onNewToken: $token")
    }

    fun subscribeToTopic(topic: String?, myCallback: (result: String?) -> Unit) {
        topic?.let {
            Firebase.messaging.subscribeToTopic(it)
                .addOnCompleteListener { task ->
                    var msg = "Subscribed"
                    if (!task.isSuccessful)
                        msg = "Subscribe failed"
                    myCallback.invoke("subscribeToTopic: $topic, result: $msg")
                }
        }
    }

    fun unsubscribeFromTopic(topic: String?, myCallback: (result: String?) -> Unit) {
        topic?.let {
            Firebase.messaging.unsubscribeFromTopic(it)
                .addOnCompleteListener { task ->
                    var msg = "Unsubscribed"
                    if (!task.isSuccessful)
                        msg = "Unsubscribe failed"
                    myCallback.invoke("unsubscribeFromTopic: $topic, result: $msg")
                }
        }
    }

    private fun sendNotification(context: Context, notification: RemoteMessage.Notification) {
        val notificationManager = ContextCompat.getSystemService(
            context,
            NotificationManager::class.java
        ) as NotificationManager
        notificationManager.sendNotification(notification, context)
    }

}

private fun NotificationManager.sendNotification(
    notification: RemoteMessage.Notification,
    applicationContext: Context
) {
    val notificationBuilder = NotificationCompat.Builder(
        applicationContext,
        applicationContext.getString(R.string.default_notification_channel_id)
    )
        .setSmallIcon(R.drawable.ic_default_notification)
        .setContentTitle(notification.title)
        .setContentText(notification.body)
        .setAutoCancel(true)

    notification.channelId?.let {
        notificationBuilder.setChannelId(it)
    }

    notify(NOTIFICATION_ID, notificationBuilder.build())
}
